# 亮仔扫描 (lz-scan) 架构文档

> 版本: 1.6.5  
> 更新日期: 2025年10月12日  
> 作者: 克亮

---

## 📋 目录

1. [概述](#1-概述)
2. [技术栈](#2-技术栈)
3. [系统架构](#3-系统架构)
4. [核心模块](#4-核心模块)
5. [数据流](#5-数据流)
6. [线程模型](#6-线程模型)
7. [SANE集成](#7-sane集成)
8. [图像处理](#8-图像处理)
9. [智能合并](#9-智能合并)
10. [文档管理](#10-文档管理)
11. [性能优化](#11-性能优化)
12. [扩展性](#12-扩展性)

---

## 1. 概述

### 1.1 项目定位

亮仔扫描是一款现代化的 Linux 文档扫描应用，基于 Qt5 和 SANE 开发，提供：
- 完整的扫描功能（单页/多页/双面/延时）
- 强大的图像处理能力
- 智能拆分合并功能
- 多格式文档导出

### 1.2 设计理念

- **模块化**: 清晰的模块划分，低耦合高内聚
- **线程安全**: 扫描操作在独立线程，不阻塞UI
- **用户友好**: 简洁直观的操作流程
- **可扩展**: 易于添加新功能和设备支持

### 1.3 架构风格

采用 **MVC 变体** 架构：
- **View**: MainWindow + 各种 Widget
- **Controller**: ScannerController + DocumentManager
- **Model**: 图像数据 + 文档数据
- **Processor**: ImageProcessor（图像处理层）

---

## 2. 技术栈

### 2.1 核心技术

| 技术 | 版本 | 用途 |
|------|------|------|
| C++ | C++11 | 主要开发语言 |
| Qt | 5.12+ | GUI框架 |
| SANE | 1.0+ | 扫描仪后端 |
| CMake | 3.10+ | 构建系统 |

### 2.2 Qt 模块

- **Qt5::Core**: 核心功能、信号槽、线程
- **Qt5::Gui**: 图像处理、绘图
- **Qt5::Widgets**: UI组件
- **Qt5::PrintSupport**: PDF导出
- **Qt5::Concurrent**: 并发任务

### 2.3 构建配置

```cmake
project(lz-scan VERSION 1.6.5 LANGUAGES CXX)
set(CMAKE_CXX_STANDARD 11)
set(CMAKE_AUTOMOC ON)
set(CMAKE_AUTORCC ON)
set(CMAKE_AUTOUIC ON)
```

---

## 3. 系统架构

### 3.1 整体架构图

```
┌─────────────────────────────────────────────────────────┐
│                      用户界面层                          │
│  ┌──────────────┐  ┌──────────────┐  ┌──────────────┐ │
│  │  MainWindow  │  │  Dialogs     │  │  Widgets     │ │
│  └──────┬───────┘  └──────┬───────┘  └──────┬───────┘ │
└─────────┼──────────────────┼──────────────────┼─────────┘
          │                  │                  │
┌─────────┼──────────────────┼──────────────────┼─────────┐
│         │        业务逻辑层 │                  │         │
│  ┌──────▼───────┐  ┌───────▼──────┐  ┌───────▼──────┐ │
│  │  Scanner     │  │  Document    │  │  Image       │ │
│  │  Controller  │  │  Manager     │  │  Processor   │ │
│  └──────┬───────┘  └──────┬───────┘  └──────┬───────┘ │
└─────────┼──────────────────┼──────────────────┼─────────┘
          │                  │                  │
┌─────────┼──────────────────┼──────────────────┼─────────┐
│         │        数据访问层 │                  │         │
│  ┌──────▼───────┐  ┌───────▼──────┐  ┌───────▼──────┐ │
│  │  SANE API    │  │  File I/O    │  │  Qt Image    │ │
│  └──────────────┘  └──────────────┘  └──────────────┘ │
└─────────────────────────────────────────────────────────┘
```

### 3.2 模块依赖关系

```
MainWindow
    ├── ScannerController
    │   ├── ScanWorker (QThread)
    │   └── SANE API
    ├── DocumentManager
    │   └── QPdfWriter / QImageWriter
    ├── ImageProcessor
    │   └── Qt Image Processing
    └── CroppableImageWidget
```


---

## 4. 核心模块

### 4.1 MainWindow (主窗口)

**职责**:
- UI 组织和布局管理
- 用户交互响应
- 业务模块协调
- 状态显示和更新

**关键组件**:
```cpp
class MainWindow : public QMainWindow {
    // 核心组件
    ScannerController* m_scannerController;
    ImageProcessor* m_imageProcessor;
    DocumentManager* m_documentManager;
    
    // UI 组件
    QListWidget* m_pagesList;          // 页面列表
    CroppableImageWidget* m_imageLabel; // 图像显示
    QProgressBar* m_progressBar;        // 进度条
    QStatusBar* m_statusBar;            // 状态栏
    
    // 扫描参数
    QComboBox* m_scannerComboBox;      // 扫描仪选择
    QComboBox* m_resolutionComboBox;   // 分辨率
    QComboBox* m_colorModeComboBox;    // 色彩模式
    QComboBox* m_paperSourceComboBox;  // 纸张来源
};
```

**主要功能**:
- 扫描控制（开始/停止/预览/延时）
- 图像处理（旋转/裁剪/增强）
- 文档管理（添加/删除/排序/保存）
- 智能合并（拆分/合并/裁剪）

### 4.2 ScannerController (扫描控制器)

**职责**:
- 扫描仪设备管理
- SANE 接口封装
- 扫描线程调度
- 参数配置管理

**关键接口**:
```cpp
class ScannerController : public QObject {
public:
    // 设备管理
    void refreshScanners(bool includeNetwork);
    bool selectScanner(const QString& displayName);
    QString getCurrentScanner() const;
    
    // 扫描操作
    void startScan(QString outputPath, QString colorMode, 
                   int resolution, QString paperSource, 
                   QString duplexMode);
    void startPreview(QString outputPath);
    void cancelScan();
    
    // 能力查询
    bool supportsADF() const;
    bool supportsDuplex() const;
    QStringList getAvailablePaperSources() const;
    
signals:
    void scanProgress(int percentage);
    void scanComplete(const QImage& image);
    void scanError(const QString& errorMessage);
    void scannersFound(const QStringList& scanners);
};
```

**工作流程**:
1. 初始化 SANE (`sane_init`)
2. 发现设备 (`sane_get_devices`)
3. 打开设备 (`sane_open`)
4. 设置参数 (`sane_control_option`)
5. 开始扫描 (`sane_start`)
6. 读取数据 (`sane_read`)
7. 关闭设备 (`sane_close`)

### 4.3 ScanWorker (扫描工作线程)

**职责**:
- 在独立线程执行扫描
- 避免阻塞 UI 线程
- 处理取消请求

**实现**:
```cpp
class ScanWorker : public QObject {
    Q_OBJECT
public slots:
    void doScan(QString deviceName, QString outputPath,
                QString colorMode, int resolution,
                QString paperSource, QString duplexMode);
    void doPreview(QString deviceName, QString outputPath);
    void doCancel();
    
signals:
    void scanProgress(int percentage);
    void scanComplete(const QImage& image);
    void scanError(const QString& errorMessage);
    
private:
    QMutex m_cancelMutex;
    bool m_cancelRequested;
};
```

**线程模型**:
```
主线程 (UI)                扫描线程 (Worker)
    │                           │
    ├─ startScan() ────────────>│
    │                           ├─ sane_open()
    │                           ├─ sane_start()
    │<──── scanProgress() ──────┤
    │                           ├─ sane_read()
    │<──── scanProgress() ──────┤
    │                           ├─ sane_read()
    │<──── scanComplete() ──────┤
    │                           └─ sane_close()
```

### 4.4 ImageProcessor (图像处理器)

**职责**:
- 图像基础操作
- 图像增强处理
- 文档优化
- 智能拆分合并

**功能分类**:

#### 基础操作
```cpp
QImage rotateLeft(const QImage& image);
QImage rotateRight(const QImage& image);
QImage rotateByAngle(const QImage& image, double angle);
QImage crop(const QImage& image, const QRect& rect);
```

#### 色彩调整
```cpp
QImage adjustBrightnessContrast(const QImage& image, 
                                 int brightness, int contrast);
QImage adjustHueSaturation(const QImage& image, 
                           int hue, int saturation);
QImage autoLevel(const QImage& image);
```

#### 图像增强
```cpp
QImage sharpen(const QImage& image, int amount);
QImage denoise(const QImage& image, int level);
QImage convertToGrayscale(const QImage& image);
QImage convertToBinary(const QImage& image, int threshold);
```

#### 文档处理
```cpp
QImage deskew(const QImage& image);
QImage removeBorder(const QImage& image, int borderWidth);
QImage enhanceText(const QImage& image);
QImage bleachBackground(const QImage& image, 
                        int threshold, int strength);
```

#### 智能功能
```cpp
bool isBlankPage(const QImage& image, 
                 int threshold, double blankRatio);
QImage autoTrimWhitespace(const QImage& image, int threshold);
QRect detectContentBounds(const QImage& image, int threshold);
```

#### 拆分合并
```cpp
QImage mergeImagesHorizontal(const QImage& img1, 
                              const QImage& img2, int spacing);
QImage mergeImagesVertical(const QImage& img1, 
                            const QImage& img2, int spacing);
QVector<QImage> splitImageHorizontal(const QImage& image, 
                                      int threshold);
QVector<QImage> splitImageVertical(const QImage& image, 
                                    int threshold);
QVector<QRect> detectMultipleContentBounds(const QImage& image,
                                            bool isHorizontal,
                                            int threshold);
```

### 4.5 DocumentManager (文档管理器)

**职责**:
- 多页文档管理
- 页面操作（增删改查）
- 文档导出（PDF/TIFF/图片）

**接口**:
```cpp
class DocumentManager : public QObject {
public:
    // 页面管理
    void addImage(const QImage& image);
    void insertImage(int index, const QImage& image);
    void removePage(int index);
    void movePage(int oldIndex, int newIndex);
    
    // 页面访问
    QImage getPage(int index) const;
    QImage getCurrentImage() const;
    int getCurrentPageIndex() const;
    int getPageCount() const;
    
    // 文档操作
    void newDocument();
    bool loadDocument(const QString& filePath);
    bool saveDocument(const QString& filePath);
    bool exportDocument(const QString& filePath, 
                        const QString& format);
    
signals:
    void pageAdded(int index);
    void pageRemoved(int index);
    void pageMoved(int oldIndex, int newIndex);
    void currentPageChanged(int index);
};
```

**导出格式**:
- **PDF**: 单页或多页 PDF 文档
- **TIFF**: 单页或多页 TIFF 图像
- **PNG/JPEG/BMP**: 单独的图片文件


---

## 5. 数据流

### 5.1 扫描数据流

```
用户操作 → MainWindow → ScannerController → ScanWorker
                                                  ↓
                                            SANE API
                                                  ↓
                                            扫描仪硬件
                                                  ↓
                                            原始图像数据
                                                  ↓
ScanWorker → scanComplete(QImage) → ScannerController
                                                  ↓
                                            MainWindow
                                                  ↓
                                    ImageProcessor (可选处理)
                                                  ↓
                                            DocumentManager
                                                  ↓
                                            显示/保存
```

### 5.2 图像处理数据流

```
原始图像 (QImage)
    ↓
ImageProcessor::处理函数
    ↓
处理后图像 (QImage)
    ↓
更新到 DocumentManager
    ↓
刷新显示
```

### 5.3 智能合并数据流 (v1.6.5)

```
选择2张图片
    ↓
检测内容区域 (detectMultipleContentBounds)
    ↓
是否需要拆分？
    ├─ 是 → 拆分图片 (splitImage*)
    │         ↓
    │     保留需要的部分
    │         ↓
    │     自动裁剪空白 (autoTrimWhitespace)
    │         ↓
    └─ 否 → 直接裁剪空白
              ↓
          合并图片 (mergeImages*)
              ↓
          添加到文档
```

### 5.4 文档保存数据流

```
DocumentManager 页面列表
    ↓
选择保存格式
    ├─ PDF → QPdfWriter
    │           ↓
    │       逐页绘制
    │           ↓
    │       生成 PDF 文件
    │
    ├─ TIFF → QImageWriter (多页)
    │           ↓
    │       逐页写入
    │           ↓
    │       生成 TIFF 文件
    │
    └─ 图片 → QImageWriter (单页)
                ↓
            逐页保存
                ↓
            生成多个图片文件
```

---

## 6. 线程模型

### 6.1 线程架构

```
┌─────────────────────────────────────────────────────┐
│                    主线程 (UI)                       │
│  ┌──────────────┐  ┌──────────────┐  ┌──────────┐ │
│  │  MainWindow  │  │  Document    │  │  Image   │ │
│  │              │  │  Manager     │  │  Display │ │
│  └──────┬───────┘  └──────────────┘  └──────────┘ │
└─────────┼───────────────────────────────────────────┘
          │ 信号/槽 (QueuedConnection)
┌─────────┼───────────────────────────────────────────┐
│         │              扫描线程                      │
│  ┌──────▼───────┐                                   │
│  │  ScanWorker  │                                   │
│  │              │                                   │
│  │  - doScan()  │                                   │
│  │  - doCancel()│                                   │
│  └──────┬───────┘                                   │
└─────────┼───────────────────────────────────────────┘
          │
┌─────────▼───────────────────────────────────────────┐
│                   SANE 库                            │
│  - sane_open()                                      │
│  - sane_start()                                     │
│  - sane_read()  (阻塞调用)                          │
│  - sane_close()                                     │
└─────────────────────────────────────────────────────┘
```

### 6.2 线程通信

**信号槽机制**:
```cpp
// 主线程 → 扫描线程
connect(this, &ScannerController::startScanWorker,
        m_worker, &ScanWorker::doScan,
        Qt::QueuedConnection);

// 扫描线程 → 主线程
connect(m_worker, &ScanWorker::scanProgress,
        this, &ScannerController::scanProgress,
        Qt::QueuedConnection);
```

**取消机制**:
```cpp
// 主线程请求取消
void ScannerController::cancelScan() {
    emit cancelScanWorker();
}

// 扫描线程响应取消
void ScanWorker::doCancel() {
    QMutexLocker locker(&m_cancelMutex);
    m_cancelRequested = true;
}

// 扫描循环中检查
while (sane_read(...) == SANE_STATUS_GOOD) {
    QMutexLocker locker(&m_cancelMutex);
    if (m_cancelRequested) {
        break;  // 退出扫描
    }
}
```

### 6.3 线程安全

**保护措施**:
- 使用 `QMutex` 保护共享数据
- 使用 `Qt::QueuedConnection` 跨线程通信
- 避免在工作线程中直接操作 UI
- 使用信号槽传递数据副本

---

## 7. SANE集成

### 7.1 SANE 架构

```
lz-scan 应用
    ↓
ScannerController
    ↓
SANE API (libsane)
    ↓
SANE 后端 (Backend)
    ├─ net (网络设备)
    ├─ plustek (Plustek 扫描仪)
    ├─ epson2 (Epson 扫描仪)
    ├─ hpaio (HP 扫描仪)
    └─ ...
    ↓
扫描仪硬件
```

### 7.2 设备发现

**流程**:
```cpp
// 1. 初始化 SANE
sane_init(&version, nullptr);

// 2. 获取设备列表
const SANE_Device** device_list;
sane_get_devices(&device_list, SANE_FALSE);

// 3. 遍历设备
for (int i = 0; device_list[i]; i++) {
    QString name = device_list[i]->name;
    QString vendor = device_list[i]->vendor;
    QString model = device_list[i]->model;
    QString type = device_list[i]->type;
    
    // 判断是否为网络设备
    bool isNetwork = name.contains("net:") || 
                     name.contains("tcp:") ||
                     name.contains("http:");
    
    // 构建显示名称
    QString displayName = buildDisplayName(vendor, model, name);
}
```

**设备名称处理**:
- **本地设备**: 清理 libusb 地址，只显示品牌和型号
- **网络设备**: 保留完整连接信息（IP、端口、序列号）
- **重名处理**: 自动添加 (2)、(3) 等后缀

### 7.3 参数设置

**常用参数**:
```cpp
// 分辨率
sane_control_option(handle, resolution_option, 
                    SANE_ACTION_SET_VALUE, &dpi, nullptr);

// 色彩模式
const char* mode = "Color";  // "Gray", "Lineart"
sane_control_option(handle, mode_option,
                    SANE_ACTION_SET_VALUE, &mode, nullptr);

// 扫描区域
SANE_Fixed tl_x, tl_y, br_x, br_y;
sane_control_option(handle, tl_x_option,
                    SANE_ACTION_SET_VALUE, &tl_x, nullptr);

// 纸张来源
const char* source = "ADF";  // "Flatbed"
sane_control_option(handle, source_option,
                    SANE_ACTION_SET_VALUE, &source, nullptr);
```

### 7.4 数据读取

**扫描循环**:
```cpp
SANE_Status status;
SANE_Int len;
SANE_Byte buffer[32768];

// 开始扫描
sane_start(handle);

// 获取参数
SANE_Parameters params;
sane_get_parameters(handle, &params);

// 读取数据
QByteArray imageData;
while (true) {
    status = sane_read(handle, buffer, sizeof(buffer), &len);
    
    if (status == SANE_STATUS_GOOD) {
        imageData.append((char*)buffer, len);
        // 更新进度
        int progress = (imageData.size() * 100) / totalBytes;
        emit scanProgress(progress);
    } else if (status == SANE_STATUS_EOF) {
        break;  // 扫描完成
    } else {
        // 错误处理
        emit scanError(sane_strstatus(status));
        break;
    }
}

// 转换为 QImage
QImage image = convertToQImage(imageData, params);
```


---

## 8. 图像处理

### 8.1 处理管线

```
原始图像
    ↓
┌─────────────────┐
│  基础操作       │ 旋转、裁剪
└────────┬────────┘
         ↓
┌─────────────────┐
│  色彩调整       │ 亮度、对比度、色度、饱和度
└────────┬────────┘
         ↓
┌─────────────────┐
│  图像增强       │ 锐化、去噪、自动色阶
└────────┬────────┘
         ↓
┌─────────────────┐
│  文档优化       │ 倾斜校正、背景漂白、文本增强
└────────┬────────┘
         ↓
处理后图像
```

### 8.2 卷积滤波器

**实现**:
```cpp
QImage applyFilter(const QImage& image, 
                   const QVector<qreal>& kernel,
                   int kernelSize) {
    // 对每个像素应用卷积核
    for (int y = 0; y < height; y++) {
        for (int x = 0; x < width; x++) {
            qreal sumR = 0, sumG = 0, sumB = 0;
            
            // 应用卷积核
            for (int ky = 0; ky < kernelSize; ky++) {
                for (int kx = 0; kx < kernelSize; kx++) {
                    int sourceX = x + kx - offset;
                    int sourceY = y + ky - offset;
                    
                    QRgb pixel = getPixel(sourceX, sourceY);
                    qreal kernelValue = kernel[ky * kernelSize + kx];
                    
                    sumR += qRed(pixel) * kernelValue;
                    sumG += qGreen(pixel) * kernelValue;
                    sumB += qBlue(pixel) * kernelValue;
                }
            }
            
            setPixel(x, y, qRgb(sumR, sumG, sumB));
        }
    }
}
```

**常用卷积核**:
```cpp
// 锐化
QVector<qreal> sharpenKernel = {
    -1, -1, -1,
    -1,  9, -1,
    -1, -1, -1
};

// 高斯模糊
QVector<qreal> blurKernel = {
    1/16.0, 2/16.0, 1/16.0,
    2/16.0, 4/16.0, 2/16.0,
    1/16.0, 2/16.0, 1/16.0
};

// 边缘检测 (Sobel)
QVector<qreal> sobelX = {
    -1, 0, 1,
    -2, 0, 2,
    -1, 0, 1
};
```

### 8.3 空白页检测

**算法**:
```cpp
bool isBlankPage(const QImage& image, 
                 int threshold, double blankRatio) {
    int totalPixels = image.width() * image.height();
    int blankPixels = 0;
    
    // 统计空白像素
    for (int y = 0; y < image.height(); y++) {
        for (int x = 0; x < image.width(); x++) {
            QRgb pixel = image.pixel(x, y);
            int brightness = (qRed(pixel) + qGreen(pixel) + 
                             qBlue(pixel)) / 3;
            
            if (brightness > threshold) {
                blankPixels++;
            }
        }
    }
    
    // 计算空白比例
    double ratio = (double)blankPixels / totalPixels;
    return ratio >= blankRatio;
}
```

**参数**:
- `threshold`: 亮度阈值（默认 240）
- `blankRatio`: 空白比例阈值（默认 0.98，即 98%）

---

## 9. 智能合并

### 9.1 功能概述 (v1.6.5 核心功能)

智能合并是 v1.6.5 的核心功能，能够：
- 自动检测图片中的多个内容区域
- 智能拆分并保留需要的部分
- 自动裁剪空白边缘
- 智能合并成一张图片

### 9.2 内容区域检测

**算法**:
```cpp
QVector<QRect> detectMultipleContentBounds(
    const QImage& image, bool isHorizontal, int threshold) {
    
    // 1. 转换为灰度图
    QImage grayImage = convertToGrayscale(image);
    
    // 2. 二值化
    QImage binaryImage = applyThreshold(grayImage, threshold);
    
    // 3. 投影分析
    QVector<int> projection;
    if (isHorizontal) {
        // 水平投影（检测左右分布）
        projection = horizontalProjection(binaryImage);
    } else {
        // 垂直投影（检测上下分布）
        projection = verticalProjection(binaryImage);
    }
    
    // 4. 查找分隔区域
    QVector<int> gaps = findGaps(projection, minGapSize);
    
    // 5. 确定内容边界
    QVector<QRect> bounds;
    for (int i = 0; i < gaps.size() - 1; i++) {
        QRect rect = calculateBounds(gaps[i], gaps[i+1]);
        bounds.append(rect);
    }
    
    return bounds;
}
```

**投影分析**:
```
原始图像:              垂直投影:
┌─────────────┐       
│   内容A     │       ████████████
│             │       ████████████
├─────────────┤       ░░░░░░░░░░  ← 分隔区域
│   空白      │       ░░░░░░░░░░
├─────────────┤       ░░░░░░░░░░
│   内容B     │       ████████████
│             │       ████████████
└─────────────┘       
```

### 9.3 智能拆分

**左右拆分**:
```cpp
QVector<QImage> splitImageHorizontal(const QImage& image, 
                                      int threshold) {
    // 1. 检测多个内容区域
    QVector<QRect> bounds = 
        detectMultipleContentBounds(image, true, threshold);
    
    // 2. 提取每个区域
    QVector<QImage> parts;
    for (const QRect& rect : bounds) {
        QImage part = image.copy(rect);
        parts.append(part);
    }
    
    return parts;
}
```

**上下拆分**:
```cpp
QVector<QImage> splitImageVertical(const QImage& image,
                                    int threshold) {
    // 类似左右拆分，但使用垂直投影
    QVector<QRect> bounds = 
        detectMultipleContentBounds(image, false, threshold);
    
    QVector<QImage> parts;
    for (const QRect& rect : bounds) {
        QImage part = image.copy(rect);
        parts.append(part);
    }
    
    return parts;
}
```

### 9.4 自动裁剪

**算法**:
```cpp
QImage autoTrimWhitespace(const QImage& image, int threshold) {
    // 1. 检测内容边界
    QRect bounds = detectContentBounds(image, threshold);
    
    // 2. 裁剪到内容区域
    if (bounds.isValid()) {
        return image.copy(bounds);
    }
    
    return image;
}

QRect detectContentBounds(const QImage& image, int threshold) {
    int left = image.width();
    int right = 0;
    int top = image.height();
    int bottom = 0;
    
    // 扫描所有像素，找到内容边界
    for (int y = 0; y < image.height(); y++) {
        for (int x = 0; x < image.width(); x++) {
            QRgb pixel = image.pixel(x, y);
            int brightness = (qRed(pixel) + qGreen(pixel) + 
                             qBlue(pixel)) / 3;
            
            if (brightness < threshold) {
                // 这是内容像素
                left = qMin(left, x);
                right = qMax(right, x);
                top = qMin(top, y);
                bottom = qMax(bottom, y);
            }
        }
    }
    
    return QRect(left, top, right - left + 1, bottom - top + 1);
}
```

### 9.5 智能合并

**左右合并**:
```cpp
QImage mergeImagesHorizontal(const QImage& img1,
                              const QImage& img2,
                              int spacing) {
    // 1. 计算合并后的尺寸
    int maxHeight = qMax(img1.height(), img2.height());
    int totalWidth = img1.width() + spacing + img2.width();
    
    // 2. 创建新图像（白色背景）
    QImage merged(totalWidth, maxHeight, QImage::Format_RGB32);
    merged.fill(Qt::white);
    
    // 3. 绘制第一张图片（居中对齐）
    QPainter painter(&merged);
    int y1 = (maxHeight - img1.height()) / 2;
    painter.drawImage(0, y1, img1);
    
    // 4. 绘制第二张图片（居中对齐）
    int x2 = img1.width() + spacing;
    int y2 = (maxHeight - img2.height()) / 2;
    painter.drawImage(x2, y2, img2);
    
    return merged;
}
```

**上下合并**:
```cpp
QImage mergeImagesVertical(const QImage& img1,
                            const QImage& img2,
                            int spacing) {
    // 类似左右合并，但垂直排列
    int maxWidth = qMax(img1.width(), img2.width());
    int totalHeight = img1.height() + spacing + img2.height();
    
    QImage merged(maxWidth, totalHeight, QImage::Format_RGB32);
    merged.fill(Qt::white);
    
    QPainter painter(&merged);
    
    // 第一张图片（水平居中）
    int x1 = (maxWidth - img1.width()) / 2;
    painter.drawImage(x1, 0, img1);
    
    // 第二张图片（水平居中）
    int x2 = (maxWidth - img2.width()) / 2;
    int y2 = img1.height() + spacing;
    painter.drawImage(x2, y2, img2);
    
    return merged;
}
```

### 9.6 完整工作流程

**智能合并流程**:
```
用户选择2张图片
    ↓
MainWindow::smartMergeImages()
    ↓
┌─────────────────────────────────────┐
│ 1. 检测第一张图片的内容区域         │
│    detectMultipleContentBounds()    │
└────────────┬────────────────────────┘
             ↓
        是否有多个区域？
             ├─ 是 → 拆分图片
             │       splitImage*()
             │           ↓
             │       用户选择保留哪部分
             │           ↓
             └─ 否 → 使用原图
                         ↓
┌─────────────────────────────────────┐
│ 2. 检测第二张图片的内容区域         │
│    detectMultipleContentBounds()    │
└────────────┬────────────────────────┘
             ↓
        是否有多个区域？
             ├─ 是 → 拆分图片
             │       splitImage*()
             │           ↓
             │       用户选择保留哪部分
             │           ↓
             └─ 否 → 使用原图
                         ↓
┌─────────────────────────────────────┐
│ 3. 自动裁剪空白边缘                 │
│    autoTrimWhitespace()             │
└────────────┬────────────────────────┘
             ↓
┌─────────────────────────────────────┐
│ 4. 合并图片                         │
│    mergeImages*(左右/上下)          │
└────────────┬────────────────────────┘
             ↓
┌─────────────────────────────────────┐
│ 5. 添加到文档                       │
│    DocumentManager::addImage()      │
└─────────────────────────────────────┘
```

**用户交互**:
```cpp
// 当检测到多个内容区域时，弹出选择对话框
QStringList choices;
choices << "保留左侧部分" << "保留右侧部分" 
        << "保留全部" << "取消合并";

QString choice = QInputDialog::getItem(
    this, "选择保留部分",
    "检测到图片包含多个内容区域，请选择：",
    choices, 0, false);
```

---

## 10. 文档管理

### 10.1 数据结构

**页面存储**:
```cpp
class DocumentManager {
private:
    QVector<QImage> m_pages;           // 页面列表
    int m_currentPageIndex;            // 当前页索引
    QString m_currentFilePath;         // 当前文档路径
    bool m_modified;                   // 修改标志
};
```

**页面元数据**:
```cpp
struct PageMetadata {
    QString originalPath;              // 原始文件路径
    QDateTime scanTime;                // 扫描时间
    int resolution;                    // 分辨率
    QString colorMode;                 // 色彩模式
    QSize originalSize;                // 原始尺寸
    bool processed;                    // 是否已处理
};
```

### 10.2 页面操作

**添加页面**:
```cpp
void DocumentManager::addImage(const QImage& image) {
    m_pages.append(image);
    m_modified = true;
    emit pageAdded(m_pages.size() - 1);
}

void DocumentManager::insertImage(int index, const QImage& image) {
    m_pages.insert(index, image);
    m_modified = true;
    emit pageAdded(index);
}
```

**删除页面**:
```cpp
void DocumentManager::removePage(int index) {
    if (index >= 0 && index < m_pages.size()) {
        m_pages.removeAt(index);
        m_modified = true;
        emit pageRemoved(index);
        
        // 调整当前页索引
        if (m_currentPageIndex >= m_pages.size()) {
            m_currentPageIndex = m_pages.size() - 1;
        }
    }
}
```

**移动页面**:
```cpp
void DocumentManager::movePage(int oldIndex, int newIndex) {
    if (oldIndex >= 0 && oldIndex < m_pages.size() &&
        newIndex >= 0 && newIndex < m_pages.size()) {
        
        QImage page = m_pages.takeAt(oldIndex);
        m_pages.insert(newIndex, page);
        m_modified = true;
        emit pageMoved(oldIndex, newIndex);
    }
}
```

### 10.3 文档导出

**PDF 导出**:
```cpp
bool DocumentManager::exportToPDF(const QString& filePath) {
    QPdfWriter pdfWriter(filePath);
    pdfWriter.setPageSize(QPageSize(QPageSize::A4));
    pdfWriter.setResolution(300);
    
    QPainter painter(&pdfWriter);
    
    for (int i = 0; i < m_pages.size(); i++) {
        if (i > 0) {
            pdfWriter.newPage();  // 新页面
        }
        
        // 计算缩放以适应页面
        QImage page = m_pages[i];
        QRect pageRect = pdfWriter.pageLayout().paintRectPixels(300);
        
        QSize scaledSize = page.size().scaled(
            pageRect.size(), Qt::KeepAspectRatio);
        
        // 居中绘制
        int x = (pageRect.width() - scaledSize.width()) / 2;
        int y = (pageRect.height() - scaledSize.height()) / 2;
        
        painter.drawImage(QRect(x, y, scaledSize.width(), 
                                scaledSize.height()), page);
    }
    
    painter.end();
    return true;
}
```

**多页 TIFF 导出**:
```cpp
bool DocumentManager::exportToTIFF(const QString& filePath) {
    if (m_pages.isEmpty()) {
        return false;
    }
    
    // 保存第一页
    QImageWriter writer(filePath, "TIFF");
    writer.setCompression(1);  // LZW 压缩
    
    if (!writer.write(m_pages[0])) {
        return false;
    }
    
    // 追加其他页面
    for (int i = 1; i < m_pages.size(); i++) {
        // TIFF 多页支持需要特殊处理
        // 这里简化处理，实际可能需要使用 libtiff
        QString pagePath = QString("%1_page%2.tiff")
            .arg(filePath).arg(i + 1);
        
        if (!m_pages[i].save(pagePath, "TIFF")) {
            return false;
        }
    }
    
    return true;
}
```

**图片序列导出**:
```cpp
bool DocumentManager::exportToImages(const QString& basePath,
                                      const QString& format) {
    for (int i = 0; i < m_pages.size(); i++) {
        QString filePath = QString("%1_page%2.%3")
            .arg(basePath)
            .arg(i + 1, 3, 10, QChar('0'))  // 001, 002, ...
            .arg(format.toLower());
        
        if (!m_pages[i].save(filePath, format.toUtf8())) {
            return false;
        }
    }
    
    return true;
}
```

### 10.4 文档加载

**支持格式**:
- 单页图片: PNG, JPEG, BMP, TIFF
- 多页 PDF: 使用 Poppler 或 Qt PDF 模块
- 多页 TIFF: 使用 libtiff

**加载流程**:
```cpp
bool DocumentManager::loadDocument(const QString& filePath) {
    QFileInfo fileInfo(filePath);
    QString suffix = fileInfo.suffix().toLower();
    
    if (suffix == "pdf") {
        return loadPDF(filePath);
    } else if (suffix == "tiff" || suffix == "tif") {
        return loadTIFF(filePath);
    } else {
        // 单页图片
        QImage image(filePath);
        if (!image.isNull()) {
            newDocument();
            addImage(image);
            m_currentFilePath = filePath;
            m_modified = false;
            return true;
        }
    }
    
    return false;
}
```

---

## 11. 性能优化

### 11.1 图像处理优化

**多线程处理**:
```cpp
// 使用 QtConcurrent 并行处理多页
QFuture<QImage> future = QtConcurrent::mapped(
    m_pages,
    [](const QImage& page) {
        return ImageProcessor::sharpen(page, 50);
    }
);

// 等待完成
QVector<QImage> processedPages = future.results();
```

**内存优化**:
```cpp
// 使用 QImage::Format_RGB888 而不是 Format_ARGB32
// 节省 25% 内存（3字节 vs 4字节）
QImage image(width, height, QImage::Format_RGB888);

// 及时释放不需要的图像
QImage temp = processImage(original);
original = QImage();  // 释放原图内存
```

**缓存策略**:
```cpp
class ImageCache {
private:
    QCache<QString, QImage> m_cache;  // LRU 缓存
    
public:
    ImageCache() {
        m_cache.setMaxCost(100 * 1024 * 1024);  // 100MB
    }
    
    QImage* get(const QString& key) {
        return m_cache.object(key);
    }
    
    void insert(const QString& key, QImage* image) {
        int cost = image->sizeInBytes();
        m_cache.insert(key, image, cost);
    }
};
```

### 11.2 扫描优化

**批量扫描**:
```cpp
// ADF 批量扫描，避免重复打开关闭设备
void scanMultiplePages() {
    sane_open(device);
    
    while (hasMorePages()) {
        sane_start(handle);
        QImage page = readPage();
        emit pageScanned(page);
    }
    
    sane_close(handle);
}
```

**预览优化**:
```cpp
// 预览使用低分辨率
void startPreview() {
    int previewDPI = 75;  // 而不是 300
    setResolution(previewDPI);
    startScan();
}
```

### 11.3 UI 响应优化

**延迟加载**:
```cpp
// 页面列表使用缩略图
QImage thumbnail = page.scaled(
    200, 200,
    Qt::KeepAspectRatio,
    Qt::SmoothTransformation
);

QListWidgetItem* item = new QListWidgetItem();
item->setIcon(QPixmap::fromImage(thumbnail));
```

**增量更新**:
```cpp
// 扫描进度增量更新，避免频繁刷新
void updateProgress(int progress) {
    static int lastProgress = 0;
    
    if (progress - lastProgress >= 5) {  // 每 5% 更新一次
        m_progressBar->setValue(progress);
        lastProgress = progress;
    }
}
```

**异步操作**:
```cpp
// 耗时操作使用 QFuture
QFuture<void> future = QtConcurrent::run([this]() {
    // 耗时的图像处理
    processAllPages();
});

// 监听完成
QFutureWatcher<void>* watcher = new QFutureWatcher<void>();
connect(watcher, &QFutureWatcher<void>::finished,
        this, &MainWindow::onProcessingComplete);
watcher->setFuture(future);
```

### 11.4 内存管理

**智能指针**:
```cpp
// 使用智能指针管理资源
std::unique_ptr<SANE_Handle> m_handle;
std::shared_ptr<QImage> m_currentImage;
```

**资源池**:
```cpp
class ImageBufferPool {
private:
    QQueue<QByteArray*> m_pool;
    QMutex m_mutex;
    
public:
    QByteArray* acquire(int size) {
        QMutexLocker locker(&m_mutex);
        
        if (!m_pool.isEmpty()) {
            QByteArray* buffer = m_pool.dequeue();
            buffer->resize(size);
            return buffer;
        }
        
        return new QByteArray(size, 0);
    }
    
    void release(QByteArray* buffer) {
        QMutexLocker locker(&m_mutex);
        
        if (m_pool.size() < 10) {  // 最多缓存 10 个
            m_pool.enqueue(buffer);
        } else {
            delete buffer;
        }
    }
};
```

---

## 12. 扩展性

### 12.1 插件架构

**设计思路**:
```cpp
// 插件接口
class IScanPlugin {
public:
    virtual ~IScanPlugin() = default;
    
    virtual QString name() const = 0;
    virtual QString version() const = 0;
    virtual QString description() const = 0;
    
    virtual bool initialize() = 0;
    virtual void shutdown() = 0;
    
    virtual QWidget* createConfigWidget() = 0;
    virtual QImage processImage(const QImage& input) = 0;
};

// 插件管理器
class PluginManager {
public:
    void loadPlugins(const QString& pluginDir);
    QList<IScanPlugin*> getPlugins() const;
    IScanPlugin* getPlugin(const QString& name) const;
    
private:
    QMap<QString, IScanPlugin*> m_plugins;
};
```

**插件加载**:
```cpp
void PluginManager::loadPlugins(const QString& pluginDir) {
    QDir dir(pluginDir);
    QStringList filters;
    filters << "*.so" << "*.dll" << "*.dylib";
    
    for (const QString& fileName : dir.entryList(filters)) {
        QString filePath = dir.absoluteFilePath(fileName);
        
        QPluginLoader loader(filePath);
        QObject* plugin = loader.instance();
        
        if (plugin) {
            IScanPlugin* scanPlugin = 
                qobject_cast<IScanPlugin*>(plugin);
            
            if (scanPlugin && scanPlugin->initialize()) {
                m_plugins[scanPlugin->name()] = scanPlugin;
            }
        }
    }
}
```

### 12.2 自定义处理器

**处理器接口**:
```cpp
class IImageProcessor {
public:
    virtual ~IImageProcessor() = default;
    
    virtual QString name() const = 0;
    virtual QString category() const = 0;  // "基础", "增强", "文档"
    virtual QStringList parameters() const = 0;
    
    virtual QImage process(const QImage& input,
                          const QVariantMap& params) = 0;
};

// 示例：自定义锐化处理器
class CustomSharpenProcessor : public IImageProcessor {
public:
    QString name() const override {
        return "自定义锐化";
    }
    
    QString category() const override {
        return "增强";
    }
    
    QStringList parameters() const override {
        return {"amount", "radius", "threshold"};
    }
    
    QImage process(const QImage& input,
                   const QVariantMap& params) override {
        int amount = params.value("amount", 100).toInt();
        int radius = params.value("radius", 1).toInt();
        int threshold = params.value("threshold", 0).toInt();
        
        // 自定义锐化算法
        return customSharpen(input, amount, radius, threshold);
    }
};
```

### 12.3 脚本支持

**设计思路**:
```cpp
// 使用 QtScript 或 Python 绑定
class ScriptEngine {
public:
    void registerAPI();
    QVariant evaluate(const QString& script);
    
    // 暴露给脚本的 API
    void exposeFunction(const QString& name,
                       std::function<QVariant(QVariantList)> func);
};

// 注册 API
void ScriptEngine::registerAPI() {
    // 图像处理
    exposeFunction("rotate", [](QVariantList args) {
        QImage image = args[0].value<QImage>();
        double angle = args[1].toDouble();
        return ImageProcessor::rotateByAngle(image, angle);
    });
    
    // 文档操作
    exposeFunction("addPage", [](QVariantList args) {
        QImage image = args[0].value<QImage>();
        DocumentManager::instance()->addImage(image);
        return QVariant();
    });
}
```

**脚本示例**:
```javascript
// 批量处理脚本
for (var i = 0; i < pageCount(); i++) {
    var page = getPage(i);
    
    // 旋转 90 度
    page = rotate(page, 90);
    
    // 自动裁剪
    page = autoTrim(page, 240);
    
    // 增强对比度
    page = adjustContrast(page, 20);
    
    // 更新页面
    setPage(i, page);
}
```

### 12.4 配置系统

**配置管理**:
```cpp
class ConfigManager {
public:
    static ConfigManager* instance();
    
    // 读取配置
    QVariant get(const QString& key, 
                 const QVariant& defaultValue = QVariant());
    
    // 写入配置
    void set(const QString& key, const QVariant& value);
    
    // 配置分组
    void beginGroup(const QString& group);
    void endGroup();
    
    // 保存/加载
    bool save();
    bool load();
    
private:
    QSettings* m_settings;
};
```

**配置文件**:
```ini
[General]
language=zh_CN
theme=light
autoSave=true

[Scan]
defaultResolution=300
defaultColorMode=Color
defaultPaperSource=Flatbed
enableNetworkScan=true

[ImageProcessing]
autoTrimThreshold=240
blankPageThreshold=240
blankPageRatio=0.98
defaultSharpenAmount=50

[Export]
defaultFormat=PDF
pdfCompression=true
jpegQuality=90
```

### 12.5 国际化

**多语言支持**:
```cpp
// 翻译文件加载
void MainWindow::loadTranslation(const QString& locale) {
    QTranslator* translator = new QTranslator(this);
    
    QString qmFile = QString("lz-scan_%1.qm").arg(locale);
    QString qmPath = QCoreApplication::applicationDirPath() + 
                     "/translations/" + qmFile;
    
    if (translator->load(qmPath)) {
        qApp->installTranslator(translator);
    }
}

// 使用翻译
QString text = tr("扫描完成");
QString status = tr("已扫描 %1 页").arg(pageCount);
```

**翻译文件** (lz-scan_en.ts):
```xml
<?xml version="1.0" encoding="utf-8"?>
<TS version="2.1" language="en_US">
    <context>
        <name>MainWindow</name>
        <message>
            <source>扫描完成</source>
            <translation>Scan Complete</translation>
        </message>
        <message>
            <source>已扫描 %1 页</source>
            <translation>Scanned %1 page(s)</translation>
        </message>
    </context>
</TS>
```

### 12.6 未来扩展方向

**可能的功能扩展**:

1. **OCR 集成**
   - 集成 Tesseract OCR
   - 文本识别和提取
   - 可搜索 PDF 生成

2. **云存储**
   - 直接上传到云盘
   - 支持 WebDAV、FTP
   - 自动备份

3. **批量处理**
   - 批量扫描任务
   - 预设处理流程
   - 定时扫描

4. **高级编辑**
   - 页面注释
   - 水印添加
   - 签名功能

5. **网络功能**
   - 远程扫描控制
   - 扫描共享
   - 协作编辑

6. **AI 功能**
   - 智能文档分类
   - 自动命名
   - 内容识别

---

## 附录

### A. 编译说明

**依赖安装** (Ubuntu/Debian):
```bash
sudo apt-get install \
    build-essential \
    cmake \
    qt5-default \
    libqt5svg5-dev \
    libsane-dev \
    libpoppler-qt5-dev
```

**编译步骤**:
```bash
mkdir build
cd build
cmake ..
make -j$(nproc)
sudo make install
```

### B. 调试技巧

**启用调试输出**:
```bash
export QT_LOGGING_RULES="*.debug=true"
export SANE_DEBUG_DLL=128
./lz-scan
```

**GDB 调试**:
```bash
gdb ./lz-scan
(gdb) break MainWindow::startScan
(gdb) run
(gdb) backtrace
```

**Valgrind 内存检查**:
```bash
valgrind --leak-check=full \
         --show-leak-kinds=all \
         ./lz-scan
```

### C. 性能分析

**使用 perf**:
```bash
perf record -g ./lz-scan
perf report
```

**Qt Creator Profiler**:
- 打开 Qt Creator
- 选择 "Analyze" → "QML Profiler"
- 运行应用并分析性能

### D. 参考资源

**官方文档**:
- Qt Documentation: https://doc.qt.io/
- SANE Documentation: http://www.sane-project.org/docs.html

**相关项目**:
- Simple Scan: https://gitlab.gnome.org/GNOME/simple-scan
- Skanlite: https://github.com/KDE/skanlite
- NAPS2: https://github.com/cyanfish/naps2

**技术文章**:
- Qt 多线程编程最佳实践
- SANE 后端开发指南
- 图像处理算法详解

---

## 结语

亮仔扫描 (lz-scan) 采用模块化、可扩展的架构设计，充分利用 Qt 和 SANE 的能力，提供了完整的文档扫描解决方案。

**核心优势**:
- ✅ 清晰的模块划分
- ✅ 线程安全的扫描操作
- ✅ 强大的图像处理能力
- ✅ 智能拆分合并功能 (v1.6.5)
- ✅ 良好的扩展性

**持续改进**:
- 性能优化
- 功能扩展
- 用户体验提升
- 代码质量改进

欢迎贡献代码和反馈建议！

---

**文档版本**: 1.0  
**最后更新**: 2025年10月10日  
**维护者**: 克亮
