#!/bin/bash

# OBS 打包检查脚本
# 用途: 在上传到 OBS 之前检查所有文件是否正确

set -e

VERSION="1.6.6"
PACKAGE_NAME="lz-scan"

# 颜色定义
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

echo -e "${BLUE}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo -e "${BLUE}  OBS 打包检查${NC}"
echo -e "${BLUE}  版本: ${VERSION}${NC}"
echo -e "${BLUE}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo ""

ERRORS=0
WARNINGS=0

# 检查函数
check_pass() {
    echo -e "${GREEN}   ✓ $1${NC}"
}

check_fail() {
    echo -e "${RED}   ✗ $1${NC}"
    ERRORS=$((ERRORS + 1))
}

check_warn() {
    echo -e "${YELLOW}   ⚠️  $1${NC}"
    WARNINGS=$((WARNINGS + 1))
}

# 1. 检查版本号一致性
echo -e "${YELLOW}1. 检查版本号一致性${NC}"
echo "-------------------------------------------"

check_version() {
    local file=$1
    local pattern=$2
    local description=$3
    
    if [ -f "$file" ]; then
        if grep -q "$pattern" "$file"; then
            check_pass "$description: $VERSION"
        else
            check_fail "$description: 版本不匹配"
        fi
    else
        check_fail "$description: 文件不存在"
    fi
}

check_version "CMakeLists.txt" "VERSION ${VERSION}" "CMakeLists.txt"
check_version "lz-scan.spec" "Version:.*${VERSION}" "lz-scan.spec"
check_version "src/main.cpp" "\"${VERSION}\"" "src/main.cpp"
check_version "README.md" "version-${VERSION}" "README.md"
check_version "debian/changelog" "lz-scan (${VERSION}-1)" "debian/changelog"

echo ""

# 2. 检查必要文件
echo -e "${YELLOW}2. 检查必要文件${NC}"
echo "-------------------------------------------"

check_file() {
    local file=$1
    local description=$2
    
    if [ -f "$file" ]; then
        check_pass "$description"
    else
        check_fail "$description: 文件不存在"
    fi
}

check_file "CMakeLists.txt" "CMakeLists.txt"
check_file "lz-scan.spec" "lz-scan.spec"
check_file "README.md" "README.md"
check_file "CHANGELOG.md" "CHANGELOG.md"
check_file "LICENSE" "LICENSE"
check_file "lz-scan.svg" "lz-scan.svg"
check_file "debian/lz-scan.desktop" "debian/lz-scan.desktop"
check_file "debian/lz-scan.1" "debian/lz-scan.1"
check_file "debian/copyright" "debian/copyright"
check_file "debian/changelog" "debian/changelog"
check_file "debian/control" "debian/control"

echo ""

# 3. 检查源文件
echo -e "${YELLOW}3. 检查源文件${NC}"
echo "-------------------------------------------"

SRC_COUNT=$(find src -name "*.cpp" -o -name "*.h" 2>/dev/null | wc -l)
if [ $SRC_COUNT -gt 0 ]; then
    check_pass "源文件数量: $SRC_COUNT"
else
    check_fail "未找到源文件"
fi

if [ -d "src" ]; then
    check_pass "src 目录存在"
else
    check_fail "src 目录不存在"
fi

if [ -d "resources" ]; then
    check_pass "resources 目录存在"
else
    check_warn "resources 目录不存在"
fi

echo ""

# 4. 检查 CMakeLists.txt 配置
echo -e "${YELLOW}4. 检查 CMakeLists.txt 配置${NC}"
echo "-------------------------------------------"

if grep -q "project(lz-scan VERSION ${VERSION}" CMakeLists.txt; then
    check_pass "项目版本配置正确"
else
    check_fail "项目版本配置错误"
fi

if grep -q "install(TARGETS lz-scan" CMakeLists.txt; then
    check_pass "可执行文件安装规则"
else
    check_fail "缺少可执行文件安装规则"
fi

if grep -q "install(FILES.*desktop" CMakeLists.txt; then
    check_pass "桌面文件安装规则"
else
    check_warn "缺少桌面文件安装规则"
fi

if grep -q "install(FILES.*svg" CMakeLists.txt; then
    check_pass "图标安装规则"
else
    check_warn "缺少图标安装规则"
fi

echo ""

# 5. 检查 spec 文件
echo -e "${YELLOW}5. 检查 spec 文件${NC}"
echo "-------------------------------------------"

if grep -q "Name:.*lz-scan" lz-scan.spec; then
    check_pass "包名称正确"
else
    check_fail "包名称错误"
fi

if grep -q "Version:.*${VERSION}" lz-scan.spec; then
    check_pass "版本号正确"
else
    check_fail "版本号错误"
fi

if grep -q "%{_bindir}/%{name}" lz-scan.spec; then
    check_pass "可执行文件路径"
else
    check_fail "缺少可执行文件路径"
fi

if grep -q "%{_datadir}/applications/%{name}.desktop" lz-scan.spec; then
    check_pass "桌面文件路径"
else
    check_warn "缺少桌面文件路径"
fi

# 检查路径末尾是否有多余的 /
if grep -q "\.png/" lz-scan.spec || grep -q "\.svg/" lz-scan.spec; then
    check_fail "图标路径末尾有多余的 /"
else
    check_pass "图标路径格式正确"
fi

# 检查 changelog
if grep -q "* Sat Oct 19 2025.*${VERSION}" lz-scan.spec; then
    check_pass "spec changelog 包含 ${VERSION}"
else
    check_warn "spec changelog 可能需要更新"
fi

echo ""

# 6. 检查 BuildRequires
echo -e "${YELLOW}6. 检查 BuildRequires${NC}"
echo "-------------------------------------------"

check_buildrequires() {
    local package=$1
    if grep -q "BuildRequires:.*$package" lz-scan.spec; then
        check_pass "$package"
    else
        check_warn "缺少 $package"
    fi
}

check_buildrequires "cmake"
check_buildrequires "gcc-c++"
check_buildrequires "qt5-qtbase-devel"
check_buildrequires "sane-backends-devel"

echo ""

# 7. 检查 debian 打包文件
echo -e "${YELLOW}7. 检查 debian 打包文件${NC}"
echo "-------------------------------------------"

if [ -f "debian/control" ]; then
    if grep -q "Package: lz-scan" debian/control; then
        check_pass "debian/control 包名正确"
    else
        check_fail "debian/control 包名错误"
    fi
    
    if grep -q "qtbase5-dev" debian/control; then
        check_pass "debian/control 包含 Qt 依赖"
    else
        check_warn "debian/control 可能缺少 Qt 依赖"
    fi
fi

if [ -f "debian/changelog" ]; then
    if grep -q "lz-scan (${VERSION}-1)" debian/changelog; then
        check_pass "debian/changelog 版本正确"
    else
        check_fail "debian/changelog 版本错误"
    fi
fi

echo ""

# 8. 检查 OBS 打包文件
echo -e "${YELLOW}8. 检查 OBS 打包文件${NC}"
echo "-------------------------------------------"

if [ -d "obs_package" ]; then
    check_pass "obs_package 目录存在"
    
    if [ -f "obs_package/${PACKAGE_NAME}-${VERSION}.tar.gz" ]; then
        SIZE=$(du -h "obs_package/${PACKAGE_NAME}-${VERSION}.tar.gz" | cut -f1)
        check_pass "源码包: ${PACKAGE_NAME}-${VERSION}.tar.gz (${SIZE})"
        
        # 检查源码包内容
        if tar -tzf "obs_package/${PACKAGE_NAME}-${VERSION}.tar.gz" | grep -q "CMakeLists.txt"; then
            check_pass "源码包包含 CMakeLists.txt"
        else
            check_fail "源码包缺少 CMakeLists.txt"
        fi
        
        if tar -tzf "obs_package/${PACKAGE_NAME}-${VERSION}.tar.gz" | grep -q "src/main.cpp"; then
            check_pass "源码包包含源代码"
        else
            check_fail "源码包缺少源代码"
        fi
    else
        check_fail "源码包不存在"
    fi
    
    if [ -f "obs_package/${PACKAGE_NAME}.spec" ]; then
        check_pass "spec 文件已复制"
    else
        check_fail "spec 文件未复制"
    fi
    
    if [ -f "obs_package/debian.tar.gz" ]; then
        SIZE=$(du -h "obs_package/debian.tar.gz" | cut -f1)
        check_pass "debian.tar.gz (${SIZE})"
    else
        check_warn "debian.tar.gz 不存在（可选）"
    fi
    
    if [ -f "obs_package/${PACKAGE_NAME}.dsc" ]; then
        check_pass "${PACKAGE_NAME}.dsc"
    else
        check_warn "${PACKAGE_NAME}.dsc 不存在（可选）"
    fi
else
    check_warn "obs_package 目录不存在"
    echo -e "${YELLOW}      运行: ./package_for_obs.sh${NC}"
fi

echo ""

# 9. 检查本地编译
echo -e "${YELLOW}9. 检查本地编译${NC}"
echo "-------------------------------------------"

if [ -d "build" ]; then
    check_pass "build 目录存在"
    
    if [ -f "build/lz-scan" ]; then
        check_pass "可执行文件已编译"
        
        # 检查版本
        if strings build/lz-scan | grep -q "${VERSION}"; then
            check_pass "可执行文件版本正确"
        else
            check_warn "可执行文件版本可能不正确"
        fi
    else
        check_warn "可执行文件不存在"
    fi
else
    check_warn "build 目录不存在，建议先编译测试"
fi

echo ""

# 10. spec 文件语法检查
echo -e "${YELLOW}10. spec 文件语法检查${NC}"
echo "-------------------------------------------"

if command -v rpmlint >/dev/null 2>&1; then
    echo "   运行 rpmlint 检查..."
    RPMLINT_OUTPUT=$(rpmlint lz-scan.spec 2>&1)
    
    if echo "$RPMLINT_OUTPUT" | grep -q "error"; then
        check_fail "rpmlint 发现错误"
        echo "$RPMLINT_OUTPUT" | grep "error" | head -3
    else
        check_pass "rpmlint 检查通过"
    fi
    
    if echo "$RPMLINT_OUTPUT" | grep -q "warning"; then
        check_warn "rpmlint 发现警告（可忽略）"
    fi
else
    check_warn "rpmlint 未安装，跳过语法检查"
fi

echo ""

# 总结
echo -e "${BLUE}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo -e "${BLUE}  检查总结${NC}"
echo -e "${BLUE}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo ""
echo -e "错误数: ${RED}$ERRORS${NC}"
echo -e "警告数: ${YELLOW}$WARNINGS${NC}"
echo ""

if [ $ERRORS -eq 0 ] && [ $WARNINGS -eq 0 ]; then
    echo -e "${GREEN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    echo -e "${GREEN}✓ 所有检查通过！可以上传到 OBS${NC}"
    echo -e "${GREEN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    echo ""
    echo -e "${YELLOW}下一步操作:${NC}"
    echo "  1. 访问 https://build.opensuse.org/"
    echo "  2. 登录账号"
    echo "  3. 创建/进入项目"
    echo "  4. 创建包 'lz-scan'"
    echo "  5. 上传 obs_package/ 中的文件"
    echo "  6. 等待构建完成"
    echo ""
    echo -e "${YELLOW}详细指南:${NC}"
    echo "  • OBS快速开始.md - 5分钟快速上手"
    echo "  • OBS多发行版编译指南.md - 完整指南"
    echo ""
elif [ $ERRORS -eq 0 ]; then
    echo -e "${YELLOW}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    echo -e "${YELLOW}⚠️  有 $WARNINGS 个警告，建议修复后再上传${NC}"
    echo -e "${YELLOW}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    echo ""
    echo "可以继续上传，但建议先修复警告"
    echo ""
else
    echo -e "${RED}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    echo -e "${RED}✗ 发现 $ERRORS 个错误，必须修复后才能上传${NC}"
    echo -e "${RED}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    echo ""
    echo "请修复上述错误后重新检查"
    echo ""
    exit 1
fi

echo ""
