#!/bin/bash

# 为 OBS (Open Build Service) 打包所有需要的文件
# 版本: 1.6.6
# 用途: 准备上传到 https://build.opensuse.org/ 的所有文件

set -e

VERSION="1.6.6"
PACKAGE_NAME="lz-scan"

# 颜色定义
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

echo -e "${BLUE}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo -e "${BLUE}  为 OBS 准备所有文件${NC}"
echo -e "${BLUE}  版本: ${VERSION}${NC}"
echo -e "${BLUE}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo ""

# 检查必要文件
echo -e "${YELLOW}检查必要文件...${NC}"
MISSING_FILES=0

check_file() {
    if [ ! -f "$1" ]; then
        echo -e "${RED}✗ 缺少文件: $1${NC}"
        MISSING_FILES=$((MISSING_FILES + 1))
    else
        echo -e "${GREEN}✓ $1${NC}"
    fi
}

check_file "CMakeLists.txt"
check_file "lz-scan.spec"
check_file "README.md"
check_file "CHANGELOG.md"
check_file "LICENSE"
check_file "lz-scan.svg"
check_file "debian/changelog"
check_file "debian/control"
check_file "debian/copyright"
check_file "debian/lz-scan.desktop"
check_file "debian/lz-scan.1"

if [ $MISSING_FILES -gt 0 ]; then
    echo -e "${RED}错误: 缺少 $MISSING_FILES 个必要文件${NC}"
    exit 1
fi

echo ""

# 1. 创建源码 tarball
echo -e "${YELLOW}1. 创建源码包...${NC}"
TEMP_DIR=$(mktemp -d)
SOURCE_DIR="${TEMP_DIR}/${PACKAGE_NAME}-${VERSION}"

mkdir -p "${SOURCE_DIR}"

echo "   复制文件到临时目录..."
rsync -av \
    --exclude='.git' \
    --exclude='build' \
    --exclude='*.deb' \
    --exclude='*.rpm' \
    --exclude='*.tar.gz' \
    --exclude='.vscode' \
    --exclude='.kiro' \
    --exclude='*.o' \
    --exclude='*.so' \
    --exclude='moc_*' \
    --exclude='ui_*' \
    --exclude='obs_package' \
    --exclude='obj-*' \
    --exclude='deepin-scanner-master' \
    --exclude='github-upload-temp' \
    ./ "${SOURCE_DIR}/" > /dev/null

echo "   创建 tar.gz 压缩包..."
cd "${TEMP_DIR}"
tar czf "${PACKAGE_NAME}-${VERSION}.tar.gz" "${PACKAGE_NAME}-${VERSION}"
mv "${PACKAGE_NAME}-${VERSION}.tar.gz" "${OLDPWD}/"
cd "${OLDPWD}"
rm -rf "${TEMP_DIR}"

SIZE=$(du -h "${PACKAGE_NAME}-${VERSION}.tar.gz" | cut -f1)
echo -e "${GREEN}   ✓ 已创建: ${PACKAGE_NAME}-${VERSION}.tar.gz (${SIZE})${NC}"

# 2. 打包 debian 目录
echo ""
echo -e "${YELLOW}2. 打包 debian 目录...${NC}"
tar czf debian.tar.gz debian/
SIZE=$(du -h debian.tar.gz | cut -f1)
echo -e "${GREEN}   ✓ 已创建: debian.tar.gz (${SIZE})${NC}"

# 3. 创建 .dsc 文件
echo ""
echo -e "${YELLOW}3. 创建 .dsc 文件...${NC}"

# 计算文件的 MD5 和大小
TAR_SIZE=$(stat -c%s "${PACKAGE_NAME}-${VERSION}.tar.gz" 2>/dev/null || stat -f%z "${PACKAGE_NAME}-${VERSION}.tar.gz")
TAR_MD5=$(md5sum "${PACKAGE_NAME}-${VERSION}.tar.gz" 2>/dev/null | cut -d' ' -f1 || md5 -q "${PACKAGE_NAME}-${VERSION}.tar.gz")
DEB_SIZE=$(stat -c%s "debian.tar.gz" 2>/dev/null || stat -f%z "debian.tar.gz")
DEB_MD5=$(md5sum "debian.tar.gz" 2>/dev/null | cut -d' ' -f1 || md5 -q "debian.tar.gz")

cat > "${PACKAGE_NAME}.dsc" << EOF
Format: 3.0 (quilt)
Source: ${PACKAGE_NAME}
Binary: ${PACKAGE_NAME}
Architecture: any
Version: ${VERSION}-1
Maintainer: 克亮 <315707022@qq.com>
Homepage: https://github.com/tonglingcn/lz-scan
Standards-Version: 4.6.0
Vcs-Browser: https://github.com/tonglingcn/lz-scan
Vcs-Git: https://github.com/tonglingcn/lz-scan.git
Build-Depends: debhelper-compat (= 13), cmake (>= 3.10), qtbase5-dev (>= 5.12.0), qttools5-dev, libsane-dev (>= 1.0.0), pkg-config
Package-List:
 ${PACKAGE_NAME} deb graphics optional arch=any
Checksums-Md5:
 ${TAR_MD5} ${TAR_SIZE} ${PACKAGE_NAME}-${VERSION}.tar.gz
 ${DEB_MD5} ${DEB_SIZE} debian.tar.gz
Files:
 ${TAR_MD5} ${TAR_SIZE} ${PACKAGE_NAME}-${VERSION}.tar.gz
 ${DEB_MD5} ${DEB_SIZE} debian.tar.gz
EOF

echo -e "${GREEN}   ✓ 已创建: ${PACKAGE_NAME}.dsc${NC}"

# 4. 验证 spec 文件
echo ""
echo -e "${YELLOW}4. 验证 spec 文件...${NC}"

if grep -q "Version:.*${VERSION}" lz-scan.spec; then
    echo -e "${GREEN}   ✓ spec 文件版本正确: ${VERSION}${NC}"
else
    echo -e "${RED}   ✗ spec 文件版本不匹配${NC}"
    echo -e "${YELLOW}   当前版本应该是: ${VERSION}${NC}"
    exit 1
fi

# 检查 spec 文件语法
if command -v rpmlint >/dev/null 2>&1; then
    echo "   运行 rpmlint 检查..."
    if rpmlint lz-scan.spec 2>&1 | grep -q "error"; then
        echo -e "${YELLOW}   ⚠️  发现一些警告，但可以继续${NC}"
    else
        echo -e "${GREEN}   ✓ spec 文件语法正确${NC}"
    fi
else
    echo -e "${YELLOW}   ⚠️  rpmlint 未安装，跳过语法检查${NC}"
fi

# 5. 创建 OBS 包目录
echo ""
echo -e "${YELLOW}5. 创建 OBS 包目录...${NC}"
OBS_DIR="obs_package"
rm -rf "${OBS_DIR}"
mkdir -p "${OBS_DIR}"

# 复制所有需要的文件
cp "${PACKAGE_NAME}-${VERSION}.tar.gz" "${OBS_DIR}/"
cp "debian.tar.gz" "${OBS_DIR}/"
cp "${PACKAGE_NAME}.dsc" "${OBS_DIR}/"
cp "${PACKAGE_NAME}.spec" "${OBS_DIR}/"

echo -e "${GREEN}   ✓ 已创建目录: ${OBS_DIR}/${NC}"

# 6. 创建 README
echo ""
echo -e "${YELLOW}6. 创建上传说明...${NC}"

cat > "${OBS_DIR}/README.txt" << 'EOF'
OBS 上传文件说明
================

这个目录包含了上传到 OBS (Open Build Service) 所需的所有文件。

文件列表:
---------
1. lz-scan-1.6.6.tar.gz  - 源码包（必需）
2. lz-scan.spec          - RPM spec 文件（必需）
3. debian.tar.gz         - Debian 打包文件（可选，用于 Debian/Ubuntu）
4. lz-scan.dsc           - Debian 源码控制文件（可选，用于 Debian/Ubuntu）

上传步骤:
---------

方法 1: Web 界面（推荐）
1. 访问 https://build.opensuse.org/
2. 登录账号
3. 创建或进入项目（home:你的用户名）
4. 创建包 "lz-scan"
5. 上传文件：
   - lz-scan-1.6.6.tar.gz（必需）
   - lz-scan.spec（必需）
   - debian.tar.gz（可选）
   - lz-scan.dsc（可选）
6. 等待构建完成

方法 2: osc 命令行
1. 安装 osc: sudo apt-get install osc
2. 配置: osc -A https://api.opensuse.org
3. 检出: osc co home:你的用户名
4. 创建包: osc mkpac lz-scan
5. 复制文件: cp * ~/home:你的用户名/lz-scan/
6. 添加: cd ~/home:你的用户名/lz-scan && osc add *
7. 提交: osc commit -m "Update to 1.6.6"

支持的发行版:
-------------
- openSUSE Tumbleweed
- openSUSE Leap 15.5, 15.6
- Fedora 39, 40
- Debian 11, 12
- Ubuntu 20.04, 22.04, 24.04

更多信息:
---------
- OBS 官网: https://build.opensuse.org/
- 详细指南: ../OBS多发行版编译指南.md
- 快速开始: ../OBS快速开始.md
EOF

echo -e "${GREEN}   ✓ 已创建: ${OBS_DIR}/README.txt${NC}"

# 7. 显示摘要
echo ""
echo -e "${BLUE}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo -e "${BLUE}  准备完成！${NC}"
echo -e "${BLUE}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo ""
echo -e "${GREEN}已创建的文件：${NC}"
echo ""

# 显示文件列表和大小
cd "${OBS_DIR}"
for file in *; do
    if [ -f "$file" ]; then
        SIZE=$(du -h "$file" | cut -f1)
        case "$file" in
            *.tar.gz)
                echo -e "  📦 ${file} ${YELLOW}(${SIZE})${NC}"
                ;;
            *.spec)
                echo -e "  📄 ${file} ${YELLOW}(${SIZE})${NC}"
                ;;
            *.dsc)
                echo -e "  📄 ${file} ${YELLOW}(${SIZE})${NC}"
                ;;
            *.txt)
                echo -e "  📝 ${file} ${YELLOW}(${SIZE})${NC}"
                ;;
            *)
                echo -e "  📄 ${file} ${YELLOW}(${SIZE})${NC}"
                ;;
        esac
    fi
done
cd ..

echo ""
echo -e "${GREEN}所有文件已复制到: ${OBS_DIR}/${NC}"
echo ""

# 8. 显示下一步操作
echo -e "${BLUE}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo -e "${BLUE}  下一步操作${NC}"
echo -e "${BLUE}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo ""
echo -e "${YELLOW}方法 1: 使用 Web 界面（推荐新手）${NC}"
echo "  1. 访问 https://build.opensuse.org/"
echo "  2. 登录并创建项目（如果还没有）"
echo "  3. 创建包 'lz-scan'"
echo "  4. 上传以下文件："
echo -e "     ${GREEN}• ${PACKAGE_NAME}-${VERSION}.tar.gz${NC}"
echo -e "     ${GREEN}• ${PACKAGE_NAME}.spec${NC}"
echo "  5. 等待构建完成"
echo ""
echo -e "${YELLOW}方法 2: 使用 osc 命令行（推荐高级用户）${NC}"
echo "  1. 安装 osc:"
echo "     sudo apt-get install osc"
echo ""
echo "  2. 配置 osc:"
echo "     osc -A https://api.opensuse.org"
echo ""
echo "  3. 上传文件:"
echo "     osc co home:你的用户名"
echo "     osc mkpac lz-scan"
echo "     cp ${OBS_DIR}/* home:你的用户名/lz-scan/"
echo "     cd home:你的用户名/lz-scan"
echo "     osc add *"
echo "     osc commit -m 'Update to ${VERSION}'"
echo ""
echo "  4. 查看构建状态:"
echo "     osc results"
echo ""
echo -e "${YELLOW}详细说明请查看:${NC}"
echo "  • OBS多发行版编译指南.md - 完整指南"
echo "  • OBS快速开始.md - 5分钟快速上手"
echo "  • OBS命令行参考.md - osc 命令参考"
echo "  • ${OBS_DIR}/README.txt - 上传说明"
echo ""

# 9. 验证检查
echo -e "${BLUE}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo -e "${BLUE}  验证检查${NC}"
echo -e "${BLUE}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo ""

# 检查源码包内容
echo -e "${YELLOW}检查源码包内容...${NC}"
if tar -tzf "${PACKAGE_NAME}-${VERSION}.tar.gz" | grep -q "CMakeLists.txt"; then
    echo -e "${GREEN}✓ 源码包包含 CMakeLists.txt${NC}"
else
    echo -e "${RED}✗ 源码包缺少 CMakeLists.txt${NC}"
fi

if tar -tzf "${PACKAGE_NAME}-${VERSION}.tar.gz" | grep -q "src/main.cpp"; then
    echo -e "${GREEN}✓ 源码包包含源代码${NC}"
else
    echo -e "${RED}✗ 源码包缺少源代码${NC}"
fi

if tar -tzf "${PACKAGE_NAME}-${VERSION}.tar.gz" | grep -q "debian/"; then
    echo -e "${GREEN}✓ 源码包包含 debian 目录${NC}"
else
    echo -e "${YELLOW}⚠️  源码包缺少 debian 目录${NC}"
fi

echo ""
echo -e "${GREEN}✓ 所有文件准备完成！${NC}"
echo ""
echo -e "${YELLOW}提示: 运行 ./check_obs_package.sh 进行完整检查${NC}"
echo ""
