#include "croppableimagewidget.h"
#include <QApplication>
#include <QDebug>
#include <QKeyEvent>
#include <QPainter>

CroppableImageWidget::CroppableImageWidget(QWidget *parent)
    : QLabel(parent), m_croppingEnabled(false), m_dragging(false),
      m_rubberBand(nullptr) {
  setAlignment(Qt::AlignCenter);
  setMinimumSize(1, 1);
  setScaledContents(false);
  setFocusPolicy(Qt::StrongFocus); // 允许接收键盘焦点

  // 创建橡皮筋选择框
  m_rubberBand = new QRubberBand(QRubberBand::Rectangle, this);
  m_rubberBand->hide();
}

CroppableImageWidget::~CroppableImageWidget() {
  if (m_rubberBand) {
    delete m_rubberBand;
  }
}

void CroppableImageWidget::setCroppingEnabled(bool enabled) {
  m_croppingEnabled = enabled;

  if (!enabled) {
    clearCropRect();
    setCursor(Qt::ArrowCursor);
  } else {
    setCursor(Qt::CrossCursor);
  }

  update();
}

bool CroppableImageWidget::isCroppingEnabled() const {
  return m_croppingEnabled;
}

void CroppableImageWidget::setImage(const QImage &image) {
  m_originalImage = image;

  if (!image.isNull()) {
    // 缩放图像以适应控件大小，保持宽高比
    QPixmap pixmap = QPixmap::fromImage(image);
    QSize labelSize = size();

    if (labelSize.width() > 0 && labelSize.height() > 0) {
      pixmap = pixmap.scaled(labelSize, Qt::KeepAspectRatio,
                             Qt::SmoothTransformation);
    }

    setPixmap(pixmap);
  } else {
    clear();
  }

  clearCropRect();
}

QImage CroppableImageWidget::getImage() const { return m_originalImage; }

QRect CroppableImageWidget::getCropRect() const { return m_cropRect; }

void CroppableImageWidget::clearCropRect() {
  m_cropRect = QRect();
  m_rubberBand->hide();
  update();
}

void CroppableImageWidget::mousePressEvent(QMouseEvent *event) {
  if (!m_croppingEnabled || m_originalImage.isNull()) {
    QLabel::mousePressEvent(event);
    return;
  }

  if (event->button() == Qt::LeftButton) {
    m_startPoint = event->pos();
    m_endPoint = m_startPoint;
    m_dragging = true;

    m_rubberBand->setGeometry(QRect(m_startPoint, QSize()));
    m_rubberBand->show();
  }

  QLabel::mousePressEvent(event);
}

void CroppableImageWidget::mouseMoveEvent(QMouseEvent *event) {
  if (!m_croppingEnabled || !m_dragging || m_originalImage.isNull()) {
    QLabel::mouseMoveEvent(event);
    return;
  }

  m_endPoint = event->pos();

  // 更新橡皮筋选择框
  QRect rect = normalizeRect(m_startPoint, m_endPoint);
  m_rubberBand->setGeometry(rect);

  // 计算对应的图像坐标
  QRect imageRect = mapToImageCoordinates(rect);
  emit cropRectChanged(imageRect);

  QLabel::mouseMoveEvent(event);
}

void CroppableImageWidget::mouseReleaseEvent(QMouseEvent *event) {
  if (!m_croppingEnabled || !m_dragging || m_originalImage.isNull()) {
    QLabel::mouseReleaseEvent(event);
    return;
  }

  if (event->button() == Qt::LeftButton) {
    m_dragging = false;
    m_endPoint = event->pos();

    // 计算最终的裁剪区域
    QRect widgetRect = normalizeRect(m_startPoint, m_endPoint);
    m_cropRect = mapToImageCoordinates(widgetRect);

    // 验证裁剪区域
    if (m_cropRect.width() > 10 && m_cropRect.height() > 10) {
      emit cropCompleted(m_cropRect);
    } else {
      // 区域太小，清除选择
      clearCropRect();
    }
  }

  QLabel::mouseReleaseEvent(event);
}

void CroppableImageWidget::paintEvent(QPaintEvent *event) {
  QLabel::paintEvent(event);

  if (!m_croppingEnabled || m_cropRect.isNull() || m_originalImage.isNull()) {
    return;
  }

  // 绘制裁剪区域的边框
  QPainter painter(this);
  painter.setPen(QPen(Qt::red, 2, Qt::DashLine));

  QRect widgetRect = mapToWidgetCoordinates(m_cropRect);
  painter.drawRect(widgetRect);

  // 绘制半透明遮罩
  painter.fillRect(rect(), QColor(0, 0, 0, 100));
  painter.setCompositionMode(QPainter::CompositionMode_Clear);
  painter.fillRect(widgetRect, Qt::transparent);
}

QRect CroppableImageWidget::normalizeRect(const QPoint &p1,
                                          const QPoint &p2) const {
  return QRect(qMin(p1.x(), p2.x()), qMin(p1.y(), p2.y()),
               qAbs(p2.x() - p1.x()), qAbs(p2.y() - p1.y()));
}

QRect CroppableImageWidget::mapToImageCoordinates(
    const QRect &widgetRect) const {
  QPixmap currentPixmap = pixmap(Qt::ReturnByValue);
  if (m_originalImage.isNull() || currentPixmap.isNull()) {
    return QRect();
  }

  // 获取显示的像素图大小和位置
  QSize pixmapSize = currentPixmap.size();
  QSize labelSize = size();

  // 计算像素图在标签中的位置（居中显示）
  int pixmapX = (labelSize.width() - pixmapSize.width()) / 2;
  int pixmapY = (labelSize.height() - pixmapSize.height()) / 2;

  // 将控件坐标转换为像素图坐标
  QRect pixmapRect = widgetRect.translated(-pixmapX, -pixmapY);

  // 确保在像素图范围内
  pixmapRect = pixmapRect.intersected(
      QRect(0, 0, pixmapSize.width(), pixmapSize.height()));

  // 计算缩放比例
  double scaleX = (double)m_originalImage.width() / pixmapSize.width();
  double scaleY = (double)m_originalImage.height() / pixmapSize.height();

  // 转换为原始图像坐标
  QRect imageRect(
      (int)(pixmapRect.x() * scaleX), (int)(pixmapRect.y() * scaleY),
      (int)(pixmapRect.width() * scaleX), (int)(pixmapRect.height() * scaleY));

  // 确保在原始图像范围内
  return imageRect.intersected(m_originalImage.rect());
}

QRect CroppableImageWidget::mapToWidgetCoordinates(
    const QRect &imageRect) const {
  QPixmap currentPixmap = pixmap(Qt::ReturnByValue);
  if (m_originalImage.isNull() || currentPixmap.isNull()) {
    return QRect();
  }

  // 获取显示的像素图大小和位置
  QSize pixmapSize = currentPixmap.size();
  QSize labelSize = size();

  // 计算缩放比例
  double scaleX = (double)pixmapSize.width() / m_originalImage.width();
  double scaleY = (double)pixmapSize.height() / m_originalImage.height();

  // 转换为像素图坐标
  QRect pixmapRect((int)(imageRect.x() * scaleX), (int)(imageRect.y() * scaleY),
                   (int)(imageRect.width() * scaleX),
                   (int)(imageRect.height() * scaleY));

  // 计算像素图在标签中的位置（居中显示）
  int pixmapX = (labelSize.width() - pixmapSize.width()) / 2;
  int pixmapY = (labelSize.height() - pixmapSize.height()) / 2;

  // 转换为控件坐标
  return pixmapRect.translated(pixmapX, pixmapY);
}

void CroppableImageWidget::keyPressEvent(QKeyEvent *event) {
  if (m_croppingEnabled && event->key() == Qt::Key_Escape) {
    // ESC键取消裁剪
    clearCropRect();
    setCroppingEnabled(false);
    emit cropCancelled();
    event->accept();
    return;
  }

  QLabel::keyPressEvent(event);
}