#include "documentmanager.h"

#include <QDebug>
#include <QFile>
#include <QFileInfo>
#include <QDir>
#include <QPainter>
#include <QImageReader>
#include <QImageWriter>
#include <QPdfWriter>
#include <QPageSize>
#include <QPageLayout>
#include <QBuffer>

DocumentManager::DocumentManager(QObject *parent)
    : QObject(parent)
    , m_currentPageIndex(-1)
    , m_hasUnsavedChanges(false)
{
    // 初始化支持的图像格式
    m_supportedImageFormats << "png" << "jpg" << "jpeg" << "bmp" << "tiff" << "tif" << "pdf";
    
    // 创建新文档
    newDocument();
}

DocumentManager::~DocumentManager()
{
}

void DocumentManager::newDocument()
{
    // 清除当前文档
    m_pages.clear();
    m_currentPageIndex = -1;
    m_filePath.clear();
    
    // 使用年月日+四位数递增格式作为默认文档标题
    QDateTime currentDateTime = QDateTime::currentDateTime();
    QString dateStr = currentDateTime.toString("yyyyMMdd");
    
    // 生成四位数递增序号 (从0001开始)
    static int counter = 1;
    QString counterStr = QString("%1").arg(counter, 4, 10, QChar('0'));
    counter = (counter % 9999) + 1;  // 循环使用1-9999
    
    m_documentTitle = dateStr + counterStr;
    m_hasUnsavedChanges = false;
    
    emit documentChanged();
    emit currentPageChanged(m_currentPageIndex);
    
    qDebug() << "已创建新文档";
}

bool DocumentManager::loadDocument(const QString &filePath)
{
    QFileInfo fileInfo(filePath);
    
    if (!fileInfo.exists()) {
        emit documentError(tr("文件不存在: %1").arg(filePath));
        return false;
    }
    
    QString extension = fileInfo.suffix().toLower();
    
    if (!m_supportedImageFormats.contains(extension)) {
        emit documentError(tr("不支持的文件格式: %1").arg(extension));
        return false;
    }
    
    // 清除当前文档
    newDocument();
    
    // 加载文件
    if (extension == "pdf") {
        // PDF文件需要特殊处理，这里简化处理
        emit documentError(tr("PDF导入功能尚未实现"));
        return false;
    } else {
        // 加载图像文件
        QImageReader reader(filePath);
        
        if (!reader.canRead()) {
            emit documentError(tr("无法读取图像文件: %1").arg(filePath));
            return false;
        }
        
        QImage image = reader.read();
        
        if (image.isNull()) {
            emit documentError(tr("读取图像失败: %1").arg(reader.errorString()));
            return false;
        }
        
        // 添加图像到文档
        addImage(image);
        
        // 更新文档信息
        m_filePath = filePath;
        m_documentTitle = fileInfo.baseName();
        m_hasUnsavedChanges = false;
        
        emit documentLoaded(filePath);
        qDebug() << "已加载文档: " << filePath;
        
        return true;
    }
}

bool DocumentManager::saveDocument(const QString &filePath)
{
    if (m_pages.isEmpty()) {
        emit documentError(tr("没有要保存的页面"));
        return false;
    }
    
    QFileInfo fileInfo(filePath);
    QString extension = fileInfo.suffix().toLower();
    
    if (!m_supportedImageFormats.contains(extension)) {
        emit documentError(tr("不支持的文件格式: %1").arg(extension));
        return false;
    }
    
    bool success = false;
    
    // 多页文档处理
    if (m_pages.size() > 1) {
        if (extension == "pdf") {
            // PDF天然支持多页
            success = saveAsPDF(filePath);
            if (success) {
                qDebug() << "成功将" << m_pages.size() << "页保存为单个PDF文件";
            }
        } else if (extension == "tiff" || extension == "tif") {
            // TIFF支持多页
            success = saveAsMultiPageTIFF(filePath);
            if (success) {
                qDebug() << "成功将" << m_pages.size() << "页保存为单个TIFF文件";
            }
        } else {
            // 其他格式保存为多个文件
            success = saveAsImage(filePath);
            if (success) {
                qDebug() << "成功将" << m_pages.size() << "页保存为多个图像文件";
            }
        }
    } else {
        // 单页文档处理
        if (extension == "pdf") {
            success = saveAsPDF(filePath);
        } else if (extension == "tiff" || extension == "tif") {
            success = saveAsMultiPageTIFF(filePath);
        } else {
            success = saveAsImage(filePath);
        }
    }
    
    if (success) {
        m_filePath = filePath;
        m_documentTitle = fileInfo.baseName();
        m_hasUnsavedChanges = false;
        
        emit documentSaved(filePath);
        qDebug() << "文档已保存到: " << filePath;
    }
    
    return success;
}

bool DocumentManager::exportDocument(const QString &filePath, const QString &format)
{
    QString actualFilePath = filePath;
    
    // 如果没有指定扩展名，则添加扩展名
    if (!actualFilePath.endsWith("." + format, Qt::CaseInsensitive)) {
        actualFilePath += "." + format;
    }
    
    return saveDocument(actualFilePath);
}

void DocumentManager::addImage(const QImage &image)
{
    if (image.isNull()) {
        qDebug() << "尝试添加空图像";
        return;
    }
    
    m_pages.append(image);
    int newIndex = m_pages.size() - 1;
    
    // 如果这是第一个页面，则设置为当前页面
    if (m_currentPageIndex == -1) {
        m_currentPageIndex = 0;
        emit currentPageChanged(m_currentPageIndex);
    }
    
    m_hasUnsavedChanges = true;
    
    emit pageAdded(newIndex);
    emit documentChanged();
    
    qDebug() << "已添加图像，当前页面数: " << m_pages.size();
}

void DocumentManager::insertImage(int index, const QImage &image)
{
    if (image.isNull()) {
        qDebug() << "尝试插入空图像";
        return;
    }
    
    // 确保索引有效
    index = qBound(0, index, m_pages.size());
    
    m_pages.insert(index, image);
    
    // 如果这是第一个页面，则设置为当前页面
    if (m_currentPageIndex == -1) {
        m_currentPageIndex = 0;
        emit currentPageChanged(m_currentPageIndex);
    }
    // 如果在当前页面之前插入，则更新当前页面索引
    else if (index <= m_currentPageIndex) {
        m_currentPageIndex++;
        emit currentPageChanged(m_currentPageIndex);
    }
    
    m_hasUnsavedChanges = true;
    
    emit pageAdded(index);
    emit documentChanged();
    
    qDebug() << "已在位置 " << index << " 插入图像，当前页面数: " << m_pages.size();
}

void DocumentManager::removePage(int index)
{
    if (index < 0 || index >= m_pages.size()) {
        qDebug() << "尝试删除无效页面: " << index;
        return;
    }
    
    m_pages.removeAt(index);
    
    // 更新当前页面索引
    if (m_pages.isEmpty()) {
        m_currentPageIndex = -1;
    } else if (index <= m_currentPageIndex) {
        if (m_currentPageIndex > 0) {
            m_currentPageIndex--;
        }
    }
    
    m_hasUnsavedChanges = true;
    
    emit pageRemoved(index);
    emit currentPageChanged(m_currentPageIndex);
    emit documentChanged();
    
    qDebug() << "已删除页面 " << index << "，当前页面数: " << m_pages.size();
}

void DocumentManager::movePageUp(int index)
{
    if (index <= 0 || index >= m_pages.size()) {
        qDebug() << "无法上移页面: " << index;
        return;
    }
    
    m_pages.swapItemsAt(index, index - 1);
    
    // 更新当前页面索引
    if (index == m_currentPageIndex) {
        m_currentPageIndex--;
        emit currentPageChanged(m_currentPageIndex);
    } else if (index - 1 == m_currentPageIndex) {
        m_currentPageIndex++;
        emit currentPageChanged(m_currentPageIndex);
    }
    
    m_hasUnsavedChanges = true;
    
    emit pageMoved(index, index - 1);
    emit documentChanged();
    
    qDebug() << "页面 " << index << " 已上移";
}

void DocumentManager::movePageDown(int index)
{
    if (index < 0 || index >= m_pages.size() - 1) {
        qDebug() << "无法下移页面: " << index;
        return;
    }
    
    m_pages.swapItemsAt(index, index + 1);
    
    // 更新当前页面索引
    if (index == m_currentPageIndex) {
        m_currentPageIndex++;
        emit currentPageChanged(m_currentPageIndex);
    } else if (index + 1 == m_currentPageIndex) {
        m_currentPageIndex--;
        emit currentPageChanged(m_currentPageIndex);
    }
    
    m_hasUnsavedChanges = true;
    
    emit pageMoved(index, index + 1);
    emit documentChanged();
    
    qDebug() << "页面 " << index << " 已下移";
}

void DocumentManager::setCurrentPage(int index)
{
    if (index < 0 || index >= m_pages.size()) {
        qDebug() << "尝试设置无效的当前页面: " << index;
        return;
    }
    
    if (m_currentPageIndex != index) {
        m_currentPageIndex = index;
        emit currentPageChanged(m_currentPageIndex);
        qDebug() << "当前页面已设置为: " << index;
    }
}

QImage DocumentManager::getPage(int index) const
{
    if (index < 0 || index >= m_pages.size()) {
        qDebug() << "尝试获取无效页面: " << index;
        return QImage();
    }
    
    return m_pages.at(index);
}

QImage DocumentManager::getCurrentImage() const
{
    if (m_currentPageIndex < 0 || m_currentPageIndex >= m_pages.size()) {
        return QImage();
    }
    
    return m_pages.at(m_currentPageIndex);
}

int DocumentManager::getCurrentPageIndex() const
{
    return m_currentPageIndex;
}

int DocumentManager::getPageCount() const
{
    return m_pages.size();
}

bool DocumentManager::hasUnsavedChanges() const
{
    return m_hasUnsavedChanges;
}

QString DocumentManager::getCurrentFilePath() const
{
    return m_filePath;
}

QString DocumentManager::getDocumentTitle() const
{
    return m_documentTitle;
}

bool DocumentManager::saveAsPDF(const QString &filePath)
{
    if (m_pages.isEmpty()) {
        emit documentError(tr("没有要保存的页面"));
        return false;
    }
    
    QPdfWriter pdfWriter(filePath);
    pdfWriter.setCreator(tr("Simple-Scanx"));
    pdfWriter.setTitle(m_documentTitle);
    
    // 使用第一页的尺寸作为PDF页面尺寸
    QImage firstPage = m_pages.first();
    QPageSize pageSize(QSizeF(firstPage.width(), firstPage.height()), QPageSize::Point);
    pdfWriter.setPageSize(pageSize);
    
    // 设置分辨率
    pdfWriter.setResolution(300); // 300 DPI
    
    QPainter painter(&pdfWriter);
    
    for (int i = 0; i < m_pages.size(); ++i) {
        QImage image = m_pages.at(i);
        
        // 如果不是第一页，则添加新页面
        if (i > 0) {
            pdfWriter.newPage();
        }
        
        // 绘制图像
        painter.drawImage(QRect(0, 0, pdfWriter.width(), pdfWriter.height()), image);
    }
    
    painter.end();
    
    return true;
}

bool DocumentManager::saveAsImage(const QString &filePath)
{
    if (m_pages.isEmpty()) {
        emit documentError(tr("没有要保存的页面"));
        return false;
    }
    
    QFileInfo fileInfo(filePath);
    QString basePath = fileInfo.absolutePath();
    QString baseName = fileInfo.completeBaseName();
    QString suffix = fileInfo.suffix();
    
    // 如果只有一页，直接保存
    if (m_pages.size() == 1) {
        QImageWriter writer(filePath);
        
        // 设置图像质量
        if (suffix.toLower() == "jpg" || suffix.toLower() == "jpeg") {
            writer.setQuality(90);
        }
        
        if (!writer.write(m_pages.first())) {
            emit documentError(tr("保存图像失败: %1").arg(writer.errorString()));
            return false;
        }
        
        return true;
    }
    
    // 多页文档，保存为多个文件
    bool allSaved = true;
    int savedCount = 0;
    
    // 使用年月日作为基本名称
    QDateTime currentDateTime = QDateTime::currentDateTime();
    QString dateStr = currentDateTime.toString("yyyyMMdd");
    
    // 如果用户提供的文件名不是默认格式，则尊重用户的选择
    if (!baseName.startsWith(dateStr)) {
        baseName = dateStr;
    }
    
    for (int i = 0; i < m_pages.size(); ++i) {
        // 构建文件名: 年月日+四位数.jpg，例如：20230501_0001.jpg
        QString pageFilePath = QString("%1/%2_%3.%4")
                              .arg(basePath)
                              .arg(baseName)
                              .arg(i + 1, 4, 10, QChar('0'))  // 0001, 0002, ...
                              .arg(suffix);
        
        QImageWriter writer(pageFilePath);
        
        // 设置图像质量
        if (suffix.toLower() == "jpg" || suffix.toLower() == "jpeg") {
            writer.setQuality(90);
        }
        
        if (!writer.write(m_pages.at(i))) {
            emit documentError(tr("保存第%1页图像失败: %2").arg(i + 1).arg(writer.errorString()));
            allSaved = false;
            continue;
        }
        
        savedCount++;
    }
    
    if (savedCount > 0) {
        qDebug() << "成功保存" << savedCount << "个图像文件到" << basePath;
        return true;
    }
    
    return allSaved;
}

bool DocumentManager::saveAsMultiPageTIFF(const QString &filePath)
{
    if (m_pages.isEmpty()) {
        emit documentError(tr("没有要保存的页面"));
        return false;
    }
    
    // 使用QImageWriter保存多页TIFF
    QImageWriter writer(filePath, "tiff");
    
    // 设置TIFF压缩选项
    writer.setCompression(1); // LZW压缩
    
    // 保存第一页
    if (!writer.write(m_pages.first())) {
        emit documentError(tr("保存TIFF文件失败: %1").arg(writer.errorString()));
        return false;
    }
    
    // 保存剩余页面
    for (int i = 1; i < m_pages.size(); ++i) {
        if (!writer.write(m_pages.at(i))) {
            emit documentError(tr("保存TIFF文件第%1页失败: %2").arg(i+1).arg(writer.errorString()));
            return false;
        }
    }
    
    qDebug() << "成功保存" << m_pages.size() << "页到多页TIFF文件:" << filePath;
    return true;
}