#include "mainwindow.h"
#include "croppableimagewidget.h"

#include <QApplication>
#include <QCloseEvent>
#include <QDebug>
#include <QDialog>
#include <QDialogButtonBox>
#include <QFileDialog>
#include <QFormLayout>
#include <QGroupBox>
#include <QHBoxLayout>
#include <QImageReader>
#include <QImageWriter>
#include <QInputDialog>
#include <QLabel>
#include <QMenu>
#include <QMenuBar>
#include <QMessageBox>
#include <QPushButton>
#include <QRadioButton>
#include <QSettings>
#include <QStandardPaths>
#include <QStyle>
#include <QTimer>
#include <QToolButton>
#include <QVBoxLayout>
#include <QtConcurrent/QtConcurrentRun>
#include <QFutureWatcher>

MainWindow::MainWindow(QWidget *parent)
    : QMainWindow(parent), m_isScanning(false), m_isCropping(false),
      m_saveCounter(0), m_isMultiPageScanning(false), m_delayScanRemaining(0),
      m_delayScanInterval(0), m_isDelayScanMode(false), m_delayScanCount(0),
      m_finishDelayScanBtn(nullptr), m_removeBlankPagesEnabled(false),
      m_blankPageThreshold(240), m_blankPageRatio(98), m_searchNetworkDevices(false) {
  // 创建核心组件（ScannerController延迟初始化）
  m_scannerController = nullptr;
  m_imageProcessor = new ImageProcessor(this);
  m_documentManager = new DocumentManager(this);
  
  // 初始化延时扫描定时器
  m_delayScanTimer = new QTimer(this);
  m_delayScanTimer->setInterval(1000); // 每秒更新一次
  connect(m_delayScanTimer, &QTimer::timeout, this, [this]() {
    m_delayScanRemaining--;
    if (m_delayScanRemaining > 0) {
      m_statusBar->showMessage(tr("延时扫描倒计时: %1 秒... (已扫描 %2 张)").arg(m_delayScanRemaining).arg(m_delayScanCount));
    } else {
      m_delayScanTimer->stop();
      m_statusBar->showMessage(tr("开始扫描第 %1 张...").arg(m_delayScanCount + 1));
      // 执行扫描
      onScanButtonClicked();
    }
  });

  // 初始化保存相关控件
  m_formatCombo = new QComboBox();
  m_formatCombo->addItem("PDF", "pdf");
  m_formatCombo->addItem("PNG", "png");
  m_formatCombo->addItem("JPEG", "jpg");
  m_formatCombo->addItem("BMP", "bmp");
  m_formatCombo->addItem("TIFF", "tiff");

  m_pathEdit = new QLineEdit();
  m_pathEdit->setReadOnly(true);

  m_autoSaveCheck = new QCheckBox(tr("自动保存"));
  m_autoSaveCheck->setChecked(true);

  // 初始化UI
  createActions();
  createMenus();
  createToolbars();
  createStatusBar();
  createCentralWidget();
  createDockWidgets();
  connectSignals();

  // 加载设置
  loadSettings();

  // 更新窗口标题
  updateWindowTitle();

  // 更新操作状态
  updateActions();

  // 设置窗口大小
  resize(1024, 768);

  // 确保窗口完全显示后再启用查找按钮
  QTimer::singleShot(0, this, [this]() {
    if (m_findDevicesBtn) {
      m_findDevicesBtn->setEnabled(true);
    }
    // 扫描仪列表将在用户首次使用时初始化
    m_scannerComboBox->addItem(tr("点击'查找设备'初始化扫描仪"));
  });

  qDebug() << "主窗口已初始化";
}

MainWindow::~MainWindow() {
  // 确保扫描操作已停止
  if (m_scannerController && m_isScanning) {
    m_scannerController->cancelScan();
    // 等待扫描完成
    QApplication::processEvents();
    QThread::msleep(100);
  }

  // 先保存设置
  saveSettings();

  // 清理扫描仪控制器
  if (m_scannerController) {
    delete m_scannerController;
    m_scannerController = nullptr;
  }

  qDebug() << "主窗口已销毁";
}

void MainWindow::closeEvent(QCloseEvent *event) {
  if (maybeSave()) {
    event->accept();
  } else {
    event->ignore();
  }
}

void MainWindow::onScanButtonClicked() {
  // 确保扫描仪控制器已初始化
  if (!ensureScannerControllerInitialized()) {
    return;
  }

  if (m_scannerController->getCurrentScanner().isEmpty()) {
    QMessageBox::warning(this, tr("扫描错误"), tr("未选择扫描仪"));
    return;
  }

  if (m_isScanning) {
    m_scannerController->cancelScan();
    return;
  }

  // 获取扫描参数
  int resolution = m_resolutionComboBox->currentText().toInt();
  QString colorMode = m_colorModeComboBox->currentText();
  QString paperSource = m_paperSourceComboBox->currentText();
  // 双面扫描模式现在通过纸张来源自动控制，提供默认值
  QString duplexMode = "Simplex"; // 默认单面，双面自动进纸器会自动覆盖为Duplex

  // 开始扫描
  m_isScanning = true;
  updateActions();
  m_statusBar->showMessage(tr("正在扫描..."));
  m_progressBar->setValue(0);
  m_progressBar->show();

  // 开始扫描
  m_scannerController->startScan(QString(), colorMode, resolution, paperSource, duplexMode);
}

void MainWindow::onPreviewButtonClicked() {
  // 确保扫描仪控制器已初始化
  if (!ensureScannerControllerInitialized()) {
    return;
  }

  // 开始预览扫描
  m_isScanning = true;
  updateActions();
  m_statusBar->showMessage(tr("正在预览..."));
  m_progressBar->setValue(0);
  m_progressBar->show();

  // 获取扫描参数
  QString colorMode = m_colorModeComboBox->currentText();
  QString paperSource = m_paperSourceComboBox->currentText();

  // 开始预览
  m_scannerController->startPreview(QString());
}

void MainWindow::onDelayScanButtonClicked() {
  // 确保扫描仪控制器已初始化
  if (!ensureScannerControllerInitialized()) {
    return;
  }

  if (m_scannerController->getCurrentScanner().isEmpty()) {
    QMessageBox::warning(this, tr("扫描错误"), tr("未选择扫描仪"));
    return;
  }

  // 如果已经在延时扫描模式，则取消
  if (m_isDelayScanMode) {
    m_delayScanTimer->stop();
    m_isDelayScanMode = false;
    m_delayScanCount = 0;
    if (m_finishDelayScanBtn) {
      m_finishDelayScanBtn->hide();
    }
    m_statusBar->showMessage(tr("已取消延时扫描模式"), 3000);
    return;
  }

  // 弹出对话框让用户选择延时时间
  QDialog dialog(this);
  dialog.setWindowTitle(tr("循环延时扫描设置"));
  dialog.setMinimumWidth(350);
  
  QVBoxLayout *layout = new QVBoxLayout(&dialog);
  
  QLabel *label = new QLabel(tr("请选择每张扫描的延时间隔（秒）："), &dialog);
  layout->addWidget(label);
  
  QComboBox *delayCombo = new QComboBox(&dialog);
  delayCombo->addItem(tr("3 秒"), 3);
  delayCombo->addItem(tr("6 秒"), 6);
  delayCombo->addItem(tr("9 秒"), 9);
  delayCombo->addItem(tr("15 秒"), 15);
  delayCombo->addItem(tr("20 秒"), 20);
  delayCombo->setCurrentIndex(2); // 默认选择9秒
  layout->addWidget(delayCombo);
  
  QLabel *tipLabel = new QLabel(tr("\n<b>循环延时扫描模式说明：</b>\n"
                                   "• 点击确定后，系统将按设定的时间间隔自动循环扫描\n"
                                   "• 每次倒计时结束后自动扫描一张\n"
                                   "• 扫描完成后自动开始下一轮倒计时\n"
                                   "• 请在倒计时期间放好下一张原稿\n"
                                   "• 所有原稿扫描完毕后，点击\"扫描完成\"按钮结束\n"
                                   "• 点击\"扫描完成\"后才会进入保存流程"), &dialog);
  tipLabel->setWordWrap(true);
  tipLabel->setStyleSheet("color: #444; font-size: 11px; padding: 10px; background-color: #f0f0f0; border-radius: 5px;");
  layout->addWidget(tipLabel);
  
  QDialogButtonBox *buttonBox = new QDialogButtonBox(
      QDialogButtonBox::Ok | QDialogButtonBox::Cancel, &dialog);
  connect(buttonBox, &QDialogButtonBox::accepted, &dialog, &QDialog::accept);
  connect(buttonBox, &QDialogButtonBox::rejected, &dialog, &QDialog::reject);
  layout->addWidget(buttonBox);
  
  if (dialog.exec() == QDialog::Accepted) {
    // 进入延时扫描模式
    m_isDelayScanMode = true;
    m_delayScanInterval = delayCombo->currentData().toInt();
    m_delayScanCount = 0;
    m_delayScanRemaining = m_delayScanInterval;
    
    // 显示"扫描完成"按钮
    if (m_finishDelayScanBtn) {
      m_finishDelayScanBtn->show();
    }
    
    // 禁用自动保存（延时扫描模式下手动控制保存）
    bool originalAutoSave = m_autoSaveCheck->isChecked();
    m_autoSaveCheck->setChecked(false);
    m_autoSaveCheck->setEnabled(false);
    
    // 开始倒计时
    m_delayScanTimer->start();
    m_statusBar->showMessage(tr("延时扫描模式已启动，倒计时: %1 秒...").arg(m_delayScanRemaining));
  }
}

void MainWindow::onFinishDelayScanClicked() {
  if (!m_isDelayScanMode) {
    return;
  }
  
  // 停止延时扫描模式
  m_delayScanTimer->stop();
  m_isDelayScanMode = false;
  
  // 隐藏"扫描完成"按钮
  if (m_finishDelayScanBtn) {
    m_finishDelayScanBtn->hide();
  }
  
  // 恢复自动保存选项
  m_autoSaveCheck->setEnabled(true);
  
  // 显示扫描统计
  QString message = tr("延时扫描完成！共扫描 %1 张图片。").arg(m_delayScanCount);
  m_statusBar->showMessage(message, 5000);
  
  QMessageBox msgBox(this);
  msgBox.setWindowTitle(tr("延时扫描完成"));
  msgBox.setText(message);
  msgBox.setInformativeText(tr("是否现在保存扫描的图片？"));
  msgBox.setStandardButtons(QMessageBox::Yes | QMessageBox::No);
  msgBox.setDefaultButton(QMessageBox::Yes);
  
  if (msgBox.exec() == QMessageBox::Yes) {
    // 进入保存流程
    if (m_delayScanCount > 1) {
      onSaveMultiPageDocument();
    } else if (m_delayScanCount == 1) {
      onSaveCurrentImage();
    }
  }
  
  // 重置计数器
  m_delayScanCount = 0;
}

void MainWindow::onScannerSelected(int index) {
  // 忽略无效索引和"正在查找扫描仪..."等提示项
  if (index < 0 || m_scannerController == nullptr) {
    return;
  }
  
  QString displayName = m_scannerComboBox->itemText(index);
  
  // 忽略提示性文本
  if (displayName.contains("正在查找") || displayName.contains("点击") || displayName.contains("初始化")) {
    return;
  }
  
  qDebug() << "用户选择扫描仪:" << displayName << "索引:" << index;
  
  // 取消当前扫描
  if (m_isScanning) {
    m_scannerController->cancelScan();
  }
  
  // 确保控制器已初始化
  if (!m_scannerController->isInitialized()) {
    m_statusBar->showMessage(tr("扫描仪控制器未初始化，请先点击查找设备"), 3000);
    return;
  }
  
  // 显示选择进度
  m_statusBar->showMessage(tr("正在连接扫描仪: %1...").arg(displayName), 1000);
  
  // 选择扫描仪
  if (m_scannerController->selectScanner(displayName)) {
    qDebug() << "成功选择扫描仪:" << displayName;
    
    // 更新纸张来源选项（根据设备能力自动适应）
    updatePaperSourceOptions();
    
    // 构建状态信息
    QString statusMessage = tr("✅ 已选择: %1").arg(displayName);
    
    // 添加设备能力信息
    QStringList capabilities;
    if (m_scannerController->supportsADF()) {
      capabilities << tr("支持ADF");
    } else {
      capabilities << tr("仅平板");
    }
    
    if (m_scannerController->supportsDuplex()) {
      capabilities << tr("支持双面");
    }
    
    if (!capabilities.isEmpty()) {
      statusMessage += tr(" (%1)").arg(capabilities.join(", "));
    }
    
    // 在状态栏显示设备信息
    m_statusBar->showMessage(statusMessage, 8000);
    
    // 强制刷新设备参数以适应新设备
    m_scannerController->updateScannerParameters();
    
    // 调试输出设备选项（可能导致某些设备崩溃，暂时禁用）
    // m_scannerController->debugScannerOptions();
    
    // 更新UI状态
    updateActions();
    
    qDebug() << "设备切换完成，ADF支持:" << m_scannerController->supportsADF() 
             << "双面支持:" << m_scannerController->supportsDuplex();
  } else {
    qDebug() << "选择扫描仪失败:" << displayName;
    
    // 显示详细的错误信息
    QString errorMsg = tr("❌ 无法连接扫描仪: %1").arg(displayName);
    m_statusBar->showMessage(errorMsg, 5000);
    
    // 弹出详细的错误对话框
    QMessageBox::warning(this, tr("扫描仪连接失败"), 
                        tr("无法连接到扫描仪 '%1'。\n\n可能的原因：\n"
                           "• 设备正在被其他程序使用\n"
                           "• 设备驱动未正确安装\n"
                           "• USB连接不稳定\n"
                           "• 网络设备连接超时\n\n"
                           "建议：\n"
                           "• 检查设备连接\n"
                           "• 重启设备\n"
                           "• 重新点击'查找设备'").arg(displayName));
    
    // 尝试重新选择第一个可用设备
    if (m_scannerComboBox->count() > 1) {
      for (int i = 0; i < m_scannerComboBox->count(); ++i) {
        if (i != index) { // 跳过当前失败的设备
          QString otherDevice = m_scannerComboBox->itemText(i);
          if (!otherDevice.contains("正在查找") && !otherDevice.contains("点击") && !otherDevice.contains("初始化")) {
            qDebug() << "尝试自动选择其他设备:" << otherDevice;
            m_scannerComboBox->setCurrentIndex(i);
            // 这会触发新的onScannerSelected调用
            break;
          }
        }
      }
    }
  }
}

void MainWindow::onScanProgress(int percentage) {
  m_progressBar->setValue(percentage);
}

QString MainWindow::generateFileName(const QString &ext) {
  QDateTime now = QDateTime::currentDateTime();
  // 使用年月日时分秒格式：yyyyMMdd_HHmmss
  QString dateTimeStr = now.toString("yyyyMMdd_HHmmss");

  return QString("%1/%2.%3")
      .arg(m_pathEdit->text())
      .arg(dateTimeStr)
      .arg(ext);
}

bool MainWindow::saveImage(const QImage &image, const QString &path) {
  QString format = m_formatCombo->currentData().toString().toUpper();
  if (format == "PDF") {
    QPdfWriter writer(path);
    writer.setPageSize(QPageSize(QPageSize::A4));
    writer.setTitle(tr("扫描文档"));
    QPainter painter(&writer);
    painter.drawImage(QRect(0, 0, writer.width(), writer.height()), image);
    painter.end();
  } else {
    int quality = -1; // 默认质量
    if (format == "JPG" || format == "JPEG") {
      quality = m_qualitySlider->value();
    }

    if (!image.save(path, qPrintable(format), quality)) {
      QMessageBox::warning(this, tr("保存失败"),
                           tr("无法保存图像到: %1").arg(path));
      return false;
    }
  }

  m_saveCounter++;
  return true;
}

void MainWindow::onScanComplete(const QImage &image) {
  if (!image.isNull()) {
    QImage processedImage = image;

    // 自动倾斜校正功能已移除，用户可在高级设置中手动调整角度
    
    // 检测是否为空白页
    bool isBlank = false;
    if (m_removeBlankPagesEnabled) {
      isBlank = m_imageProcessor->isBlankPage(processedImage, m_blankPageThreshold, m_blankPageRatio / 100.0);
      if (isBlank) {
        qDebug() << "检测到空白页，已自动跳过";
        m_statusBar->showMessage(tr("检测到空白页，已自动跳过"), 2000);
        
        // 如果是延时扫描模式，继续下一轮
        if (m_isDelayScanMode) {
          m_isScanning = false;
          m_progressBar->hide();
          
          // 开始下一轮倒计时
          m_delayScanRemaining = m_delayScanInterval;
          m_delayScanTimer->start();
          m_statusBar->showMessage(tr("检测到空白页已跳过，准备扫描下一张，倒计时: %1 秒... (已扫描 %2 张)").arg(m_delayScanRemaining).arg(m_delayScanCount));
        } else {
          m_isScanning = false;
          m_progressBar->hide();
        }
        
        updateActions();
        return;  // 跳过空白页，不添加到文档
      }
    }

    // 添加处理后的图像到文档
    m_documentManager->addImage(processedImage);

    // 显示图像
    displayImage(processedImage);

    // 如果是延时扫描模式
    if (m_isDelayScanMode) {
      m_delayScanCount++;
      m_isScanning = false;
      m_progressBar->hide();
      
      // 开始下一轮倒计时
      m_delayScanRemaining = m_delayScanInterval;
      m_delayScanTimer->start();
      m_statusBar->showMessage(tr("第 %1 张扫描完成，准备扫描下一张，倒计时: %2 秒...").arg(m_delayScanCount).arg(m_delayScanRemaining));
    }
    // 如果是多页扫描，暂存图像，不立即自动保存
    else if (m_isMultiPageScanning) {
      m_multiPageImages.append(processedImage);
      m_statusBar->showMessage(
          tr("已扫描第 %1 页").arg(m_multiPageImages.size()), 2000);
    } else {
      // 单页扫描，立即处理自动保存
      m_isScanning = false;
      m_progressBar->hide();
      m_statusBar->showMessage(tr("扫描完成"), 3000);

      performAutoSave(processedImage);
    }
  }

  updateActions();
}

void MainWindow::onScanError(const QString &errorMessage) {
  m_isScanning = false;
  m_progressBar->hide();
  m_statusBar->showMessage(tr("扫描错误"), 3000);

  QMessageBox::critical(this, tr("扫描错误"), errorMessage);

  updateActions();
}

void MainWindow::onRotateLeftClicked() {
  QImage currentImage = m_documentManager->getCurrentImage();
  if (currentImage.isNull()) {
    return;
  }

  QImage rotatedImage = m_imageProcessor->rotateLeft(currentImage);
  if (!rotatedImage.isNull()) {
    int currentIndex = m_documentManager->getCurrentPageIndex();
    m_documentManager->removePage(currentIndex);
    m_documentManager->insertImage(currentIndex, rotatedImage);
    displayImage(rotatedImage);
  }
}

void MainWindow::onRotateRightClicked() {
  QImage currentImage = m_documentManager->getCurrentImage();
  if (currentImage.isNull()) {
    return;
  }

  QImage rotatedImage = m_imageProcessor->rotateRight(currentImage);
  if (!rotatedImage.isNull()) {
    int currentIndex = m_documentManager->getCurrentPageIndex();
    m_documentManager->removePage(currentIndex);
    m_documentManager->insertImage(currentIndex, rotatedImage);
    displayImage(rotatedImage);
  }
}

void MainWindow::onCropClicked() {
  if (m_currentImage.isNull()) {
    QMessageBox::warning(this, tr("裁剪"), tr("没有可裁剪的图像"));
    return;
  }

  if (m_imageLabel->isCroppingEnabled()) {
    // 取消裁剪模式
    m_imageLabel->setCroppingEnabled(false);
    m_statusBar->showMessage(tr("已取消裁剪模式"), 2000);
  } else {
    // 启用裁剪模式
    m_imageLabel->setCroppingEnabled(true);
    m_imageLabel->setFocus(); // 设置焦点以接收键盘事件
    m_statusBar->showMessage(
        tr("裁剪模式已启用 - 拖拽选择区域，ESC取消，Ctrl+R切换"), 0);

    QMessageBox::information(this, tr("裁剪模式"),
                             tr("裁剪模式已启用！\n\n"
                                "操作说明：\n"
                                "1. 用鼠标左键拖拽选择要裁剪的区域\n"
                                "2. 松开鼠标确认裁剪区域\n"
                                "3. 按ESC键或再次点击裁剪按钮取消裁剪模式\n"
                                "4. 快捷键：Ctrl+R 切换裁剪模式"));
  }
}

void MainWindow::onAdjustBrightnessContrastClicked() {
  // 简单实现，实际应用中需要对话框
  QMessageBox::information(this, tr("调整亮度和对比度"),
                           tr("亮度和对比度调整功能尚未实现"));
}

void MainWindow::onAutoLevelClicked() {
  QImage currentImage = m_documentManager->getCurrentImage();
  if (currentImage.isNull()) {
    return;
  }

  m_statusBar->showMessage(tr("正在自动调整图像..."));
  m_progressBar->setValue(0);
  m_progressBar->show();

  QImage processedImage = m_imageProcessor->autoLevel(currentImage);

  m_progressBar->hide();
  m_statusBar->showMessage(tr("自动调整完成"), 3000);

  if (!processedImage.isNull()) {
    int currentIndex = m_documentManager->getCurrentPageIndex();
    m_documentManager->removePage(currentIndex);
    m_documentManager->insertImage(currentIndex, processedImage);
    displayImage(processedImage);
  }
}

void MainWindow::onConvertToGrayscaleClicked() {
  QImage currentImage = m_documentManager->getCurrentImage();
  if (currentImage.isNull()) {
    return;
  }

  QImage grayImage = m_imageProcessor->convertToGrayscale(currentImage);
  if (!grayImage.isNull()) {
    int currentIndex = m_documentManager->getCurrentPageIndex();
    m_documentManager->removePage(currentIndex);
    m_documentManager->insertImage(currentIndex, grayImage);
    displayImage(grayImage);
  }
}

void MainWindow::onConvertToBinaryClicked() {
  QImage currentImage = m_documentManager->getCurrentImage();
  if (currentImage.isNull()) {
    return;
  }

  QImage binaryImage =
      m_imageProcessor->convertToBinary(currentImage, -1); // 自动阈值
  if (!binaryImage.isNull()) {
    int currentIndex = m_documentManager->getCurrentPageIndex();
    m_documentManager->removePage(currentIndex);
    m_documentManager->insertImage(currentIndex, binaryImage);
    displayImage(binaryImage);
  }
}

void MainWindow::onCropCompleted(const QRect &rect) {
  if (rect.isValid() && !m_currentImage.isNull()) {
    // 执行裁剪操作
    QImage croppedImage = m_imageProcessor->crop(m_currentImage, rect);

    if (!croppedImage.isNull()) {
      // 更新文档管理器中的图像
      int currentIndex = m_documentManager->getCurrentPageIndex();
      m_documentManager->removePage(currentIndex);
      m_documentManager->insertImage(currentIndex, croppedImage);

      // 显示裁剪后的图像
      displayImage(croppedImage);

      // 退出裁剪模式
      m_imageLabel->setCroppingEnabled(false);

      m_statusBar->showMessage(tr("图像裁剪完成"), 3000);

      // 显示裁剪信息
      QMessageBox::information(this, tr("裁剪完成"), tr("图像已成功裁剪！"));
    } else {
      QMessageBox::warning(this, tr("裁剪失败"),
                           tr("无法裁剪图像，请检查选择区域"));
    }
  }
}

void MainWindow::onCropCancelled() {
  m_imageLabel->setCroppingEnabled(false);
  m_statusBar->showMessage(tr("已取消裁剪操作"), 2000);
}

void MainWindow::onNewDocument() {
  if (maybeSave()) {
    m_documentManager->newDocument();
    m_currentImage = QImage();
    m_imageLabel->clear();
    updateWindowTitle();
    updateActions();
  }
}

void MainWindow::onOpenDocument() {
  if (maybeSave()) {
    QString filePath = QFileDialog::getOpenFileName(
        this, tr("打开文档"),
        QStandardPaths::writableLocation(QStandardPaths::DocumentsLocation),
        tr("图像文件 (*.png *.jpg *.jpeg *.bmp *.tiff *.tif);;PDF文件 "
           "(*.pdf);;所有文件 (*.*)"));

    if (!filePath.isEmpty()) {
      if (m_documentManager->loadDocument(filePath)) {
        m_currentImage = m_documentManager->getCurrentImage();
        displayImage(m_currentImage);
        updateWindowTitle();
        updateActions();
      }
    }
  }
}

void MainWindow::onSaveDocument() {
  QString filePath = m_documentManager->getCurrentFilePath();

  if (filePath.isEmpty()) {
    onSaveAsDocument();
  } else {
    m_documentManager->saveDocument(filePath);
  }
}

void MainWindow::onSaveAsDocument() {
  // 如果是多页文档，显示格式选择提示
  if (m_documentManager->getPageCount() > 1) {
    QMessageBox msgBox;
    msgBox.setWindowTitle(tr("多页文档保存提示"));
    msgBox.setText(tr("您正在保存多页扫描文档，请选择合适的保存格式："));
    msgBox.setInformativeText(
        tr("- PDF: 所有页面保存在一个PDF文件中\n"
           "- TIFF: 所有页面保存在一个TIFF文件中\n"
           "- JPG/PNG/BMP: 每页将保存为单独的图像文件，自动添加序号"));
    msgBox.setIcon(QMessageBox::Information);
    msgBox.addButton(QMessageBox::Ok);
    msgBox.exec();
  }

  // 生成自动递增的默认文件名
  QString defaultFileName = generateFileName("pdf"); // 默认使用PDF格式

  QString filePath = QFileDialog::getSaveFileName(
      this, tr("保存文档"), defaultFileName,
      tr("PDF文件 (*.pdf);;TIFF图像 (*.tiff);;PNG图像 (*.png);;JPEG图像 "
         "(*.jpg);;BMP图像 (*.bmp);;所有文件 (*.*)"));

  if (!filePath.isEmpty()) {
    m_documentManager->saveDocument(filePath);
    updateWindowTitle();
  }
}

void MainWindow::onSaveMultiPageDocument() {
  if (m_documentManager->getPageCount() <= 1) {
    QMessageBox::information(this, tr("提示"),
                             tr("当前文档只有一页，请使用普通保存功能"));
    return;
  }

  // 显示保存选项对话框
  QMessageBox msgBox;
  msgBox.setWindowTitle(tr("多页保存选项"));
  msgBox.setText(tr("请选择多页保存方式："));

  QPushButton *pdfButton =
      msgBox.addButton(tr("保存为PDF"), QMessageBox::ActionRole);
  QPushButton *tiffButton =
      msgBox.addButton(tr("保存为多页TIFF"), QMessageBox::ActionRole);
  QPushButton *imagesButton =
      msgBox.addButton(tr("保存为多个图像文件"), QMessageBox::ActionRole);
  msgBox.addButton(QMessageBox::Cancel);

  msgBox.exec();

  QString filePath;
  QString filter;

  if (msgBox.clickedButton() == pdfButton) {
    filter = tr("PDF文件 (*.pdf)");
  } else if (msgBox.clickedButton() == tiffButton) {
    filter = tr("TIFF图像 (*.tiff)");
  } else if (msgBox.clickedButton() == imagesButton) {
    filter = tr("JPEG图像 (*.jpg)");
  } else {
    return; // 用户取消
  }

  // 生成默认文件名，确保自动递增
  QString defaultFileName;
  if (msgBox.clickedButton() == pdfButton) {
    defaultFileName = generateFileName("pdf");
  } else if (msgBox.clickedButton() == tiffButton) {
    defaultFileName = generateFileName("tiff");
  } else if (msgBox.clickedButton() == imagesButton) {
    defaultFileName = generateFileName("jpg");
  }

  filePath = QFileDialog::getSaveFileName(this, tr("保存多页文档"),
                                          defaultFileName, filter);

  if (!filePath.isEmpty()) {
    if (msgBox.clickedButton() == imagesButton) {
      // 对于图像文件，使用saveAsImage方法会自动保存为多个文件
      m_documentManager->saveAsImage(filePath);
    } else {
      m_documentManager->saveDocument(filePath);
    }

    updateWindowTitle();

    QMessageBox::information(this, tr("保存成功"), tr("多页文档已成功保存"));
  }
}

void MainWindow::onSaveCurrentImage() {
  if (m_currentImage.isNull()) {
    QMessageBox::warning(this, tr("警告"), tr("没有可保存的图像"));
    return;
  }

  QString ext = m_formatCombo->currentData().toString();
  QString defaultPath = generateFileName(ext);

  QString filePath = QFileDialog::getSaveFileName(
      this, tr("保存当前图像"), defaultPath,
      tr("%1文件 (*.%2);;所有文件 (*.*)").arg(ext.toUpper()).arg(ext));

  if (!filePath.isEmpty()) {
    if (saveImage(m_currentImage, filePath)) {
      // 显示保存成功的消息对话框
      QMessageBox::information(this, tr("保存成功"),
                               tr("图像已成功保存到:\n%1").arg(filePath));

      // 在状态栏显示更详细的保存信息
      QString format = m_formatCombo->currentText();
      QString message = tr("已保存为 %1 格式到: %2")
                            .arg(format)
                            .arg(QFileInfo(filePath).fileName());
      m_statusBar->showMessage(message, 5000);

      // 更新自动保存计数器显示
      if (m_autoSaveCheck && m_autoSaveCheck->isChecked()) {
        updateAutoSaveCounter();
      }
    }
  }
}

void MainWindow::onExportDocument() {
  QString format = QInputDialog::getItem(
      this, tr("导出格式"), tr("选择导出格式:"),
      QStringList() << "PDF" << "PNG" << "JPEG" << "BMP" << "TIFF", 0, false);

  if (!format.isEmpty()) {
    QString filePath = QFileDialog::getSaveFileName(
        this, tr("导出文档"),
        QStandardPaths::writableLocation(QStandardPaths::DocumentsLocation) +
            "/" + m_documentManager->getDocumentTitle(),
        tr("所有文件 (*.*)"));

    if (!filePath.isEmpty()) {
      m_documentManager->exportDocument(filePath, format.toLower());
    }
  }
}

void MainWindow::onPageAdded(int index) {
  // 更新页面列表
  QListWidgetItem *item = new QListWidgetItem(tr("页面 %1").arg(index + 1));
  QImage thumbnail = m_documentManager->getPage(index).scaled(
      100, 100, Qt::KeepAspectRatio, Qt::SmoothTransformation);
  item->setIcon(QIcon(QPixmap::fromImage(thumbnail)));
  m_pagesList->insertItem(index, item);

  // 选择新添加的页面
  m_pagesList->setCurrentRow(index);
}

void MainWindow::onPageRemoved(int index) {
  // 从页面列表中删除
  delete m_pagesList->takeItem(index);

  // 更新页面编号
  for (int i = index; i < m_pagesList->count(); ++i) {
    m_pagesList->item(i)->setText(tr("页面 %1").arg(i + 1));
  }
}

void MainWindow::onPageMoved(int oldIndex, int newIndex) {
  // 移动列表项
  QListWidgetItem *item = m_pagesList->takeItem(oldIndex);
  m_pagesList->insertItem(newIndex, item);
  
  // 更新所有页面编号
  for (int i = 0; i < m_pagesList->count(); ++i) {
    m_pagesList->item(i)->setText(tr("页面 %1").arg(i + 1));
  }
  
  // 选中移动后的页面
  m_pagesList->setCurrentRow(newIndex);
}

void MainWindow::onCurrentPageChanged(int index) {
  // 更新页面列表选择
  if (index >= 0 && index < m_pagesList->count()) {
    m_pagesList->setCurrentRow(index);

    // 显示当前页面
    m_currentImage = m_documentManager->getPage(index);
    displayImage(m_currentImage);
  } else {
    m_currentImage = QImage();
    m_imageLabel->clear();
  }

  updateActions();
}

void MainWindow::onPageSelected(int index) {
  if (index >= 0 && index < m_documentManager->getPageCount()) {
    m_documentManager->setCurrentPage(index);
  }
}

void MainWindow::onClearAllClicked() {
  if (m_documentManager->getPageCount() == 0) {
    return;
  }

  QMessageBox::StandardButton reply;
  reply = QMessageBox::question(
      this, tr("确认清除"), tr("确定要删除所有扫描的图片吗？此操作不可撤销！"),
      QMessageBox::Yes | QMessageBox::No);

  if (reply == QMessageBox::Yes) {
    while (m_documentManager->getPageCount() > 0) {
      m_documentManager->removePage(0);
    }
  }
}

void MainWindow::onAboutClicked() {
  QString aboutText =
      tr("<div style='text-align:center;'>"
         "<img src=\":/icons/lz-scan.svg\" width=\"96\" height=\"96\"/><br/>"
         "<h2>亮仔扫描</h2>"
         "<p>版本: %1</p>"
         "<p>基于 Qt %2 与 SANE 后端</p>"
         "<p>版权所有 © 2025 克亮</p>"
         "<p>一个简单的扫描工具，支持多种扫描仪设备。</p>"
         "<p>本软件不提供任何的担保。</p>"
         "<p>感谢所使用的开源软件。</p>"
         "</div>")
          .arg(QCoreApplication::applicationVersion())
          .arg(QT_VERSION_STR);

  QMessageBox::about(this, tr("关于亮仔扫描"), aboutText);
}

void MainWindow::onPreferencesClicked() {
  QMessageBox::information(this, tr("首选项"), tr("首选项对话框尚未实现"));
}

void MainWindow::updateStatusBar() {
  int pageCount = m_documentManager->getPageCount();
  int currentPage = m_documentManager->getCurrentPageIndex() + 1;

  if (pageCount > 0) {
    m_statusBar->showMessage(tr("页面 %1/%2").arg(currentPage).arg(pageCount));
  } else {
    m_statusBar->showMessage(tr("就绪"));
  }
}

void MainWindow::updateAutoSaveCounter() {
  QString ext = m_formatCombo->currentData().toString();
  QDir dir(m_pathEdit->text());
  QStringList filters;
  filters << QString("*.%1").arg(ext);
  int count = dir.entryList(filters).count();

  // 更新状态栏显示自动保存计数
  m_statusBar->showMessage(tr("已自动保存 %1 个文件").arg(count), 3000);
}

void MainWindow::createActions() {
  // 这里简化实现，实际应用中需要创建所有操作
}

void MainWindow::createMenus() {
  // 这里简化实现，实际应用中需要创建所有菜单
}

void MainWindow::createToolbars() {
  // 创建主工具栏
  QToolBar *mainToolBar = new QToolBar(tr("主工具栏"), this);
  mainToolBar->setObjectName("mainToolBar");
  addToolBar(Qt::TopToolBarArea, mainToolBar);

  // 添加扫描按钮
  QAction *scanAction = new QAction(QIcon(":/icons/scan.svg"),
                                    tr("扫描"), this);
  scanAction->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_S));
  scanAction->setToolTip(tr("开始/停止扫描 "));
  connect(scanAction, &QAction::triggered, this,
          &MainWindow::onScanButtonClicked);
  mainToolBar->addAction(scanAction);

  // 添加保存按钮
  QAction *saveAction =
      new QAction(QIcon(":/icons/save-single.svg"), tr("保存"), this);
  saveAction->setShortcut(QKeySequence::Save);
  saveAction->setToolTip(tr("保存当前图像"));
  connect(saveAction, &QAction::triggered, this,
          &MainWindow::onSaveCurrentImage);
  mainToolBar->addAction(saveAction);

  // 添加多页保存按钮
  QAction *saveMultiAction =
      new QAction(QIcon(":/icons/save-multi.svg"), tr("多页保存"), this);
  saveMultiAction->setToolTip(tr("保存多页文档"));
  connect(saveMultiAction, &QAction::triggered, this,
          &MainWindow::onSaveMultiPageDocument);
  mainToolBar->addAction(saveMultiAction);

  // 添加分隔线
  mainToolBar->addSeparator();

  // 添加旋转按钮
  QAction *rotateLeftAction = new QAction(
      QIcon(":/icons/rotate-left.svg"), tr("左旋转"), this);
  rotateLeftAction->setToolTip(tr("向左旋转90度"));
  connect(rotateLeftAction, &QAction::triggered, this,
          &MainWindow::onRotateLeftClicked);
  mainToolBar->addAction(rotateLeftAction);

  QAction *rotateRightAction = new QAction(
      QIcon(":/icons/rotate-right.svg"), tr("右旋转"), this);
  rotateRightAction->setToolTip(tr("向右旋转90度"));
  connect(rotateRightAction, &QAction::triggered, this,
          &MainWindow::onRotateRightClicked);
  mainToolBar->addAction(rotateRightAction);

  // 添加高级设置按钮
  QAction *advancedAction =
      new QAction(QIcon(":/icons/settings-gear.svg"), tr("高级设置"), this);
  advancedAction->setToolTip(tr("高级图像处理设置"));
  connect(advancedAction, &QAction::triggered, this,
          &MainWindow::onAdvancedSettingsClicked);
  mainToolBar->addAction(advancedAction);
  
  // 添加扫描设置按钮
  QAction *scanSettingsAction =
      new QAction(QIcon(":/icons/scan-network.svg"), tr("扫描设置"), this);
  scanSettingsAction->setToolTip(tr("扫描仪搜索设置"));
  connect(scanSettingsAction, &QAction::triggered, this,
          &MainWindow::onScanSettingsClicked);
  mainToolBar->addAction(scanSettingsAction);
  
  // 添加界面布局按钮
  QAction *layoutAction =
      new QAction(QIcon(":/icons/layout-switch.svg"), tr("界面布局"), this);
  layoutAction->setToolTip(tr("切换界面布局"));
  connect(layoutAction, &QAction::triggered, this,
          &MainWindow::onLayoutSettingsClicked);
  mainToolBar->addAction(layoutAction);
  
  // 添加合并图片按钮
  QAction *mergeAction =
      new QAction(QIcon(":/icons/layout-columns.svg"), tr("合并图片"), this);
  mergeAction->setToolTip(tr("将选中的图片合并为一张"));
  connect(mergeAction, &QAction::triggered, this,
          &MainWindow::onMergeImagesClicked);
  mainToolBar->addAction(mergeAction);

  // 添加裁剪按钮
  QAction *cropAction =
      new QAction(QIcon(":/icons/crop.svg"),
                  tr("裁剪"), this);
  cropAction->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_R));
  cropAction->setToolTip(tr("裁剪当前图像 (Ctrl+R)"));
  connect(cropAction, &QAction::triggered, this, &MainWindow::onCropClicked);
  mainToolBar->addAction(cropAction);

  // 添加删除按钮
  QAction *deleteAction = new QAction(
      QIcon(":/icons/delete.svg"), tr("删除"), this);
  deleteAction->setToolTip(tr("删除当前图像"));
  connect(deleteAction, &QAction::triggered, this, [this]() {
    int currentIndex = m_documentManager->getCurrentPageIndex();
    if (currentIndex >= 0) {
      m_documentManager->removePage(currentIndex);
    }
  });
  mainToolBar->addAction(deleteAction);

  // 添加清除所有按钮
  QAction *clearAllAction = new QAction(
      QIcon(":/icons/clear-all.svg"), tr("清除"), this);
  clearAllAction->setToolTip(tr("清除所有扫描图片"));
  connect(clearAllAction, &QAction::triggered, this,
          &MainWindow::onClearAllClicked);
  mainToolBar->addAction(clearAllAction);

  // 添加关于按钮
  QAction *aboutAction =
      new QAction(QIcon(":/icons/about.svg"),
                  tr("关于"), this);
  aboutAction->setToolTip(tr("关于本软件"));
  connect(aboutAction, &QAction::triggered, this, &MainWindow::onAboutClicked);
  mainToolBar->addAction(aboutAction);

  // 确保工具栏显示工具提示
  mainToolBar->setToolButtonStyle(Qt::ToolButtonTextUnderIcon);
}

void MainWindow::createStatusBar() {
  m_statusBar = statusBar();

  m_progressBar = new QProgressBar(this);
  m_progressBar->setRange(0, 100);
  m_progressBar->setTextVisible(true);
  m_progressBar->setFixedWidth(150);
  m_progressBar->hide();

  m_statusBar->addPermanentWidget(m_progressBar);
  m_statusBar->showMessage(tr("就绪"));
}

void MainWindow::createCentralWidget() {
  // 创建图像显示区域
  m_imageLabel = new CroppableImageWidget(this);
  m_imageLabel->setBackgroundRole(QPalette::Base);
  m_imageLabel->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
  m_imageLabel->setScaledContents(false);
  m_imageLabel->setAlignment(Qt::AlignCenter);
  m_imageLabel->setMinimumSize(1, 1); // 确保即使图像很小也能显示

  m_imageScrollArea = new QScrollArea(this);
  m_imageScrollArea->setBackgroundRole(QPalette::Dark);
  m_imageScrollArea->setWidget(m_imageLabel);
  m_imageScrollArea->setWidgetResizable(true);

  setCentralWidget(m_imageScrollArea);
}

void MainWindow::createDockWidgets() {
  // 创建页面列表停靠窗口
  m_pagesDock = new QDockWidget(tr("页面"), this);
  m_pagesDock->setObjectName("pagesDock");
  m_pagesDock->setAllowedAreas(Qt::LeftDockWidgetArea |
                               Qt::RightDockWidgetArea);

  m_pagesList = new QListWidget(m_pagesDock);
  m_pagesList->setIconSize(QSize(80, 80));
  m_pagesList->setResizeMode(QListView::Adjust);
  m_pagesList->setViewMode(QListView::IconMode);
  m_pagesList->setDragDropMode(QAbstractItemView::InternalMove);
  m_pagesList->setSelectionMode(QAbstractItemView::ExtendedSelection);  // 支持 Ctrl/Shift 多选
  
  // 设置右键菜单
  m_pagesList->setContextMenuPolicy(Qt::CustomContextMenu);
  connect(m_pagesList, &QListWidget::customContextMenuRequested,
          this, &MainWindow::showPagesListContextMenu);

  m_pagesDock->setWidget(m_pagesList);
  m_pagesDock->setFeatures(QDockWidget::DockWidgetMovable |
                           QDockWidget::DockWidgetFloatable);
  m_pagesDock->show();
  addDockWidget(Qt::LeftDockWidgetArea, m_pagesDock);

  // 创建扫描设置停靠窗口
  QDockWidget *scanDock = new QDockWidget(tr(""), this);
  scanDock->setObjectName("scanSettingsDock");
  scanDock->setAllowedAreas(Qt::LeftDockWidgetArea | Qt::RightDockWidgetArea);
  scanDock->setFeatures(QDockWidget::DockWidgetMovable |
                        QDockWidget::DockWidgetFloatable);

  QWidget *scanWidget = new QWidget(scanDock);
  QVBoxLayout *scanLayout = new QVBoxLayout(scanWidget);

  // 保存设置部分 - 重新组织为更清晰的布局
  QGroupBox *saveGroup = new QGroupBox(tr("保存设置"), scanWidget);
  QVBoxLayout *saveLayout = new QVBoxLayout(saveGroup);

  // 路径设置
  QGroupBox *pathGroup = new QGroupBox(tr("保存路径"), saveGroup);
  QVBoxLayout *pathLayout = new QVBoxLayout(pathGroup);

  m_pathEdit = new QLineEdit(
      QStandardPaths::writableLocation(QStandardPaths::DocumentsLocation),
      pathGroup);
  m_pathEdit->setToolTip(tr("设置扫描文件的保存路径"));

  QPushButton *pathButton = new QPushButton(tr("选择保存文件夹..."), pathGroup);
  pathButton->setToolTip(tr("选择保存扫描文件的文件夹"));
  connect(pathButton, &QPushButton::clicked, this, &MainWindow::selectSavePath);

  pathLayout->addWidget(m_pathEdit);
  pathLayout->addWidget(pathButton);

  // 格式设置
  QGroupBox *formatGroup = new QGroupBox(tr("保存格式"), saveGroup);
  QFormLayout *formatLayout = new QFormLayout(formatGroup);

  m_formatCombo = new QComboBox(formatGroup);
  m_formatCombo->addItem("PDF", "pdf");
  m_formatCombo->addItem("PNG", "png");
  m_formatCombo->addItem("JPEG", "jpg");
  m_formatCombo->addItem("BMP", "bmp");
  m_formatCombo->addItem("TIFF", "tiff");
  m_formatCombo->setToolTip(tr("选择输出文件格式"));

  // JPEG质量设置
  m_qualitySlider = new QSlider(Qt::Horizontal, formatGroup);
  m_qualitySlider->setRange(50, 100);
  m_qualitySlider->setValue(90);
  m_qualitySlider->setTickInterval(10);
  m_qualitySlider->setTickPosition(QSlider::TicksBelow);
  m_qualitySlider->setToolTip(tr("设置JPEG图像质量 (50-100)"));

  formatLayout->addRow(tr("文件格式:"), m_formatCombo);
  formatLayout->addRow(tr("JPEG质量:"), m_qualitySlider);

  // 去除空白页选项（放在自动保存上方）
  QCheckBox *removeBlankPagesCheck = new QCheckBox(tr("自动删除空白页"), saveGroup);
  removeBlankPagesCheck->setChecked(m_removeBlankPagesEnabled);
  removeBlankPagesCheck->setToolTip(tr("启用后，扫描过程中检测到空白页将自动跳过"));
  connect(removeBlankPagesCheck, &QCheckBox::toggled, this, &MainWindow::onRemoveBlankPagesToggled);
  
  // 自动保存选项
  m_autoSaveCheck = new QCheckBox(tr("扫描后自动保存"), saveGroup);
  m_autoSaveCheck->setChecked(true);
  m_autoSaveCheck->setToolTip(tr("启用后，扫描完成后自动保存图像"));

  // 将所有组件添加到保存设置布局
  saveLayout->addWidget(pathGroup);
  saveLayout->addWidget(formatGroup);
  saveLayout->addWidget(removeBlankPagesCheck);  // 去除空白页选项
  saveLayout->addWidget(m_autoSaveCheck);        // 自动保存选项
  saveLayout->addStretch();

  // 扫描仪选择
  QGroupBox *scannerGroup = new QGroupBox(tr("扫描仪"), scanWidget);
  QVBoxLayout *scannerLayout = new QVBoxLayout(scannerGroup);

  m_scannerComboBox = new QComboBox(scannerGroup);
  m_scannerComboBox->setToolTip(tr("选择要使用的扫描仪设备"));

  QPushButton *refreshButton = new QPushButton(tr("查找设备"), scannerGroup);
  refreshButton->setEnabled(false);    // 初始禁用，等待窗口完全显示
  m_findDevicesBtn = refreshButton;    // 保存引用
  m_findDevicesBtn->setEnabled(false); // 初始禁用，等待窗口完全显示
  m_findDevicesBtn->setEnabled(false); // 初始禁用，等待窗口完全显示
  refreshButton->setToolTip(tr("查找可用的扫描仪设备"));

  scannerLayout->addWidget(m_scannerComboBox);
  scannerLayout->addWidget(refreshButton);

  // 扫描参数
  QGroupBox *paramGroup = new QGroupBox(tr("扫描参数"), scanWidget);
  QFormLayout *paramLayout = new QFormLayout(paramGroup);

  m_resolutionComboBox = new QComboBox(paramGroup);
  m_resolutionComboBox->addItems(QStringList()
                                 << "75" << "100" << "150" << "200" << "300"
                                 << "600" << "1200");
  m_resolutionComboBox->setCurrentText("300");
  m_resolutionComboBox->setToolTip(
      tr("设置扫描分辨率(DPI) - 数值越大图像质量越高"));

  m_colorModeComboBox = new QComboBox(paramGroup);
  m_colorModeComboBox->addItems(QStringList() << tr("彩色") << tr("灰度"));
  m_colorModeComboBox->setToolTip(tr("设置扫描颜色模式 - 彩色/灰度/黑白"));

  // 页面大小选择
  m_pageSizeComboBox = new QComboBox(paramGroup);
  m_pageSizeComboBox->addItems(QStringList()
                               << "A4" << "A5" << "Letter" << "Legal");
  m_pageSizeComboBox->setToolTip(tr("设置扫描页面大小"));

  // 纸张来源选择
  m_paperSourceComboBox = new QComboBox(paramGroup);
  m_paperSourceComboBox->addItem(tr("平板")); // 默认只有平板选项
  m_paperSourceComboBox->setToolTip(tr("设置扫描纸张来源"));

  // 双面扫描模式选择 - 已隐藏，通过纸张来源自动控制
  m_duplexModeComboBox = new QComboBox(paramGroup);
  m_duplexModeComboBox->addItem(tr("单面"), "Simplex");
  m_duplexModeComboBox->addItem(tr("双面"), "Duplex");
  m_duplexModeComboBox->setToolTip(tr("设置双面扫描模式（仅支持双面扫描的设备有效）"));
  m_duplexModeComboBox->setEnabled(false); // 默认禁用，等检测到支持双面扫描的设备后启用
  m_duplexModeComboBox->setVisible(false); // 隐藏双面扫描控件

  paramLayout->addRow(tr("分辨率:"), m_resolutionComboBox);
  paramLayout->addRow(tr("颜色模式:"), m_colorModeComboBox);
  paramLayout->addRow(tr("页面大小:"), m_pageSizeComboBox);
  paramLayout->addRow(tr("纸张来源:"), m_paperSourceComboBox);
  // paramLayout->addRow(tr("双面扫描:"), m_duplexModeComboBox); // 已隐藏

  // 连接页面大小选择信号
  connect(
      m_pageSizeComboBox, QOverload<int>::of(&QComboBox::currentIndexChanged),
      this, [this](int index) {
        QString size = m_pageSizeComboBox->itemText(index);
        // 设置扫描仪控制器的页面大小
        if (m_scannerController != nullptr) {
          m_scannerController->setPageSize(size);
        }

        // 在状态栏显示页面大小变更信息
        m_statusBar->showMessage(tr("页面大小已设置为: %1").arg(size), 2000);
      });

  // 创建上部设置区域容器
  QWidget *settingsContainer = new QWidget(scanWidget);
  QVBoxLayout *settingsLayout = new QVBoxLayout(settingsContainer);
  settingsLayout->setContentsMargins(0, 0, 0, 0);

  // 添加扫描设备、参数和保存设置到上部区域
  settingsLayout->addWidget(scannerGroup);
  settingsLayout->addWidget(paramGroup);
  settingsLayout->addWidget(saveGroup);
  settingsLayout->addStretch();

  // 创建底部按钮区域
  QWidget *buttonContainer = new QWidget(scanWidget);
  QHBoxLayout *buttonLayout = new QHBoxLayout(buttonContainer);
  buttonLayout->setContentsMargins(0, 0, 0, 0);

  QPushButton *delayScanButton = new QPushButton(tr("延时扫描"), buttonContainer);
  delayScanButton->setToolTip(tr("循环延时扫描，适合平板扫描仪多张扫描"));

  QPushButton *scanButton = new QPushButton(tr("扫描"), buttonContainer);
  scanButton->setToolTip(tr("执行完整扫描"));
  
  // 创建"扫描完成"按钮（初始隐藏）
  m_finishDelayScanBtn = new QPushButton(tr("扫描完成"), buttonContainer);
  m_finishDelayScanBtn->setToolTip(tr("结束延时扫描模式并保存"));
  m_finishDelayScanBtn->setStyleSheet("QPushButton { background-color: #4CAF50; color: white; font-weight: bold; }");
  m_finishDelayScanBtn->hide();

  buttonLayout->addWidget(delayScanButton);
  buttonLayout->addWidget(scanButton);
  buttonLayout->addWidget(m_finishDelayScanBtn);

  // 添加所有组件到主布局
  scanLayout->addWidget(settingsContainer);
  scanLayout->addWidget(buttonContainer);

  scanDock->setWidget(scanWidget);
  addDockWidget(Qt::RightDockWidgetArea, scanDock);

  // 连接信号
  connect(refreshButton, &QPushButton::clicked, this,
          &MainWindow::refreshScannerList);
  connect(scanButton, &QPushButton::clicked, this,
          &MainWindow::onScanButtonClicked);
  connect(delayScanButton, &QPushButton::clicked, this,
          &MainWindow::onDelayScanButtonClicked);
  connect(m_finishDelayScanBtn, &QPushButton::clicked, this,
          &MainWindow::onFinishDelayScanClicked);
}

void MainWindow::connectSignals() {
  // 扫描仪控制器信号将在延迟初始化时连接

  // 连接扫描仪选择信号 - 这是关键的修复！
  connect(m_scannerComboBox, QOverload<int>::of(&QComboBox::currentIndexChanged),
          this, &MainWindow::onScannerSelected);

  // 连接图像处理器信号
  connect(m_imageProcessor, &ImageProcessor::processingProgress, this,
          &MainWindow::onScanProgress);

  // 连接文档管理器信号
  connect(m_documentManager, &DocumentManager::pageAdded, this,
          &MainWindow::onPageAdded);
  connect(m_documentManager, &DocumentManager::pageRemoved, this,
          &MainWindow::onPageRemoved);
  connect(m_documentManager, &DocumentManager::pageMoved, this,
          &MainWindow::onPageMoved);
  connect(m_documentManager, &DocumentManager::currentPageChanged, this,
          &MainWindow::onCurrentPageChanged);
  connect(m_documentManager, &DocumentManager::documentChanged, this,
          &MainWindow::updateWindowTitle);
  connect(m_documentManager, &DocumentManager::documentError, this,
          [this](const QString &error) {
            QMessageBox::critical(this, tr("文档错误"), error);
          });

  // 连接裁剪控件信号
  connect(m_imageLabel, &CroppableImageWidget::cropCompleted, this,
          &MainWindow::onCropCompleted);
  connect(m_imageLabel, &CroppableImageWidget::cropCancelled, this,
          &MainWindow::onCropCancelled);

  // 连接页面列表信号
  connect(m_pagesList, &QListWidget::currentRowChanged, this,
          &MainWindow::onPageSelected);

  // 连接页面大小选择信号
  connect(m_pageSizeComboBox,
          QOverload<const QString &>::of(&QComboBox::currentTextChanged), this,
          [=](const QString &size) { 
            if (m_scannerController) {
              m_scannerController->setPageSize(size); 
            }
          });

  // 确保所有部件可见并设置合适大小
  m_pagesDock->show();
  findChild<QDockWidget *>("scanSettingsDock")->show();
  m_imageScrollArea->setMinimumSize(800, 600);
  update();
}

void MainWindow::loadSettings() {
  QSettings settings;

  // 加载窗口几何形状
  restoreGeometry(settings.value("geometry").toByteArray());
  restoreState(settings.value("windowState").toByteArray());

  // 确保Dock部件可见
  m_pagesDock->setFeatures(QDockWidget::DockWidgetClosable |
                           QDockWidget::DockWidgetMovable |
                           QDockWidget::DockWidgetFloatable);
  m_pagesDock->show();

  QDockWidget *scanDock = findChild<QDockWidget *>("scanSettingsDock");
  if (scanDock) {
    scanDock->setFeatures(QDockWidget::DockWidgetClosable |
                          QDockWidget::DockWidgetMovable |
                          QDockWidget::DockWidgetFloatable);
    scanDock->show();
  }

  // 加载最近使用的扫描仪
  QString lastScanner = settings.value("lastScanner").toString();
  if (!lastScanner.isEmpty()) {
    // 在扫描仪列表中查找并选择
    for (int i = 0; i < m_scannerComboBox->count(); ++i) {
      if (m_scannerComboBox->itemData(i).toString() == lastScanner) {
        m_scannerComboBox->setCurrentIndex(i);
        break;
      }
    }
  }

  // 加载扫描参数
  m_resolutionComboBox->setCurrentText(
      settings.value("resolution", "300").toString());
  m_colorModeComboBox->setCurrentIndex(settings.value("colorMode", 0).toInt());

  // 加载保存设置
  if (m_formatCombo) {
    m_formatCombo->setCurrentText(
        settings.value("save/format", "PDF").toString());
  }
  if (m_pathEdit) {
    m_pathEdit->setText(
        settings
            .value("save/path", QStandardPaths::writableLocation(
                                    QStandardPaths::DocumentsLocation))
            .toString());
  }
  if (m_autoSaveCheck) {
    m_autoSaveCheck->setChecked(settings.value("save/autoSave", true).toBool());
  }
  // 自动倾斜校正选项已移除
  m_saveCounter = settings.value("save/counter", 0).toInt();
  
  // 加载去除空白页设置
  m_removeBlankPagesEnabled = settings.value("removeBlankPages", false).toBool();
  m_blankPageThreshold = settings.value("blankPageThreshold", 240).toInt();
  m_blankPageRatio = settings.value("blankPageRatio", 98).toInt();
  
  // 加载扫描设置
  m_searchNetworkDevices = settings.value("searchNetworkDevices", false).toBool();
}

void MainWindow::selectSavePath() {
  QString dir = QFileDialog::getExistingDirectory(
      this, tr("选择保存路径"), m_pathEdit->text(),
      QFileDialog::ShowDirsOnly | QFileDialog::DontResolveSymlinks);

  if (!dir.isEmpty()) {
    m_pathEdit->setText(dir);
  }
}

void MainWindow::saveSettings() {
  QSettings settings;

  // 保存窗口几何形状
  settings.setValue("geometry", saveGeometry());
  settings.setValue("windowState", saveState());

  // 保存当前扫描仪
  int currentIndex = m_scannerComboBox->currentIndex();
  if (currentIndex >= 0) {
    settings.setValue("lastScanner",
                      m_scannerComboBox->itemData(currentIndex).toString());
  }

  // 保存扫描参数
  settings.setValue("resolution", m_resolutionComboBox->currentText());
  settings.setValue("colorMode", m_colorModeComboBox->currentIndex());

  // 保存保存设置
  if (m_formatCombo) {
    settings.setValue("save/format", m_formatCombo->currentText());
  }
  if (m_pathEdit) {
    settings.setValue("save/path", m_pathEdit->text());
  }
  if (m_autoSaveCheck) {
    settings.setValue("save/autoSave", m_autoSaveCheck->isChecked());
  }
  // 自动倾斜校正选项已移除
  settings.setValue("save/counter", m_saveCounter);
  
  // 保存去除空白页设置
  settings.setValue("removeBlankPages", m_removeBlankPagesEnabled);
  settings.setValue("blankPageThreshold", m_blankPageThreshold);
  settings.setValue("blankPageRatio", m_blankPageRatio);
  
  // 保存扫描设置
  settings.setValue("searchNetworkDevices", m_searchNetworkDevices);
}

bool MainWindow::maybeSave() {
  if (m_documentManager->hasUnsavedChanges()) {
    QMessageBox::StandardButton ret = QMessageBox::warning(
        this, tr("未保存的更改"), tr("文档包含未保存的更改。\n是否保存更改?"),
        QMessageBox::Save | QMessageBox::Discard | QMessageBox::Cancel);

    if (ret == QMessageBox::Save) {
      onSaveDocument();
      return !m_documentManager->hasUnsavedChanges();
    } else if (ret == QMessageBox::Cancel) {
      return false;
    } else if (ret == QMessageBox::Discard) {
      // 废弃更改，直接返回true允许关闭
      return true;
    }
  }
  return true;
}

void MainWindow::updateWindowTitle() {
  QString title = m_documentManager->getDocumentTitle();

  if (m_documentManager->hasUnsavedChanges()) {
    title += " *";
  }

  setWindowTitle(title + " - " + tr("亮仔扫描"));
}

void MainWindow::updateActions() {
  bool hasPages = m_documentManager->getPageCount() > 0;
  bool hasCurrentPage = m_documentManager->getCurrentPageIndex() >= 0;

  // 更新文档操作
  // 在实际应用中，这里会启用/禁用相应的操作

  // 更新扫描按钮
  // 在实际应用中，这里会根据扫描状态更改扫描按钮的文本

  // 更新状态栏
  updateStatusBar();
}

void MainWindow::displayImage(const QImage &image) {
  if (image.isNull()) {
    m_imageLabel->setImage(QImage());
    return;
  }

  m_currentImage = image;

  // 使用CroppableImageWidget的setImage方法
  m_imageLabel->setImage(image);
}

void MainWindow::resizeEvent(QResizeEvent *event) {
  QMainWindow::resizeEvent(event);
  // 当窗口大小改变时，重新缩放当前图像
  if (!m_currentImage.isNull()) {
    displayImage(m_currentImage);
  }
}

void MainWindow::refreshScannerList() {
  // 使用成员变量来跟踪刷新状态
  if (m_isScanning) // 重用现有的成员变量
    return;

  m_isScanning = true; // 临时使用这个变量

  // 临时断开信号连接，避免在更新列表时触发选择事件
  disconnect(m_scannerComboBox, QOverload<int>::of(&QComboBox::currentIndexChanged),
             this, &MainWindow::onScannerSelected);

  m_scannerComboBox->clear();
  QString searchMode = m_searchNetworkDevices ? tr("USB+局域网") : tr("仅USB");
  m_scannerComboBox->addItem(tr("正在查找扫描仪（%1模式）...").arg(searchMode));
  
  // 显示搜索状态
  m_statusBar->showMessage(tr("🔍 正在搜索扫描仪（%1模式），请稍候...").arg(searchMode));
  
  // 显示进度条（不确定进度）
  m_progressBar->setRange(0, 0);
  m_progressBar->show();
  
  // 处理事件，确保UI更新
  QApplication::processEvents();

  // 确保控制器存在
  if (m_scannerController == nullptr) {
    initializeScannerController();
  }

  // 检查初始化状态
  if (!m_scannerController->isInitialized()) {
    // 如果未初始化，触发异步初始化
    m_scannerController->initializeAsync();
    m_isScanning = false; // 重置状态，等待异步完成
    m_progressBar->hide();
    
    // 重新连接信号
    connect(m_scannerComboBox, QOverload<int>::of(&QComboBox::currentIndexChanged),
            this, &MainWindow::onScannerSelected);
    return;
  }

  // 如果已经初始化，直接获取扫描仪列表（使用当前搜索模式）
  QStringList scanners = m_scannerController->refreshScanners(m_searchNetworkDevices);
  
  // 隐藏进度条
  m_progressBar->hide();
  
  qDebug() << "发现扫描仪设备:" << scanners.size() << "个";
  for (int i = 0; i < scanners.size(); ++i) {
    qDebug() << "  设备" << (i+1) << ":" << scanners[i];
  }
  
  // 更新UI
  m_scannerComboBox->clear();
  
  for (const QString &scanner : scanners) {
    m_scannerComboBox->addItem(scanner);
  }

  // 重新连接信号
  connect(m_scannerComboBox, QOverload<int>::of(&QComboBox::currentIndexChanged),
          this, &MainWindow::onScannerSelected);

  if (m_scannerComboBox->count() > 0) {
    // 默认选择第一个设备
    m_scannerComboBox->setCurrentIndex(0);
    
    // 手动触发选择事件以初始化第一个设备
    onScannerSelected(0);
    
    // 显示成功消息
    if (scanners.size() == 1) {
      m_statusBar->showMessage(tr("✓ 找到 1 个扫描仪，已自动选择"), 3000);
    } else {
      // 统计本地和网络设备数量
      int networkCount = 0;
      for (const QString &scanner : scanners) {
        if (scanner.startsWith("[网络]")) {
          networkCount++;
        }
      }
      int localCount = scanners.size() - networkCount;
      
      QString message;
      if (networkCount > 0) {
        message = tr("✓ 找到 %1 个扫描仪（本地 %2 个，网络 %3 个），已选择第一个")
                    .arg(scanners.size()).arg(localCount).arg(networkCount);
      } else {
        message = tr("✓ 找到 %1 个扫描仪，已选择第一个，可手动切换").arg(scanners.size());
      }
      m_statusBar->showMessage(message, 5000);
    }
  } else {
    m_statusBar->showMessage(tr("✗ 未找到扫描仪设备，请检查连接或切换搜索模式"), 5000);
  }

  m_isScanning = false; // 重置状态
}

QImage MainWindow::performAutoDeskew(const QImage &image) {
  if (image.isNull()) {
    return image;
  }

  // 静默执行自动纠偏，不显示进度条和消息
  // 断开进度信号连接，避免干扰扫描进度显示
  disconnect(m_imageProcessor, &ImageProcessor::processingProgress, this,
             &MainWindow::onScanProgress);

  // 执行倾斜校正处理
  QImage deskewedImage = m_imageProcessor->deskew(image);

  // 重新连接进度信号
  connect(m_imageProcessor, &ImageProcessor::processingProgress, this,
          &MainWindow::onScanProgress, Qt::UniqueConnection);

  // 如果校正成功且图像有明显变化，返回校正后的图像
  if (!deskewedImage.isNull() && deskewedImage.size() == image.size()) {
    qDebug() << "自动倾斜校正处理完成";
    return deskewedImage;
  } else {
    qDebug() << "自动倾斜校正未执行或无需校正";
    return image;
  }
}

void MainWindow::saveToUndoStack(const QImage &image) {
  if (image.isNull()) {
    return;
  }
  
  // 添加到撤销栈
  m_undoStack.push(image);
  
  // 限制撤销栈大小
  while (m_undoStack.size() > MAX_UNDO_STEPS) {
    m_undoStack.removeFirst();
  }
  
  qDebug() << "已保存到撤销栈，当前栈大小:" << m_undoStack.size();
}

void MainWindow::onUndoClicked() {
  if (m_undoStack.isEmpty()) {
    QMessageBox::information(this, tr("提示"), tr("没有可撤销的操作"));
    return;
  }
  
  // 从撤销栈中取出上一个图像
  QImage previousImage = m_undoStack.pop();
  
  // 替换当前图像
  int currentIndex = m_documentManager->getCurrentPageIndex();
  if (currentIndex >= 0) {
    m_documentManager->removePage(currentIndex);
    m_documentManager->insertImage(currentIndex, previousImage);
    displayImage(previousImage);
    
    m_statusBar->showMessage(tr("已撤销操作（剩余 %1 步可撤销）").arg(m_undoStack.size()), 3000);
    qDebug() << "已撤销操作，剩余撤销步数:" << m_undoStack.size();
  }
}

void MainWindow::onRemoveBlankPagesToggled(bool enabled) {
  m_removeBlankPagesEnabled = enabled;
  
  if (enabled) {
    m_statusBar->showMessage(tr("已启用自动去除空白页功能"), 3000);
    qDebug() << "自动去除空白页功能已启用";
  } else {
    m_statusBar->showMessage(tr("已关闭自动去除空白页功能"), 3000);
    qDebug() << "自动去除空白页功能已关闭";
  }
  
  // 保存设置
  QSettings settings("LiangZai", "LZ-Scan");
  settings.setValue("removeBlankPages", enabled);
}

void MainWindow::onBlankPageSettingsClicked() {
  // 创建空白页参数设置对话框
  QDialog dialog(this);
  dialog.setWindowTitle(tr("空白页参数设置"));
  dialog.setMinimumWidth(500);
  dialog.setMinimumHeight(400);
  
  QVBoxLayout *layout = new QVBoxLayout(&dialog);
  
  QLabel *titleLabel = new QLabel(tr("<b>调整空白页检测灵敏度</b>"), &dialog);
  layout->addWidget(titleLabel);
  
  // 参数设置组
  QGroupBox *paramGroup = new QGroupBox(tr("检测参数"), &dialog);
  QVBoxLayout *paramLayout = new QVBoxLayout(paramGroup);
  
  // === 白色阈值设置 ===
  QLabel *thresholdLabel = new QLabel(tr("白色阈值"), paramGroup);
  thresholdLabel->setStyleSheet("font-weight: bold; font-size: 13px;");
  paramLayout->addWidget(thresholdLabel);
  
  // 滑动条和数值框的水平布局
  QHBoxLayout *thresholdLayout = new QHBoxLayout();
  
  // 亮度阈值滑动条
  QSlider *thresholdSlider = new QSlider(Qt::Horizontal, paramGroup);
  thresholdSlider->setRange(200, 255);
  thresholdSlider->setValue(m_blankPageThreshold);
  thresholdSlider->setTickPosition(QSlider::TicksBelow);
  thresholdSlider->setTickInterval(10);
  thresholdSlider->setMinimumWidth(300);
  thresholdSlider->setStyleSheet(
      "QSlider::groove:horizontal {"
      "    border: 1px solid #999999;"
      "    height: 8px;"
      "    background: qlineargradient(x1:0, y1:0, x2:1, y2:0, stop:0 #B0B0B0, stop:1 #00BFFF);"
      "    margin: 2px 0;"
      "    border-radius: 4px;"
      "}"
      "QSlider::handle:horizontal {"
      "    background: white;"
      "    border: 2px solid #5c5c5c;"
      "    width: 18px;"
      "    margin: -5px 0;"
      "    border-radius: 9px;"
      "}"
      "QSlider::handle:horizontal:hover {"
      "    background: #00BFFF;"
      "    border: 2px solid #0080FF;"
      "}"
  );
  
  // 亮度阈值数值框
  QSpinBox *thresholdSpinBox = new QSpinBox(paramGroup);
  thresholdSpinBox->setRange(200, 255);
  thresholdSpinBox->setValue(m_blankPageThreshold);
  thresholdSpinBox->setMinimumWidth(80);
  thresholdSpinBox->setAlignment(Qt::AlignCenter);
  thresholdSpinBox->setStyleSheet("font-size: 14px; padding: 5px;");
  
  // 连接滑动条和数值框
  connect(thresholdSlider, &QSlider::valueChanged, thresholdSpinBox, &QSpinBox::setValue);
  connect(thresholdSpinBox, QOverload<int>::of(&QSpinBox::valueChanged), thresholdSlider, &QSlider::setValue);
  
  thresholdLayout->addWidget(thresholdSlider);
  thresholdLayout->addWidget(thresholdSpinBox);
  paramLayout->addLayout(thresholdLayout);
  
  // 白色阈值说明
  QLabel *thresholdTip = new QLabel(
      tr("像素亮度大于此值被认为是空白像素（推荐：235-245）"), paramGroup);
  thresholdTip->setStyleSheet("color: #666; font-size: 11px; margin-bottom: 15px;");
  thresholdTip->setWordWrap(true);
  paramLayout->addWidget(thresholdTip);
  
  // === 阈值范围设置 ===
  QLabel *ratioLabel = new QLabel(tr("阈值范围"), paramGroup);
  ratioLabel->setStyleSheet("font-weight: bold; font-size: 13px;");
  paramLayout->addWidget(ratioLabel);
  
  // 滑动条和数值框的水平布局
  QHBoxLayout *ratioLayout = new QHBoxLayout();
  
  // 空白比例滑动条
  QSlider *ratioSlider = new QSlider(Qt::Horizontal, paramGroup);
  ratioSlider->setRange(85, 100);
  ratioSlider->setValue(m_blankPageRatio);
  ratioSlider->setTickPosition(QSlider::TicksBelow);
  ratioSlider->setTickInterval(5);
  ratioSlider->setMinimumWidth(300);
  ratioSlider->setStyleSheet(
      "QSlider::groove:horizontal {"
      "    border: 1px solid #999999;"
      "    height: 8px;"
      "    background: qlineargradient(x1:0, y1:0, x2:1, y2:0, stop:0 #B0B0B0, stop:1 #00BFFF);"
      "    margin: 2px 0;"
      "    border-radius: 4px;"
      "}"
      "QSlider::handle:horizontal {"
      "    background: white;"
      "    border: 2px solid #5c5c5c;"
      "    width: 18px;"
      "    margin: -5px 0;"
      "    border-radius: 9px;"
      "}"
      "QSlider::handle:horizontal:hover {"
      "    background: #00BFFF;"
      "    border: 2px solid #0080FF;"
      "}"
  );
  
  // 空白比例数值框
  QSpinBox *ratioSpinBox = new QSpinBox(paramGroup);
  ratioSpinBox->setRange(85, 100);
  ratioSpinBox->setValue(m_blankPageRatio);
  ratioSpinBox->setMinimumWidth(80);
  ratioSpinBox->setAlignment(Qt::AlignCenter);
  ratioSpinBox->setStyleSheet("font-size: 14px; padding: 5px;");
  
  // 连接滑动条和数值框
  connect(ratioSlider, &QSlider::valueChanged, ratioSpinBox, &QSpinBox::setValue);
  connect(ratioSpinBox, QOverload<int>::of(&QSpinBox::valueChanged), ratioSlider, &QSlider::setValue);
  
  ratioLayout->addWidget(ratioSlider);
  ratioLayout->addWidget(ratioSpinBox);
  paramLayout->addLayout(ratioLayout);
  
  // 阈值范围说明
  QLabel *ratioTip = new QLabel(
      tr("空白像素占比大于此值判定为空白页（推荐：96-99，已忽略边缘1%）"), paramGroup);
  ratioTip->setStyleSheet("color: #666; font-size: 11px;");
  ratioTip->setWordWrap(true);
  paramLayout->addWidget(ratioTip);
  
  layout->addWidget(paramGroup);
  
  // 说明文本
  QLabel *tipLabel = new QLabel(
      tr("<b>调整建议：</b><br>"
         "• 空白页未被检测到 → 降低白色阈值或阈值范围<br>"
         "• 非空白页被误判 → 提高白色阈值或阈值范围<br>"
         "• 有淡色背景的纸张 → 降低白色阈值到 230-235<br>"
         "• 有水印的空白页 → 降低阈值范围到 93-95<br>"
         "• 少量文字被误判 → 提高阈值范围到 99"), &dialog);
  tipLabel->setWordWrap(true);
  tipLabel->setStyleSheet("color: #444; font-size: 11px; padding: 12px; background-color: #e3f2fd; border-radius: 5px; border-left: 4px solid #2196F3;");
  layout->addWidget(tipLabel);
  
  // 按钮
  QDialogButtonBox *buttonBox = new QDialogButtonBox(
      QDialogButtonBox::Ok | QDialogButtonBox::Cancel | QDialogButtonBox::RestoreDefaults, &dialog);
  
  // 恢复默认值按钮
  QPushButton *defaultsBtn = buttonBox->button(QDialogButtonBox::RestoreDefaults);
  defaultsBtn->setText(tr("恢复默认"));
  connect(defaultsBtn, &QPushButton::clicked, [thresholdSlider, ratioSlider]() {
    thresholdSlider->setValue(240);  // 白色阈值默认值
    ratioSlider->setValue(98);       // 阈值范围默认值（配合边缘忽略）
  });
  
  connect(buttonBox, &QDialogButtonBox::accepted, &dialog, &QDialog::accept);
  connect(buttonBox, &QDialogButtonBox::rejected, &dialog, &QDialog::reject);
  layout->addWidget(buttonBox);
  
  if (dialog.exec() == QDialog::Accepted) {
    // 保存新的参数
    m_blankPageThreshold = thresholdSpinBox->value();
    m_blankPageRatio = ratioSpinBox->value();
    
    // 保存到配置文件
    QSettings settings("LiangZai", "LZ-Scan");
    settings.setValue("blankPageThreshold", m_blankPageThreshold);
    settings.setValue("blankPageRatio", m_blankPageRatio);
    
    m_statusBar->showMessage(
        tr("空白页检测参数已更新：白色阈值=%1, 阈值范围=%2")
        .arg(m_blankPageThreshold)
        .arg(m_blankPageRatio), 5000);
    
    qDebug() << "空白页检测参数已更新: threshold=" << m_blankPageThreshold 
             << "ratio=" << m_blankPageRatio;
  }
}

void MainWindow::onScanSettingsClicked() {
  // 创建扫描设置对话框
  QDialog dialog(this);
  dialog.setWindowTitle(tr("扫描设置"));
  dialog.setMinimumWidth(450);
  
  QVBoxLayout *layout = new QVBoxLayout(&dialog);
  
  QLabel *titleLabel = new QLabel(tr("<b>扫描仪搜索设置</b>"), &dialog);
  layout->addWidget(titleLabel);
  
  // 搜索模式设置组
  QGroupBox *searchGroup = new QGroupBox(tr("搜索模式"), &dialog);
  QVBoxLayout *searchLayout = new QVBoxLayout(searchGroup);
  
  // USB连接模式（默认）
  QRadioButton *usbOnlyRadio = new QRadioButton(tr("仅USB连接"), searchGroup);
  usbOnlyRadio->setChecked(!m_searchNetworkDevices);
  usbOnlyRadio->setToolTip(tr("只搜索USB连接的扫描仪（快速，推荐）"));
  searchLayout->addWidget(usbOnlyRadio);
  
  QLabel *usbTip = new QLabel(tr("  • 搜索速度：快（< 5秒）\n"
                                 "  • 设备类型：USB、并口、串口\n"
                                 "  • 推荐：日常使用"), searchGroup);
  usbTip->setStyleSheet("color: #666; font-size: 11px; margin-left: 20px; margin-bottom: 10px;");
  searchLayout->addWidget(usbTip);
  
  // USB+局域网模式
  QRadioButton *networkRadio = new QRadioButton(tr("USB + 局域网"), searchGroup);
  networkRadio->setChecked(m_searchNetworkDevices);
  networkRadio->setToolTip(tr("搜索USB和局域网中的扫描仪（较慢）"));
  searchLayout->addWidget(networkRadio);
  
  QLabel *networkTip = new QLabel(tr("  • 搜索速度：较慢（5-15秒）\n"
                                     "  • 设备类型：USB + 网络设备（eSCL/AirScan）\n"
                                     "  • 适用：有网络扫描仪时使用"), searchGroup);
  networkTip->setStyleSheet("color: #666; font-size: 11px; margin-left: 20px;");
  searchLayout->addWidget(networkTip);
  
  layout->addWidget(searchGroup);
  
  // 说明信息
  QLabel *infoLabel = new QLabel(
      tr("<b>关于网络扫描仪：</b><br>"
         "• 支持 eSCL/AirScan 协议的网络扫描仪<br>"
         "• 常见品牌：HP、Canon、Epson、Brother<br>"
         "• 网络设备会显示为 [网络] 前缀<br>"
         "• 需要安装 sane-airscan 后端<br><br>"
         "<b>提示：</b><br>"
         "• 默认使用 USB 模式可获得最快的启动速度<br>"
         "• 如果局域网中有网络扫描仪，切换到 USB+局域网 模式<br>"
         "• 切换模式后需要重新点击 [查找设备]"), &dialog);
  infoLabel->setWordWrap(true);
  infoLabel->setStyleSheet("color: #444; font-size: 11px; padding: 12px; background-color: #fff3cd; border-radius: 5px; border-left: 4px solid #ffc107;");
  layout->addWidget(infoLabel);
  
  // 按钮
  QDialogButtonBox *buttonBox = new QDialogButtonBox(
      QDialogButtonBox::Ok | QDialogButtonBox::Cancel, &dialog);
  connect(buttonBox, &QDialogButtonBox::accepted, &dialog, &QDialog::accept);
  connect(buttonBox, &QDialogButtonBox::rejected, &dialog, &QDialog::reject);
  layout->addWidget(buttonBox);
  
  if (dialog.exec() == QDialog::Accepted) {
    // 保存新的设置
    bool oldSetting = m_searchNetworkDevices;
    m_searchNetworkDevices = networkRadio->isChecked();
    
    // 保存到配置文件
    QSettings settings("LiangZai", "LZ-Scan");
    settings.setValue("searchNetworkDevices", m_searchNetworkDevices);
    
    QString mode = m_searchNetworkDevices ? tr("USB+局域网") : tr("仅USB");
    m_statusBar->showMessage(
        tr("扫描模式已更新：%1").arg(mode), 5000);
    
    qDebug() << "扫描模式已更新:" << mode;
    
    // 如果设置改变，提示用户重新搜索
    if (oldSetting != m_searchNetworkDevices) {
      QMessageBox::information(this, tr("设置已更新"), 
          tr("扫描模式已切换到：%1\n\n"
             "请点击 [查找设备] 按钮重新搜索扫描仪。").arg(mode));
    }
  }
}

#include <chrono>
#include <iostream>

void MainWindow::showEvent(QShowEvent *event) {
  QMainWindow::showEvent(event);
  m_findDevicesBtn->setEnabled(true); // 窗口完全显示后启用按钮
  auto end = std::chrono::high_resolution_clock::now();
  auto duration =
      std::chrono::duration_cast<std::chrono::milliseconds>(end - m_startTime);
  std::cout << "程序启动到主窗口完全显示耗时: " << duration.count() << " 毫秒"
            << std::endl;

  // 窗口显示时的初始化操作
  updateActions();
  updateStatusBar();
}

// 延迟初始化方法
bool MainWindow::ensureScannerControllerInitialized() {
  if (m_scannerController != nullptr && m_scannerController->isInitialized()) {
    return true; // 已经完全初始化
  }

  if (m_scannerController == nullptr) {
    // 创建控制器
    initializeScannerController();
  }

  if (!m_scannerController->isInitialized()) {
    // 显示初始化提示
    m_statusBar->showMessage(tr("正在初始化扫描仪，请稍候..."));
    m_progressBar->setRange(0, 0); // 显示不确定进度条
    m_progressBar->show();

    // 开始异步初始化
    m_scannerController->initializeAsync();

    // 对于需要立即使用的情况，返回false让调用者知道需要等待
    return false;
  }

  return true;
}

void MainWindow::initializeScannerController() {
  // 创建扫描仪控制器
  m_scannerController = new ScannerController(this);

  // 连接扫描信号槽
  connect(m_scannerController, &ScannerController::scanProgress, this,
          &MainWindow::onScanProgress);
  connect(m_scannerController, &ScannerController::scanComplete, this,
          &MainWindow::onScanComplete);
  connect(m_scannerController, &ScannerController::scanError, this,
          &MainWindow::onScanError);

  // 连接异步初始化信号槽
  connect(
      m_scannerController, &ScannerController::initializationProgress, this,
      [this](const QString &message) { m_statusBar->showMessage(message); });

  connect(m_scannerController, &ScannerController::initializationCompleted,
          this, [this]() {
            m_progressBar->hide();
            
            // 临时断开信号连接，避免在更新列表时触发选择事件
            disconnect(m_scannerComboBox, QOverload<int>::of(&QComboBox::currentIndexChanged),
                       this, &MainWindow::onScannerSelected);
            
            m_scannerComboBox->clear();

            QStringList scanners = m_scannerController->getAvailableScanners();
            qDebug() << "异步初始化完成，发现扫描仪:" << scanners.size() << "个";
            
            for (const QString &scanner : scanners) {
              m_scannerComboBox->addItem(scanner, scanner);
              qDebug() << "  添加设备:" << scanner;
            }

            // 重新连接信号
            connect(m_scannerComboBox, QOverload<int>::of(&QComboBox::currentIndexChanged),
                    this, &MainWindow::onScannerSelected);

            if (m_scannerComboBox->count() > 0) {
              // 默认选择第一个设备
              m_scannerComboBox->setCurrentIndex(0);
              
              // 手动触发选择事件以初始化第一个设备
              onScannerSelected(0);
              
              if (scanners.size() == 1) {
                m_statusBar->showMessage(tr("找到 1 个扫描仪，已自动选择"), 3000);
              } else {
                m_statusBar->showMessage(tr("找到 %1 个扫描仪，已选择第一个，可手动切换").arg(scanners.size()), 5000);
              }
            } else {
              m_statusBar->showMessage(tr("未找到扫描仪设备"), 3000);
            }
          });

  connect(m_scannerController, &ScannerController::initializationFailed, this,
          [this](const QString &error) {
            m_progressBar->hide();
            m_statusBar->showMessage(tr("初始化失败: %1").arg(error), 5000);
            QMessageBox::warning(this, tr("初始化错误"), error);
          });

  // 连接ADF相关信号
  connect(m_scannerController, &ScannerController::adfStatusChanged, this,
          &MainWindow::onADFStatusChanged);
  connect(m_scannerController, &ScannerController::adfEmptyWarning, this,
          &MainWindow::onADFEmptyWarning);

  // 连接多页扫描信号
  connect(m_scannerController, &ScannerController::multiPageScanStarted, this,
          &MainWindow::onMultiPageScanStarted);
  connect(m_scannerController, &ScannerController::multiPageScanFinished, this,
          &MainWindow::onMultiPageScanFinished);
  connect(m_scannerController, &ScannerController::currentPageScanning, this,
          &MainWindow::onCurrentPageScanning);

  qDebug() << "扫描仪控制器已创建，准备异步初始化";
}

void MainWindow::onCurrentPageScanning(int pageNumber) {
  // 更新状态栏显示当前正在扫描的页码
  m_statusBar->showMessage(tr("正在扫描第 %1 页...").arg(pageNumber));
  qDebug() << "正在扫描第" << pageNumber << "页";
}

void MainWindow::onAdvancedSettingsClicked() {
  // 创建高级设置对话框
  QDialog *dialog = new QDialog(this);
  dialog->setWindowTitle(tr("高级图像处理设置"));
  dialog->setModal(true);
  dialog->resize(400, 300);

  QVBoxLayout *layout = new QVBoxLayout(dialog);

  // 创建功能按钮组
  QGroupBox *processingGroup = new QGroupBox(tr("图像处理功能"), dialog);
  QVBoxLayout *processingLayout = new QVBoxLayout(processingGroup);

  // 角度调整按钮
  QPushButton *deskewBtn = new QPushButton(tr("角度调整"), processingGroup);
  deskewBtn->setToolTip(tr("手动调整图像旋转角度（支持精确到0.1度）"));
  deskewBtn->setIcon(QIcon(":/icons/auto-deskew.svg"));
  connect(deskewBtn, &QPushButton::clicked, this, [this, dialog]() {
    dialog->accept();
    onAutoDeskewClicked();
  });
  processingLayout->addWidget(deskewBtn);

  // 文档增强按钮
  QPushButton *enhanceTextBtn =
      new QPushButton(tr("文档增强"), processingGroup);
  enhanceTextBtn->setToolTip(tr("保留彩色，突出内容，去除背景"));
  enhanceTextBtn->setIcon(QIcon(":/icons/enhance-text.svg"));
  connect(enhanceTextBtn, &QPushButton::clicked, this, [this, dialog]() {
    dialog->accept();
    onEnhanceTextClicked();
  });
  processingLayout->addWidget(enhanceTextBtn);

  // 去除边框按钮
  QPushButton *removeBorderBtn =
      new QPushButton(tr("去除边框"), processingGroup);
  removeBorderBtn->setToolTip(tr("自动去除扫描图像的边框"));
  removeBorderBtn->setIcon(QIcon(":/icons/remove-border.svg"));
  connect(removeBorderBtn, &QPushButton::clicked, this, [this, dialog]() {
    dialog->accept();
    onRemoveBorderClicked();
  });
  processingLayout->addWidget(removeBorderBtn);

  // 图片漂白按钮
  QPushButton *bleachBtn =
      new QPushButton(tr("图片漂白"), processingGroup);
  bleachBtn->setToolTip(tr("去除扫描图像的底色，使背景更白净"));
  bleachBtn->setIcon(QIcon(":/icons/bleach.svg"));
  connect(bleachBtn, &QPushButton::clicked, this, [this, dialog]() {
    dialog->accept();
    onBleachBackgroundClicked();
  });
  processingLayout->addWidget(bleachBtn);
  
  // 空白页参数设置按钮
  QPushButton *blankPageSettingsBtn =
      new QPushButton(tr("空白页参数"), processingGroup);
  blankPageSettingsBtn->setToolTip(tr("调整空白页检测的灵敏度参数"));
  blankPageSettingsBtn->setIcon(QIcon(":/icons/blank-page-simple.svg"));
  connect(blankPageSettingsBtn, &QPushButton::clicked, this, [this, dialog]() {
    dialog->accept();
    onBlankPageSettingsClicked();
  });
  processingLayout->addWidget(blankPageSettingsBtn);

  // 亮度对比度按钮
  QPushButton *brightnessContrastBtn =
      new QPushButton(tr("亮度、对比度"), processingGroup);
  brightnessContrastBtn->setToolTip(tr("调整图像的亮度和对比度"));
  brightnessContrastBtn->setIcon(QIcon(":/icons/brightness.svg"));
  connect(brightnessContrastBtn, &QPushButton::clicked, this, [this, dialog]() {
    dialog->accept();
    onAdjustBrightnessContrastDialogClicked();
  });
  processingLayout->addWidget(brightnessContrastBtn);

  // 色度饱和度按钮
  QPushButton *hueSaturationBtn =
      new QPushButton(tr("色度、饱和度"), processingGroup);
  hueSaturationBtn->setToolTip(tr("调整图像的色度和饱和度"));
  hueSaturationBtn->setIcon(QIcon(":/icons/color.svg"));
  connect(hueSaturationBtn, &QPushButton::clicked, this, [this, dialog]() {
    dialog->accept();
    onAdjustHueSaturationClicked();
  });
  processingLayout->addWidget(hueSaturationBtn);

  // 添加分隔线
  QFrame *line = new QFrame(processingGroup);
  line->setFrameShape(QFrame::HLine);
  line->setFrameShadow(QFrame::Sunken);
  processingLayout->addWidget(line);

  // 撤销按钮
  QPushButton *undoBtn = new QPushButton(tr("撤销上一步操作"), processingGroup);
  undoBtn->setToolTip(tr("撤销最近一次的图像处理操作（最多可撤销10步）"));
  undoBtn->setIcon(QIcon(":/icons/undo.svg"));
  undoBtn->setStyleSheet("QPushButton { background-color: #FFA500; color: white; font-weight: bold; }");
  connect(undoBtn, &QPushButton::clicked, this, [this, dialog]() {
    dialog->accept();
    onUndoClicked();
  });
  processingLayout->addWidget(undoBtn);

  layout->addWidget(processingGroup);

  // 添加说明文本
  QLabel *infoLabel =
      new QLabel(tr("提示：这些高级功能将应用于当前选中的图像。\n"
                    "处理可能需要一些时间，请耐心等待。\n"
                    "所有操作都支持撤销（最多10步）。"),
                 dialog);
  infoLabel->setWordWrap(true);
  infoLabel->setStyleSheet("QLabel { color: #666; font-size: 12px; }");
  layout->addWidget(infoLabel);

  // 添加按钮框
  QDialogButtonBox *buttonBox =
      new QDialogButtonBox(QDialogButtonBox::Close, dialog);
  connect(buttonBox, &QDialogButtonBox::rejected, dialog, &QDialog::reject);
  layout->addWidget(buttonBox);

  dialog->exec();
  dialog->deleteLater();
}

void MainWindow::onAutoDeskewClicked() {
  QImage currentImage = m_documentManager->getCurrentImage();
  if (currentImage.isNull()) {
    QMessageBox::information(this, tr("提示"), tr("请先选择要处理的图像"));
    return;
  }

  // 保存到撤销栈
  saveToUndoStack(currentImage);

  // 创建角度调整对话框
  QDialog dialog(this);
  dialog.setWindowTitle(tr("角度调整"));
  dialog.setMinimumWidth(350);
  
  QVBoxLayout *layout = new QVBoxLayout(&dialog);
  
  QLabel *titleLabel = new QLabel(tr("<b>手动调整图像旋转角度</b>"), &dialog);
  layout->addWidget(titleLabel);
  
  // 快速调整按钮
  QGroupBox *quickGroup = new QGroupBox(tr("快速调整"), &dialog);
  QHBoxLayout *quickLayout = new QHBoxLayout(quickGroup);
  
  QPushButton *leftBtn = new QPushButton(tr("← 左转 1°"), quickGroup);
  QPushButton *rightBtn = new QPushButton(tr("右转 1° →"), quickGroup);
  
  quickLayout->addWidget(leftBtn);
  quickLayout->addWidget(rightBtn);
  
  layout->addWidget(quickGroup);
  
  // 精确调整
  QGroupBox *preciseGroup = new QGroupBox(tr("精确调整"), &dialog);
  QFormLayout *preciseLayout = new QFormLayout(preciseGroup);
  
  QDoubleSpinBox *angleSpinBox = new QDoubleSpinBox(preciseGroup);
  angleSpinBox->setRange(-180.0, 180.0);
  angleSpinBox->setSingleStep(0.1);
  angleSpinBox->setValue(0.0);
  angleSpinBox->setSuffix(tr(" 度"));
  angleSpinBox->setDecimals(1);
  
  preciseLayout->addRow(tr("旋转角度:"), angleSpinBox);
  layout->addWidget(preciseGroup);
  
  // 提示信息
  QLabel *tipLabel = new QLabel(
      tr("\n<b>使用说明：</b>\n"
         "• 正值：顺时针旋转\n"
         "• 负值：逆时针旋转\n"
         "• 快速按钮：快速调整常用角度\n"
         "• 精确输入：输入任意角度（支持小数）\n"
         "• 可以多次调整，点击确定后应用"), &dialog);
  tipLabel->setWordWrap(true);
  tipLabel->setStyleSheet("color: #666; font-size: 11px; padding: 10px; background-color: #f0f0f0; border-radius: 5px;");
  layout->addWidget(tipLabel);
  
  // 按钮连接
  connect(leftBtn, &QPushButton::clicked, [angleSpinBox]() {
    angleSpinBox->setValue(angleSpinBox->value() - 1.0);
  });
  connect(rightBtn, &QPushButton::clicked, [angleSpinBox]() {
    angleSpinBox->setValue(angleSpinBox->value() + 1.0);
  });
  
  QDialogButtonBox *buttonBox = new QDialogButtonBox(
      QDialogButtonBox::Ok | QDialogButtonBox::Cancel, &dialog);
  connect(buttonBox, &QDialogButtonBox::accepted, &dialog, &QDialog::accept);
  connect(buttonBox, &QDialogButtonBox::rejected, &dialog, &QDialog::reject);
  layout->addWidget(buttonBox);
  
  if (dialog.exec() == QDialog::Accepted) {
    double angle = angleSpinBox->value();
    
    if (qAbs(angle) < 0.01) {
      QMessageBox::information(this, tr("提示"), tr("旋转角度为0，无需调整"));
      return;
    }
    
    // 显示处理进度
    m_statusBar->showMessage(tr("正在旋转图像 %1 度...").arg(angle));
    m_progressBar->setValue(0);
    m_progressBar->show();
    
    // 连接进度信号
    connect(m_imageProcessor, &ImageProcessor::processingProgress, this,
            &MainWindow::onScanProgress, Qt::UniqueConnection);
    
    // 执行旋转
    QImage rotatedImage = m_imageProcessor->rotateByAngle(currentImage, angle);
    
    m_progressBar->hide();
    
    if (!rotatedImage.isNull()) {
      // 替换当前图像
      int currentIndex = m_documentManager->getCurrentPageIndex();
      m_documentManager->removePage(currentIndex);
      m_documentManager->insertImage(currentIndex, rotatedImage);
      displayImage(rotatedImage);
      
      m_statusBar->showMessage(tr("角度调整完成：旋转 %1 度").arg(angle), 3000);
    } else {
      m_statusBar->showMessage(tr("角度调整失败"), 3000);
      QMessageBox::warning(this, tr("处理失败"), tr("角度调整失败，请重试"));
    }
  }
}

void MainWindow::onEnhanceTextClicked() {
  QImage currentImage = m_documentManager->getCurrentImage();
  if (currentImage.isNull()) {
    QMessageBox::information(this, tr("提示"), tr("请先选择要处理的图像"));
    return;
  }

  // 保存到撤销栈
  saveToUndoStack(currentImage);

  m_statusBar->showMessage(tr("正在进行文档增强处理（保留彩色，突出内容）..."));
  m_progressBar->setValue(0);
  m_progressBar->show();

  // 连接进度信号
  connect(m_imageProcessor, &ImageProcessor::processingProgress, this,
          &MainWindow::onScanProgress, Qt::UniqueConnection);

  QImage enhancedImage = m_imageProcessor->enhanceText(currentImage);

  m_progressBar->hide();

  if (!enhancedImage.isNull()) {
    int currentIndex = m_documentManager->getCurrentPageIndex();
    m_documentManager->removePage(currentIndex);
    m_documentManager->insertImage(currentIndex, enhancedImage);
    displayImage(enhancedImage);

    m_statusBar->showMessage(tr("文档增强完成"), 3000);
    QMessageBox::information(this, tr("处理完成"),
                             tr("文档增强处理已完成\n已保留彩色并突出内容"));
  }
}

void MainWindow::onRemoveBorderClicked() {
  QImage currentImage = m_documentManager->getCurrentImage();
  if (currentImage.isNull()) {
    QMessageBox::information(this, tr("提示"), tr("请先选择要处理的图像"));
    return;
  }

  bool ok;
  int borderWidth = QInputDialog::getInt(this, tr("去除边框"),
                                         tr("请输入要去除的边框宽度（像素）:"),
                                         10, 0, 100, 1, &ok);
  if (!ok) {
    return;
  }

  // 保存到撤销栈
  saveToUndoStack(currentImage);

  m_statusBar->showMessage(tr("正在去除边框..."));

  QImage processedImage =
      m_imageProcessor->removeBorder(currentImage, borderWidth);

  if (!processedImage.isNull()) {
    int currentIndex = m_documentManager->getCurrentPageIndex();
    m_documentManager->removePage(currentIndex);
    m_documentManager->insertImage(currentIndex, processedImage);
    displayImage(processedImage);

    m_statusBar->showMessage(tr("边框去除完成"), 3000);
  }
}

void MainWindow::onBleachBackgroundClicked() {
  QImage currentImage = m_documentManager->getCurrentImage();
  if (currentImage.isNull()) {
    QMessageBox::information(this, tr("提示"), tr("请先选择要处理的图像"));
    return;
  }

  // 保存到撤销栈
  saveToUndoStack(currentImage);

  // 创建漂白设置对话框
  QDialog *bleachDialog = new QDialog(this);
  bleachDialog->setWindowTitle(tr("图片漂白设置"));
  bleachDialog->setModal(true);
  bleachDialog->resize(350, 200);

  QVBoxLayout *layout = new QVBoxLayout(bleachDialog);

  // 阈值设置
  QGroupBox *thresholdGroup = new QGroupBox(tr("漂白参数"), bleachDialog);
  QFormLayout *formLayout = new QFormLayout(thresholdGroup);

  QSpinBox *thresholdSpinBox = new QSpinBox();
  thresholdSpinBox->setRange(180, 255);
  thresholdSpinBox->setValue(240);
  thresholdSpinBox->setToolTip(tr("背景色检测阈值，值越高检测越严格"));
  formLayout->addRow(tr("检测阈值:"), thresholdSpinBox);

  QSpinBox *strengthSpinBox = new QSpinBox();
  strengthSpinBox->setRange(20, 100);
  strengthSpinBox->setValue(80);
  strengthSpinBox->setToolTip(tr("漂白强度，值越高漂白效果越强"));
  formLayout->addRow(tr("漂白强度:"), strengthSpinBox);

  layout->addWidget(thresholdGroup);

  // 说明文本
  QLabel *infoLabel = new QLabel(tr("图片漂白功能可以去除扫描文档的底色，\n"
                                    "使背景更加白净。适用于有轻微底色的文档。"), bleachDialog);
  infoLabel->setWordWrap(true);
  infoLabel->setStyleSheet("QLabel { color: #666; font-size: 12px; margin: 10px; }");
  layout->addWidget(infoLabel);

  // 按钮
  QDialogButtonBox *buttonBox = new QDialogButtonBox(
      QDialogButtonBox::Ok | QDialogButtonBox::Cancel, bleachDialog);
  connect(buttonBox, &QDialogButtonBox::accepted, bleachDialog, &QDialog::accept);
  connect(buttonBox, &QDialogButtonBox::rejected, bleachDialog, &QDialog::reject);
  layout->addWidget(buttonBox);

  if (bleachDialog->exec() == QDialog::Accepted) {
    int threshold = thresholdSpinBox->value();
    int strength = strengthSpinBox->value();

    m_statusBar->showMessage(tr("正在进行图片漂白处理..."));
    m_progressBar->setValue(0);
    m_progressBar->show();

    // 连接进度信号
    connect(m_imageProcessor, &ImageProcessor::processingProgress,
            m_progressBar, &QProgressBar::setValue);

    QImage processedImage = m_imageProcessor->bleachBackground(currentImage, threshold, strength);

    // 断开进度信号
    disconnect(m_imageProcessor, &ImageProcessor::processingProgress,
               m_progressBar, &QProgressBar::setValue);

    m_progressBar->hide();

    if (!processedImage.isNull()) {
      int currentIndex = m_documentManager->getCurrentPageIndex();
      m_documentManager->removePage(currentIndex);
      m_documentManager->insertImage(currentIndex, processedImage);
      displayImage(processedImage);

      m_statusBar->showMessage(tr("图片漂白处理完成"), 3000);
      
      // 如果启用了自动保存，保存处理后的图像
      if (m_autoSaveCheck->isChecked()) {
        performAutoSave(processedImage);
      }
    } else {
      m_statusBar->showMessage(tr("图片漂白处理失败"), 3000);
      QMessageBox::warning(this, tr("错误"), tr("图片漂白处理失败，请检查图像格式"));
    }
  }

  bleachDialog->deleteLater();
}

void MainWindow::onAdjustBrightnessContrastDialogClicked() {
  QImage currentImage = m_documentManager->getCurrentImage();
  if (currentImage.isNull()) {
    QMessageBox::information(this, tr("提示"), tr("请先选择要处理的图像"));
    return;
  }

  // 保存到撤销栈
  saveToUndoStack(currentImage);
  
  // 保存原始图像用于实时预览
  QImage originalImage = currentImage.copy();

  // 创建亮度对比度调整对话框
  QDialog dialog(this);
  dialog.setWindowTitle(tr("亮度、对比度调整（实时预览）"));
  dialog.setMinimumWidth(400);
  
  QVBoxLayout *layout = new QVBoxLayout(&dialog);
  
  QLabel *titleLabel = new QLabel(tr("<b>调整图像的亮度和对比度</b>"), &dialog);
  layout->addWidget(titleLabel);
  
  // 调整参数组
  QGroupBox *adjustGroup = new QGroupBox(tr("调整参数（拖动滑块实时预览）"), &dialog);
  QFormLayout *formLayout = new QFormLayout(adjustGroup);
  
  // 亮度滑块
  QSlider *brightnessSlider = new QSlider(Qt::Horizontal, adjustGroup);
  brightnessSlider->setRange(-100, 100);
  brightnessSlider->setValue(0);
  brightnessSlider->setTickInterval(20);
  brightnessSlider->setTickPosition(QSlider::TicksBelow);
  
  QLabel *brightnessValueLabel = new QLabel(tr("0"), adjustGroup);
  QHBoxLayout *brightnessLayout = new QHBoxLayout();
  brightnessLayout->addWidget(brightnessSlider);
  brightnessLayout->addWidget(brightnessValueLabel);
  
  formLayout->addRow(tr("亮度 (-100 到 100):"), brightnessLayout);
  
  // 对比度滑块
  QSlider *contrastSlider = new QSlider(Qt::Horizontal, adjustGroup);
  contrastSlider->setRange(-100, 100);
  contrastSlider->setValue(0);
  contrastSlider->setTickInterval(20);
  contrastSlider->setTickPosition(QSlider::TicksBelow);
  
  QLabel *contrastValueLabel = new QLabel(tr("0"), adjustGroup);
  QHBoxLayout *contrastLayout = new QHBoxLayout();
  contrastLayout->addWidget(contrastSlider);
  contrastLayout->addWidget(contrastValueLabel);
  
  formLayout->addRow(tr("对比度 (-100 到 100):"), contrastLayout);
  
  layout->addWidget(adjustGroup);
  
  // 提示信息
  QLabel *tipLabel = new QLabel(
      tr("\n<b>使用说明：</b>\n"
         "• 拖动滑块即可实时预览效果\n"
         "• 亮度：正值变亮，负值变暗\n"
         "• 对比度：正值增强，负值降低\n"
         "• 点击确定应用，取消恢复原图"), &dialog);
  tipLabel->setWordWrap(true);
  tipLabel->setStyleSheet("color: #666; font-size: 11px; padding: 10px; background-color: #f0f0f0; border-radius: 5px;");
  layout->addWidget(tipLabel);
  
  // 实时预览函数
  auto updatePreview = [this, &originalImage, brightnessSlider, contrastSlider]() {
    int brightness = brightnessSlider->value();
    int contrast = contrastSlider->value();
    
    // 断开进度信号，避免干扰
    disconnect(m_imageProcessor, &ImageProcessor::processingProgress, this, &MainWindow::onScanProgress);
    
    // 执行调整
    QImage adjustedImage = m_imageProcessor->adjustBrightnessContrast(originalImage, brightness, contrast);
    
    if (!adjustedImage.isNull()) {
      // 临时显示调整后的图像
      displayImage(adjustedImage);
    }
  };
  
  // 连接滑块值变化信号 - 实时预览
  connect(brightnessSlider, &QSlider::valueChanged, [brightnessValueLabel, updatePreview](int value) {
    brightnessValueLabel->setText(QString::number(value));
    updatePreview();
  });
  connect(contrastSlider, &QSlider::valueChanged, [contrastValueLabel, updatePreview](int value) {
    contrastValueLabel->setText(QString::number(value));
    updatePreview();
  });
  
  QDialogButtonBox *buttonBox = new QDialogButtonBox(
      QDialogButtonBox::Ok | QDialogButtonBox::Cancel, &dialog);
  connect(buttonBox, &QDialogButtonBox::accepted, &dialog, &QDialog::accept);
  connect(buttonBox, &QDialogButtonBox::rejected, &dialog, &QDialog::reject);
  layout->addWidget(buttonBox);
  
  int result = dialog.exec();
  
  if (result == QDialog::Accepted) {
    int brightness = brightnessSlider->value();
    int contrast = contrastSlider->value();
    
    if (brightness == 0 && contrast == 0) {
      // 恢复原图
      displayImage(originalImage);
      QMessageBox::information(this, tr("提示"), tr("未进行任何调整"));
      return;
    }
    
    // 最终应用调整
    QImage adjustedImage = m_imageProcessor->adjustBrightnessContrast(originalImage, brightness, contrast);
    
    if (!adjustedImage.isNull()) {
      // 替换当前图像
      int currentIndex = m_documentManager->getCurrentPageIndex();
      m_documentManager->removePage(currentIndex);
      m_documentManager->insertImage(currentIndex, adjustedImage);
      displayImage(adjustedImage);
      
      m_statusBar->showMessage(tr("亮度对比度调整完成"), 3000);
    }
  } else {
    // 取消 - 恢复原图
    displayImage(originalImage);
    m_statusBar->showMessage(tr("已取消调整"), 2000);
  }
}

void MainWindow::onAdjustHueSaturationClicked() {
  QImage currentImage = m_documentManager->getCurrentImage();
  if (currentImage.isNull()) {
    QMessageBox::information(this, tr("提示"), tr("请先选择要处理的图像"));
    return;
  }

  // 保存到撤销栈
  saveToUndoStack(currentImage);
  
  // 保存原始图像用于实时预览
  QImage originalImage = currentImage.copy();

  // 创建色度饱和度调整对话框
  QDialog dialog(this);
  dialog.setWindowTitle(tr("色度、饱和度调整（实时预览）"));
  dialog.setMinimumWidth(400);
  
  QVBoxLayout *layout = new QVBoxLayout(&dialog);
  
  QLabel *titleLabel = new QLabel(tr("<b>调整图像的色度和饱和度</b>"), &dialog);
  layout->addWidget(titleLabel);
  
  // 调整参数组
  QGroupBox *adjustGroup = new QGroupBox(tr("调整参数（拖动滑块实时预览）"), &dialog);
  QFormLayout *formLayout = new QFormLayout(adjustGroup);
  
  // 色度滑块
  QSlider *hueSlider = new QSlider(Qt::Horizontal, adjustGroup);
  hueSlider->setRange(-180, 180);
  hueSlider->setValue(0);
  hueSlider->setTickInterval(30);
  hueSlider->setTickPosition(QSlider::TicksBelow);
  
  QLabel *hueValueLabel = new QLabel(tr("0"), adjustGroup);
  QHBoxLayout *hueLayout = new QHBoxLayout();
  hueLayout->addWidget(hueSlider);
  hueLayout->addWidget(hueValueLabel);
  
  formLayout->addRow(tr("色度 (-180 到 180):"), hueLayout);
  
  // 饱和度滑块
  QSlider *saturationSlider = new QSlider(Qt::Horizontal, adjustGroup);
  saturationSlider->setRange(-100, 100);
  saturationSlider->setValue(0);
  saturationSlider->setTickInterval(20);
  saturationSlider->setTickPosition(QSlider::TicksBelow);
  
  QLabel *saturationValueLabel = new QLabel(tr("0"), adjustGroup);
  QHBoxLayout *saturationLayout = new QHBoxLayout();
  saturationLayout->addWidget(saturationSlider);
  saturationLayout->addWidget(saturationValueLabel);
  
  formLayout->addRow(tr("饱和度 (-100 到 100):"), saturationLayout);
  
  layout->addWidget(adjustGroup);
  
  // 提示信息
  QLabel *tipLabel = new QLabel(
      tr("\n<b>使用说明：</b>\n"
         "• 拖动滑块即可实时预览效果\n"
         "• 色度：正值暖色调，负值冷色调\n"
         "• 饱和度：正值更鲜艳，负值更灰暗\n"
         "• 点击确定应用，取消恢复原图"), &dialog);
  tipLabel->setWordWrap(true);
  tipLabel->setStyleSheet("color: #666; font-size: 11px; padding: 10px; background-color: #f0f0f0; border-radius: 5px;");
  layout->addWidget(tipLabel);
  
  // 实时预览函数
  auto updatePreview = [this, &originalImage, hueSlider, saturationSlider]() {
    int hue = hueSlider->value();
    int saturation = saturationSlider->value();
    
    // 断开进度信号，避免干扰
    disconnect(m_imageProcessor, &ImageProcessor::processingProgress, this, &MainWindow::onScanProgress);
    
    // 执行调整
    QImage adjustedImage = m_imageProcessor->adjustHueSaturation(originalImage, hue, saturation);
    
    if (!adjustedImage.isNull()) {
      // 临时显示调整后的图像
      displayImage(adjustedImage);
    }
  };
  
  // 连接滑块值变化信号 - 实时预览
  connect(hueSlider, &QSlider::valueChanged, [hueValueLabel, updatePreview](int value) {
    hueValueLabel->setText(QString::number(value));
    updatePreview();
  });
  connect(saturationSlider, &QSlider::valueChanged, [saturationValueLabel, updatePreview](int value) {
    saturationValueLabel->setText(QString::number(value));
    updatePreview();
  });
  
  QDialogButtonBox *buttonBox = new QDialogButtonBox(
      QDialogButtonBox::Ok | QDialogButtonBox::Cancel, &dialog);
  connect(buttonBox, &QDialogButtonBox::accepted, &dialog, &QDialog::accept);
  connect(buttonBox, &QDialogButtonBox::rejected, &dialog, &QDialog::reject);
  layout->addWidget(buttonBox);
  
  int result = dialog.exec();
  
  if (result == QDialog::Accepted) {
    int hue = hueSlider->value();
    int saturation = saturationSlider->value();
    
    if (hue == 0 && saturation == 0) {
      // 恢复原图
      displayImage(originalImage);
      QMessageBox::information(this, tr("提示"), tr("未进行任何调整"));
      return;
    }
    
    // 最终应用调整
    QImage adjustedImage = m_imageProcessor->adjustHueSaturation(originalImage, hue, saturation);
    
    if (!adjustedImage.isNull()) {
      // 替换当前图像
      int currentIndex = m_documentManager->getCurrentPageIndex();
      m_documentManager->removePage(currentIndex);
      m_documentManager->insertImage(currentIndex, adjustedImage);
      displayImage(adjustedImage);
      
      m_statusBar->showMessage(tr("色度饱和度调整完成"), 3000);
    }
  } else {
    // 取消 - 恢复原图
    displayImage(originalImage);
    m_statusBar->showMessage(tr("已取消调整"), 2000);
  }
}

void MainWindow::showPagesListContextMenu(const QPoint &pos) {
  QMenu contextMenu(tr("页面操作"), this);
  
  // 获取点击位置的项目
  QListWidgetItem *item = m_pagesList->itemAt(pos);
  bool hasItem = (item != nullptr);
  bool hasPages = (m_pagesList->count() > 0);
  
  // 添加图片菜单项
  QAction *addImageAction = contextMenu.addAction(QIcon(":/icons/scan.svg"), tr("添加图片"));
  addImageAction->setToolTip(tr("从文件添加新图片到文档末尾"));
  connect(addImageAction, &QAction::triggered, this, &MainWindow::onAddImageFromFile);
  
  if (hasItem) {
    // 插入图片菜单项（只在有选中项时显示）
    QAction *insertImageAction = contextMenu.addAction(QIcon(":/icons/scan.svg"), tr("插入图片"));
    insertImageAction->setToolTip(tr("在当前位置插入新图片"));
    connect(insertImageAction, &QAction::triggered, this, &MainWindow::onInsertImageFromFile);
    
    contextMenu.addSeparator();
    
    // 获取当前选中的索引
    int currentIndex = m_pagesList->currentRow();
    
    // 上移页面
    QAction *moveUpAction = contextMenu.addAction(QIcon(":/icons/rotate-left.svg"), tr("上移"));
    moveUpAction->setToolTip(tr("将此页面向上移动一位"));
    moveUpAction->setEnabled(currentIndex > 0);  // 第一页不能上移
    connect(moveUpAction, &QAction::triggered, this, &MainWindow::onMovePageUp);
    
    // 下移页面
    QAction *moveDownAction = contextMenu.addAction(QIcon(":/icons/rotate-right.svg"), tr("下移"));
    moveDownAction->setToolTip(tr("将此页面向下移动一位"));
    moveDownAction->setEnabled(currentIndex < m_pagesList->count() - 1);  // 最后一页不能下移
    connect(moveDownAction, &QAction::triggered, this, &MainWindow::onMovePageDown);
    
    contextMenu.addSeparator();
    
    // 拆分图片菜单
    QMenu *splitMenu = contextMenu.addMenu(QIcon(":/icons/layout-columns.svg"), tr("拆分图片"));
    splitMenu->setToolTip(tr("将图片拆分为多个部分"));
    
    QAction *splitHAction = splitMenu->addAction(QIcon(":/icons/layout-columns.svg"), tr("左右拆分"));
    splitHAction->setToolTip(tr("从中间左右拆分，适合左右排列的证件"));
    connect(splitHAction, &QAction::triggered, this, &MainWindow::onSplitImageHorizontal);
    
    QAction *splitVAction = splitMenu->addAction(QIcon(":/icons/layout.svg"), tr("上下拆分"));
    splitVAction->setToolTip(tr("从中间上下拆分，适合上下排列的证件"));
    connect(splitVAction, &QAction::triggered, this, &MainWindow::onSplitImageVertical);
    
    contextMenu.addSeparator();
    
    // 删除当前页面
    QAction *deletePageAction = contextMenu.addAction(QIcon(":/icons/delete.svg"), tr("删除此页"));
    deletePageAction->setToolTip(tr("删除当前选中的页面"));
    connect(deletePageAction, &QAction::triggered, this, &MainWindow::onDeleteCurrentPage);
  }
  
  if (hasPages) {
    contextMenu.addSeparator();
    
    // 清空所有页面
    QAction *clearAllAction = contextMenu.addAction(QIcon(":/icons/clear-all.svg"), tr("清空所有页面"));
    clearAllAction->setToolTip(tr("删除所有页面，创建新文档"));
    connect(clearAllAction, &QAction::triggered, this, &MainWindow::onClearAllPages);
  }
  
  // 如果没有任何页面，显示提示信息
  if (!hasPages) {
    QAction *tipAction = contextMenu.addAction(tr("暂无页面"));
    tipAction->setEnabled(false);
    tipAction->setToolTip(tr("添加图片开始创建文档"));
  }
  
  // 显示菜单
  contextMenu.exec(m_pagesList->mapToGlobal(pos));
}

void MainWindow::onAddImageFromFile() {
  QString filePath = QFileDialog::getOpenFileName(
      this, tr("添加图片"),
      QStandardPaths::writableLocation(QStandardPaths::PicturesLocation),
      tr("图像文件 (*.png *.jpg *.jpeg *.bmp *.tiff *.tif);;所有文件 (*.*)"));

  if (!filePath.isEmpty()) {
    QImage image(filePath);
    if (!image.isNull()) {
      m_documentManager->addImage(image);
      m_statusBar->showMessage(tr("图片已添加: %1").arg(QFileInfo(filePath).fileName()), 3000);
      
      // 如果启用了自动保存，保存新添加的图像
      if (m_autoSaveCheck->isChecked()) {
        performAutoSave(image);
      }
    } else {
      QMessageBox::warning(this, tr("错误"), 
                          tr("无法加载图片文件: %1\n请检查文件格式是否支持。").arg(filePath));
    }
  }
}

void MainWindow::onInsertImageFromFile() {
  int currentIndex = m_pagesList->currentRow();
  if (currentIndex < 0) {
    // 如果没有选中项，添加到末尾
    onAddImageFromFile();
    return;
  }
  
  QString filePath = QFileDialog::getOpenFileName(
      this, tr("插入图片"),
      QStandardPaths::writableLocation(QStandardPaths::PicturesLocation),
      tr("图像文件 (*.png *.jpg *.jpeg *.bmp *.tiff *.tif);;所有文件 (*.*)"));

  if (!filePath.isEmpty()) {
    QImage image(filePath);
    if (!image.isNull()) {
      m_documentManager->insertImage(currentIndex, image);
      m_statusBar->showMessage(tr("图片已插入: %1").arg(QFileInfo(filePath).fileName()), 3000);
      
      // 如果启用了自动保存，保存插入的图像
      if (m_autoSaveCheck->isChecked()) {
        performAutoSave(image);
      }
    } else {
      QMessageBox::warning(this, tr("错误"), 
                          tr("无法加载图片文件: %1\n请检查文件格式是否支持。").arg(filePath));
    }
  }
}

void MainWindow::onDeleteCurrentPage() {
  int currentIndex = m_pagesList->currentRow();
  if (currentIndex >= 0) {
    QString pageTitle = m_pagesList->item(currentIndex)->text();
    
    int ret = QMessageBox::question(this, tr("确认删除"),
                                   tr("确定要删除 \"%1\" 吗？\n此操作无法撤销。").arg(pageTitle),
                                   QMessageBox::Yes | QMessageBox::No,
                                   QMessageBox::No);
    
    if (ret == QMessageBox::Yes) {
      m_documentManager->removePage(currentIndex);
      m_statusBar->showMessage(tr("页面已删除: %1").arg(pageTitle), 3000);
    }
  }
}

void MainWindow::onClearAllPages() {
  if (m_pagesList->count() > 0) {
    int ret = QMessageBox::question(this, tr("确认清空"),
                                   tr("确定要删除所有页面吗？\n此操作将清空当前文档，无法撤销。"),
                                   QMessageBox::Yes | QMessageBox::No,
                                   QMessageBox::No);
    
    if (ret == QMessageBox::Yes) {
      m_documentManager->newDocument();
      m_currentImage = QImage();
      displayImage(m_currentImage);
      updateWindowTitle();
      updateActions();
      m_statusBar->showMessage(tr("所有页面已清空"), 3000);
    }
  }
}

void MainWindow::onMovePageUp() {
  int currentIndex = m_pagesList->currentRow();
  if (currentIndex > 0) {
    m_documentManager->movePageUp(currentIndex);
    m_statusBar->showMessage(tr("页面已上移"), 2000);
  }
}

void MainWindow::onMovePageDown() {
  int currentIndex = m_pagesList->currentRow();
  if (currentIndex >= 0 && currentIndex < m_pagesList->count() - 1) {
    m_documentManager->movePageDown(currentIndex);
    m_statusBar->showMessage(tr("页面已下移"), 2000);
  }
}

void MainWindow::onSplitImageHorizontal() {
  int currentIndex = m_pagesList->currentRow();
  if (currentIndex < 0) {
    return;
  }
  
  // 获取当前图片
  QImage sourceImage = m_documentManager->getPage(currentIndex);
  if (sourceImage.isNull()) {
    QMessageBox::warning(this, tr("错误"), tr("无法获取当前图片"));
    return;
  }
  
  // 显示进度
  m_statusBar->showMessage(tr("正在左右拆分图片..."));
  m_progressBar->setValue(0);
  m_progressBar->show();
  
  // 连接进度信号
  connect(m_imageProcessor, &ImageProcessor::processingProgress,
          m_progressBar, &QProgressBar::setValue);
  
  // 执行拆分
  QVector<QImage> splitImages = m_imageProcessor->splitImageHorizontal(sourceImage, 240);
  
  // 断开进度信号
  disconnect(m_imageProcessor, &ImageProcessor::processingProgress,
             m_progressBar, &QProgressBar::setValue);
  
  m_progressBar->hide();
  
  if (splitImages.size() != 2) {
    m_statusBar->showMessage(tr("拆分失败"), 3000);
    QMessageBox::warning(this, tr("拆分失败"), 
                        tr("无法检测到2个独立的内容区域。\n\n"
                           "可能原因：\n"
                           "• 图片中只有1个内容\n"
                           "• 图片中有3个或更多内容\n"
                           "• 内容重叠或连接\n\n"
                           "建议：\n"
                           "• 确保图片中有2个明显分离的内容\n"
                           "• 内容应该左右分布"));
    return;
  }
  
  // 只保留左侧部分（第一部分），自动删除右侧部分
  QImage leftImage = splitImages[0];
  
  // 询问是否替换原图
  int ret = QMessageBox::question(this, tr("左右拆分成功"),
                                  tr("已提取左侧部分，右侧部分已自动丢弃。\n\n"
                                     "是否用左侧部分替换原图？\n"
                                     "• 是：替换原图\n"
                                     "• 否：在原图后添加左侧部分"),
                                  QMessageBox::Yes | QMessageBox::No,
                                  QMessageBox::Yes);
  
  if (ret == QMessageBox::Yes) {
    // 替换原图
    m_documentManager->removePage(currentIndex);
    m_documentManager->insertImage(currentIndex, leftImage);
    m_statusBar->showMessage(tr("左右拆分完成，已保留左侧部分"), 3000);
  } else {
    // 在原图后添加
    m_documentManager->addImage(leftImage);
    m_statusBar->showMessage(tr("左右拆分完成，左侧部分已添加"), 3000);
  }
  
  qDebug() << "左右拆分完成，保留左侧部分，丢弃右侧部分";
}

void MainWindow::onSplitImageVertical() {
  int currentIndex = m_pagesList->currentRow();
  if (currentIndex < 0) {
    return;
  }
  
  // 获取当前图片
  QImage sourceImage = m_documentManager->getPage(currentIndex);
  if (sourceImage.isNull()) {
    QMessageBox::warning(this, tr("错误"), tr("无法获取当前图片"));
    return;
  }
  
  // 显示进度
  m_statusBar->showMessage(tr("正在上下拆分图片..."));
  m_progressBar->setValue(0);
  m_progressBar->show();
  
  // 连接进度信号
  connect(m_imageProcessor, &ImageProcessor::processingProgress,
          m_progressBar, &QProgressBar::setValue);
  
  // 执行拆分
  QVector<QImage> splitImages = m_imageProcessor->splitImageVertical(sourceImage, 240);
  
  // 断开进度信号
  disconnect(m_imageProcessor, &ImageProcessor::processingProgress,
             m_progressBar, &QProgressBar::setValue);
  
  m_progressBar->hide();
  
  if (splitImages.size() != 2) {
    m_statusBar->showMessage(tr("拆分失败"), 3000);
    QMessageBox::warning(this, tr("拆分失败"), 
                        tr("无法检测到2个独立的内容区域。\n\n"
                           "可能原因：\n"
                           "• 图片中只有1个内容\n"
                           "• 图片中有3个或更多内容\n"
                           "• 内容重叠或连接\n\n"
                           "建议：\n"
                           "• 确保图片中有2个明显分离的内容\n"
                           "• 内容应该上下分布"));
    return;
  }
  
  // 只保留上部分（第一部分），自动删除下部分
  QImage topImage = splitImages[0];
  
  // 询问是否替换原图
  int ret = QMessageBox::question(this, tr("上下拆分成功"),
                                  tr("已提取上部分，下部分已自动丢弃。\n\n"
                                     "是否用上部分替换原图？\n"
                                     "• 是：替换原图\n"
                                     "• 否：在原图后添加上部分"),
                                  QMessageBox::Yes | QMessageBox::No,
                                  QMessageBox::Yes);
  
  if (ret == QMessageBox::Yes) {
    // 替换原图
    m_documentManager->removePage(currentIndex);
    m_documentManager->insertImage(currentIndex, topImage);
    m_statusBar->showMessage(tr("上下拆分完成，已保留上部分"), 3000);
  } else {
    // 在原图后添加
    m_documentManager->addImage(topImage);
    m_statusBar->showMessage(tr("上下拆分完成，上部分已添加"), 3000);
  }
  
  qDebug() << "上下拆分完成，保留上部分，丢弃下部分";
}

void MainWindow::onMergeImagesClicked() {
  // 获取选中的页面
  QList<QListWidgetItem*> selectedItems = m_pagesList->selectedItems();
  
  if (selectedItems.count() != 2) {
    QMessageBox::information(this, tr("提示"), 
                            tr("请选择2张图片进行合并。\n\n"
                               "提示：按住 Shift 键可以选择多张图片。\n\n"
                               "如需拆分单张图片，请使用右键菜单的拆分功能。"));
    return;
  }
  
  // 创建合并设置对话框
  QDialog dialog(this);
  dialog.setWindowTitle(tr("合并图片"));
  dialog.setMinimumWidth(400);
  
  QVBoxLayout *layout = new QVBoxLayout(&dialog);
  
  // 标题
  QLabel *titleLabel = new QLabel(tr("<b>合并图片</b>"), &dialog);
  layout->addWidget(titleLabel);
  
  // 信息
  QLabel *infoLabel = new QLabel(tr("已选择 2 张图片"), &dialog);
  infoLabel->setStyleSheet("color: #666; margin: 10px 0;");
  layout->addWidget(infoLabel);
  
  // 合并方式选择
  QGroupBox *modeGroup = new QGroupBox(tr("合并方式"), &dialog);
  QVBoxLayout *modeLayout = new QVBoxLayout(modeGroup);
  
  QRadioButton *horizontalRadio = new QRadioButton(tr("左右合并（适合身份证、驾驶证等横向证件）"), modeGroup);
  QRadioButton *verticalRadio = new QRadioButton(tr("上下合并（适合户口本、竖向证件或文档）"), modeGroup);
  horizontalRadio->setChecked(true);  // 默认左右合并
  
  modeLayout->addWidget(horizontalRadio);
  modeLayout->addWidget(verticalRadio);
  layout->addWidget(modeGroup);
  
  // 间距设置
  QGroupBox *spacingGroup = new QGroupBox(tr("图片间距"), &dialog);
  QHBoxLayout *spacingLayout = new QHBoxLayout(spacingGroup);
  
  QLabel *spacingLabel = new QLabel(tr("间距:"), spacingGroup);
  QSpinBox *spacingSpinBox = new QSpinBox(spacingGroup);
  spacingSpinBox->setRange(0, 100);
  spacingSpinBox->setValue(30);  // 默认30像素
  spacingSpinBox->setSuffix(tr(" 像素"));
  
  spacingLayout->addWidget(spacingLabel);
  spacingLayout->addWidget(spacingSpinBox);
  spacingLayout->addStretch();
  layout->addWidget(spacingGroup);
  
  // 选项
  QCheckBox *autoSplitCheckBox = new QCheckBox(tr("智能拆分图片（推荐）"), &dialog);
  autoSplitCheckBox->setChecked(true);  // 默认启用
  autoSplitCheckBox->setToolTip(tr("自动检测包含多个内容的图片，拆分后保留需要的部分"));
  layout->addWidget(autoSplitCheckBox);
  
  QCheckBox *autoTrimCheckBox = new QCheckBox(tr("自动裁剪空白边缘"), &dialog);
  autoTrimCheckBox->setChecked(true);  // 默认启用
  autoTrimCheckBox->setToolTip(tr("自动检测并裁剪图片周围的空白区域"));
  layout->addWidget(autoTrimCheckBox);
  
  QCheckBox *deleteOriginalCheckBox = new QCheckBox(tr("合并后删除原始图片"), &dialog);
  deleteOriginalCheckBox->setChecked(false);
  layout->addWidget(deleteOriginalCheckBox);
  
  // 说明
  QLabel *tipLabel = new QLabel(
      tr("<b>智能合并说明：</b><br>"
         "• 智能拆分：检测到包含多个内容的图片时，自动拆分并保留需要的部分<br>"
         "• 自动裁剪：去除图片周围的空白，提取实际内容<br>"
         "• 合并后的图片将添加到页面列表末尾<br>"
         "• 图片将自动居中对齐，背景为白色"), &dialog);
  tipLabel->setWordWrap(true);
  tipLabel->setStyleSheet("color: #444; font-size: 11px; padding: 10px; "
                         "background-color: #f5f5f5; border-radius: 5px; "
                         "margin-top: 10px;");
  layout->addWidget(tipLabel);
  
  // 按钮
  QDialogButtonBox *buttonBox = new QDialogButtonBox(
      QDialogButtonBox::Ok | QDialogButtonBox::Cancel, &dialog);
  connect(buttonBox, &QDialogButtonBox::accepted, &dialog, &QDialog::accept);
  connect(buttonBox, &QDialogButtonBox::rejected, &dialog, &QDialog::reject);
  layout->addWidget(buttonBox);
  
  // 显示对话框
  if (dialog.exec() == QDialog::Accepted) {
    // 获取选中页面的索引
    QList<int> selectedIndices;
    for (QListWidgetItem *item : selectedItems) {
      selectedIndices.append(m_pagesList->row(item));
    }
    std::sort(selectedIndices.begin(), selectedIndices.end());
    
    // 获取参数
    int spacing = spacingSpinBox->value();
    bool isHorizontal = horizontalRadio->isChecked();
    bool autoSplit = autoSplitCheckBox->isChecked();
    bool autoTrim = autoTrimCheckBox->isChecked();
    
    // 显示进度
    m_statusBar->showMessage(tr("正在处理图片..."));
    m_progressBar->setValue(0);
    m_progressBar->show();
    
    // 连接进度信号
    connect(m_imageProcessor, &ImageProcessor::processingProgress,
            m_progressBar, &QProgressBar::setValue);
    
    // 获取两张图片
    QImage sourceImage1 = m_documentManager->getPage(selectedIndices[0]);
    QImage sourceImage2 = m_documentManager->getPage(selectedIndices[1]);
    
    if (sourceImage1.isNull() || sourceImage2.isNull()) {
      QMessageBox::warning(this, tr("错误"), tr("无法获取选中的图片"));
      disconnect(m_imageProcessor, &ImageProcessor::processingProgress,
                 m_progressBar, &QProgressBar::setValue);
      m_progressBar->hide();
      return;
    }
    
    QImage image1, image2;
    bool didAutoSplit = false;
    
    // 智能拆分：检测图片中是否包含多个内容区域
    if (autoSplit) {
      // 检测图片1是否包含多个内容区域
      QVector<QRect> contentBounds1;
      if (isHorizontal) {
        contentBounds1 = m_imageProcessor->detectMultipleContentBounds(sourceImage1, true, 240);
      } else {
        contentBounds1 = m_imageProcessor->detectMultipleContentBounds(sourceImage1, false, 240);
      }
      bool needSplit1 = (contentBounds1.size() >= 2);
      
      // 检测图片2是否包含多个内容区域
      QVector<QRect> contentBounds2;
      if (isHorizontal) {
        contentBounds2 = m_imageProcessor->detectMultipleContentBounds(sourceImage2, true, 240);
      } else {
        contentBounds2 = m_imageProcessor->detectMultipleContentBounds(sourceImage2, false, 240);
      }
      bool needSplit2 = (contentBounds2.size() >= 2);
      
      if (needSplit1 || needSplit2) {
        didAutoSplit = true;
        qDebug() << "检测到包含多个内容区域的图片，启动智能拆分模式";
        qDebug() << "图片1内容区域数:" << contentBounds1.size() << "图片2内容区域数:" << contentBounds2.size();
        m_statusBar->showMessage(tr("检测到多个内容区域，正在智能拆分..."));
        
        // 处理图片1
        if (needSplit1) {
          m_statusBar->showMessage(tr("正在拆分第一张图片..."));
          QVector<QImage> splitImages1;
          
          if (isHorizontal) {
            // 左右合并模式：左右拆分图片
            splitImages1 = m_imageProcessor->splitImageHorizontal(sourceImage1, 240);
          } else {
            // 上下合并模式：上下拆分图片
            splitImages1 = m_imageProcessor->splitImageVertical(sourceImage1, 240);
          }
          
          if (splitImages1.size() >= 2) {
            // 保留第一部分（上部分或左部分）
            image1 = splitImages1[0];
            qDebug() << "图片1拆分成功，保留第一部分:" << image1.size();
          } else {
            // 拆分失败，使用原图
            qDebug() << "图片1拆分失败，使用原图";
            image1 = sourceImage1;
          }
        } else {
          image1 = sourceImage1;
        }
        
        // 处理图片2
        if (needSplit2) {
          m_statusBar->showMessage(tr("正在拆分第二张图片..."));
          QVector<QImage> splitImages2;
          
          if (isHorizontal) {
            // 左右合并模式：左右拆分图片
            splitImages2 = m_imageProcessor->splitImageHorizontal(sourceImage2, 240);
          } else {
            // 上下合并模式：上下拆分图片
            splitImages2 = m_imageProcessor->splitImageVertical(sourceImage2, 240);
          }
          
          if (splitImages2.size() >= 2) {
            // 保留第一部分（上部分或左部分）
            image2 = splitImages2[0];
            qDebug() << "图片2拆分成功，保留第一部分:" << image2.size();
          } else {
            // 拆分失败，使用原图
            qDebug() << "图片2拆分失败，使用原图";
            image2 = sourceImage2;
          }
        } else {
          image2 = sourceImage2;
        }
      } else {
        // 不是A4尺寸，直接使用原图
        image1 = sourceImage1;
        image2 = sourceImage2;
      }
    } else {
      // 未启用自动拆分，直接使用原图
      image1 = sourceImage1;
      image2 = sourceImage2;
    }
    
    // 自动裁剪空白边缘
    if (autoTrim) {
      QSize originalSize1 = image1.size();
      m_statusBar->showMessage(tr("正在裁剪第一张图片..."));
      image1 = m_imageProcessor->autoTrimWhitespace(image1, 240);
      qDebug() << "图片1裁剪: 原始" << originalSize1 << "→ 裁剪后" << image1.size();
      
      QSize originalSize2 = image2.size();
      m_statusBar->showMessage(tr("正在裁剪第二张图片..."));
      image2 = m_imageProcessor->autoTrimWhitespace(image2, 240);
      qDebug() << "图片2裁剪: 原始" << originalSize2 << "→ 裁剪后" << image2.size();
    }
    
    // 执行合并
    m_statusBar->showMessage(tr("正在合并图片..."));
    QImage mergedImage;
    if (isHorizontal) {
      mergedImage = m_imageProcessor->mergeImagesHorizontal(image1, image2, spacing);
    } else {
      mergedImage = m_imageProcessor->mergeImagesVertical(image1, image2, spacing);
    }
    
    // 断开进度信号
    disconnect(m_imageProcessor, &ImageProcessor::processingProgress,
               m_progressBar, &QProgressBar::setValue);
    
    m_progressBar->hide();
    
    if (!mergedImage.isNull()) {
      // 添加合并后的图片到文档
      m_documentManager->addImage(mergedImage);
      
      // 显示合并信息
      QString sizeInfo = QString("合并完成！尺寸: %1 × %2 像素")
                         .arg(mergedImage.width())
                         .arg(mergedImage.height());
      qDebug() << sizeInfo;
      
      // 如果选择了删除原始图片
      if (deleteOriginalCheckBox->isChecked()) {
        // 从后往前删除，避免索引变化
        for (int i = selectedIndices.count() - 1; i >= 0; --i) {
          m_documentManager->removePage(selectedIndices[i]);
        }
        m_statusBar->showMessage(tr("图片合并完成，原始图片已删除 - ") + sizeInfo, 5000);
      } else {
        m_statusBar->showMessage(sizeInfo, 5000);
      }
      
      // 显示合并后的图片
      displayImage(mergedImage);
      
      // 弹出提示框显示详细信息
      QString infoMessage = tr("图片合并完成！\n\n"
                              "合并后尺寸: %1 × %2 像素\n"
                              "文件大小: 约 %3 MB\n\n")
                              .arg(mergedImage.width())
                              .arg(mergedImage.height())
                              .arg(mergedImage.sizeInBytes() / 1024.0 / 1024.0, 0, 'f', 2);
      
      if (didAutoSplit) {
        infoMessage += tr("✓ 已智能拆分图片并保留需要的部分\n");
      }
      if (autoTrim) {
        infoMessage += tr("✓ 已自动裁剪空白边缘\n");
      }
      
      infoMessage += tr("\n注意：图片在窗口中会自动缩放显示，\n"
                       "但实际尺寸未改变。保存后可查看原始尺寸。");
      
      QMessageBox::information(this, tr("合并成功"), infoMessage);
    } else {
      m_statusBar->showMessage(tr("图片合并失败"), 3000);
      QMessageBox::warning(this, tr("错误"), tr("图片合并失败，请检查图片格式"));
    }
  }
}

void MainWindow::updatePaperSourceOptions() {
  if (!m_scannerController) {
    qDebug() << "扫描仪控制器未初始化，无法更新纸张来源选项";
    return;
  }

  QString currentScanner = m_scannerController->getCurrentScanner();
  qDebug() << "正在为设备更新扫描参数:" << currentScanner;

  // 保存当前选择（如果有的话）
  QString currentPaperSource = m_paperSourceComboBox->currentText();
  QString currentDuplexMode = m_duplexModeComboBox->currentData().toString();
  if (currentDuplexMode.isEmpty()) {
    currentDuplexMode = "Simplex"; // 默认单面
  }

  // 临时断开信号，避免在更新时触发不必要的事件
  disconnect(m_paperSourceComboBox, QOverload<int>::of(&QComboBox::currentIndexChanged),
             nullptr, nullptr);
  disconnect(m_duplexModeComboBox, QOverload<int>::of(&QComboBox::currentIndexChanged),
             nullptr, nullptr);

  // 清空并重新填充选项
  m_paperSourceComboBox->clear();
  m_duplexModeComboBox->clear();

  // 获取当前扫描仪支持的纸张来源
  QStringList availableSources = m_scannerController->getAvailablePaperSources();
  bool supportsADF = m_scannerController->supportsADF();
  bool supportsDuplex = m_scannerController->supportsDuplex();

  qDebug() << "设备能力 - ADF支持:" << supportsADF << "双面支持:" << supportsDuplex;
  qDebug() << "可用纸张来源:" << availableSources;

  // 根据设备能力添加纸张来源选项
  if (availableSources.isEmpty() || !supportsADF) {
    // 仅支持平板扫描的设备
    m_paperSourceComboBox->addItem(tr("平板"), "Flatbed");
    m_paperSourceComboBox->setToolTip(tr("当前设备仅支持平板扫描"));
  } else {
    // 支持多种纸张来源的设备
    for (const QString &source : availableSources) {
      if (source.contains("Flatbed", Qt::CaseInsensitive)) {
        m_paperSourceComboBox->addItem(tr("平板"), source);
      } else if (source.contains("ADF", Qt::CaseInsensitive) ||
                 source.contains("Feeder", Qt::CaseInsensitive) ||
                 source.contains("Automatic", Qt::CaseInsensitive)) {
        if (source.contains("Duplex", Qt::CaseInsensitive)) {
          m_paperSourceComboBox->addItem(tr("双面自动进纸器"), source);
        } else {
          m_paperSourceComboBox->addItem(tr("自动进纸器"), source);
        }
      } else {
        // 添加原始名称作为备选
        m_paperSourceComboBox->addItem(source, source);
      }
    }
    m_paperSourceComboBox->setToolTip(tr("选择纸张来源：平板扫描或自动进纸器"));
  }

  // 根据设备能力设置双面扫描选项
  if (supportsDuplex) {
    QStringList availableDuplexModes = m_scannerController->getAvailableDuplexModes();
    qDebug() << "可用双面模式:" << availableDuplexModes;
    
    if (!availableDuplexModes.isEmpty()) {
      for (const QString &mode : availableDuplexModes) {
        if (mode.contains("Simplex", Qt::CaseInsensitive) ||
            mode.contains("False", Qt::CaseInsensitive) ||
            mode.contains("Single", Qt::CaseInsensitive) ||
            mode == "0") {
          m_duplexModeComboBox->addItem(tr("单面"), mode);
        } else if (mode.contains("Duplex", Qt::CaseInsensitive) ||
                   mode.contains("True", Qt::CaseInsensitive) ||
                   mode.contains("Both", Qt::CaseInsensitive) ||
                   mode == "1") {
          m_duplexModeComboBox->addItem(tr("双面"), mode);
        } else {
          // 添加原始选项名称
          m_duplexModeComboBox->addItem(mode, mode);
        }
      }
    } else {
      // 默认双面选项
      m_duplexModeComboBox->addItem(tr("单面"), "Simplex");
      m_duplexModeComboBox->addItem(tr("双面"), "Duplex");
    }
    m_duplexModeComboBox->setEnabled(true);
    m_duplexModeComboBox->setToolTip(tr("选择扫描模式：单面或双面"));
  } else {
    // 不支持双面扫描的设备
    m_duplexModeComboBox->addItem(tr("单面"), "Simplex");
    m_duplexModeComboBox->setEnabled(false);
    m_duplexModeComboBox->setToolTip(tr("当前设备不支持双面扫描"));
  }

  // 智能恢复之前的选择或设置合理的默认值
  bool paperSourceRestored = false;
  bool duplexModeRestored = false;

  // 尝试恢复纸张来源选择
  if (!currentPaperSource.isEmpty()) {
    int index = m_paperSourceComboBox->findText(currentPaperSource);
    if (index >= 0) {
      m_paperSourceComboBox->setCurrentIndex(index);
      paperSourceRestored = true;
    }
  }

  // 如果无法恢复，设置智能默认值
  if (!paperSourceRestored) {
    if (supportsADF && m_paperSourceComboBox->count() > 1) {
      // 如果支持ADF，默认选择平板（通常是第一个选项）
      m_paperSourceComboBox->setCurrentIndex(0);
    } else {
      // 仅支持平板的设备
      m_paperSourceComboBox->setCurrentIndex(0);
    }
  }

  // 尝试恢复双面扫描模式
  if (!currentDuplexMode.isEmpty()) {
    for (int i = 0; i < m_duplexModeComboBox->count(); ++i) {
      if (m_duplexModeComboBox->itemData(i).toString() == currentDuplexMode) {
        m_duplexModeComboBox->setCurrentIndex(i);
        duplexModeRestored = true;
        break;
      }
    }
  }

  // 如果无法恢复，默认选择单面
  if (!duplexModeRestored) {
    m_duplexModeComboBox->setCurrentIndex(0); // 通常单面是第一个选项
  }

  // 重新连接信号
  connect(m_paperSourceComboBox, QOverload<int>::of(&QComboBox::currentIndexChanged),
          this, [this](int index) {
            if (m_scannerController && index >= 0) {
              QString source = m_paperSourceComboBox->itemData(index).toString();
              m_scannerController->setPaperSource(source);
              qDebug() << "纸张来源已更改为:" << source;
            }
          });

  connect(m_duplexModeComboBox, QOverload<int>::of(&QComboBox::currentIndexChanged),
          this, [this](int index) {
            if (m_scannerController && index >= 0) {
              QString mode = m_duplexModeComboBox->itemData(index).toString();
              m_scannerController->setDuplexMode(mode);
              qDebug() << "双面模式已更改为:" << mode;
            }
          });

  qDebug() << "扫描参数更新完成 - 纸张来源:" << m_paperSourceComboBox->currentText() 
           << "双面模式:" << m_duplexModeComboBox->currentText();
}

void MainWindow::onADFStatusChanged(bool hasDocument) {
  if (hasDocument) {
    m_statusBar->showMessage(tr("检测到输稿器中有文档，准备扫描"), 3000);
  } else {
    m_statusBar->showMessage(tr("未检测到输稿器中的文档（可能是传感器问题）"),
                             5000);

    // 如果正在扫描过程中检测到无纸张，询问用户是否继续
    if (m_isScanning) {
      onADFEmptyWarning();
    }
  }
}

void MainWindow::onADFEmptyWarning() {
  QMessageBox msgBox(this);
  msgBox.setWindowTitle(tr("输稿器纸张检测"));
  msgBox.setText(tr("程序未检测到输稿器中的纸张。"));
  msgBox.setInformativeText(tr("这可能是因为：\n"
                               "1. 输稿器中确实没有纸张\n"
                               "2. 纸张传感器检测不准确\n"
                               "3. 扫描仪不支持纸张检测\n\n"
                               "请确认输稿器中已放置纸张，然后选择操作："));
  msgBox.setIcon(QMessageBox::Question);

  QPushButton *continueBtn =
      msgBox.addButton(tr("继续扫描"), QMessageBox::ActionRole);
  QPushButton *cancelBtn =
      msgBox.addButton(tr("取消"), QMessageBox::RejectRole);

  msgBox.setDefaultButton(continueBtn);
  msgBox.exec();

  if (msgBox.clickedButton() == continueBtn) {
    // 用户选择继续扫描，强制启动扫描
    forceScanWithADF();
  } else {
    // 用户选择取消，停止扫描
    qDebug() << "用户取消ADF扫描";
    if (m_scannerController) {
      m_scannerController->cancelScan();
    }
    m_isScanning = false;
    m_isMultiPageScanning = false;
    m_progressBar->hide();
    updateActions();
    m_statusBar->showMessage(tr("扫描已取消"), 3000);
  }
}

void MainWindow::forceScanWithADF() {
  if (!m_scannerController) {
    return;
  }

  // 获取当前扫描参数
  int resolution = m_resolutionComboBox->currentText().toInt();
  QString colorMode = m_colorModeComboBox->currentText();
  QString paperSource = m_paperSourceComboBox->currentText();
  // 双面扫描模式现在通过纸张来源自动控制，提供默认值
  QString duplexMode = "Simplex"; // 默认单面，双面自动进纸器会自动覆盖为Duplex

  // 直接启动扫描工作线程，跳过纸张检测
  QString deviceName;
  QString currentScanner = m_scannerController->getCurrentScanner();
  int startPos = currentScanner.lastIndexOf('(');
  int endPos = currentScanner.lastIndexOf(')');
  if (startPos >= 0 && endPos > startPos) {
    deviceName = currentScanner.mid(startPos + 1, endPos - startPos - 1);
  } else {
    deviceName = currentScanner;
  }

  if (!deviceName.isEmpty()) {
    m_isScanning = true;
    updateActions();
    m_statusBar->showMessage(tr("强制启动ADF扫描..."));
    m_progressBar->setValue(0);
    m_progressBar->show();

    // 直接发送扫描信号，跳过纸张检测
    emit m_scannerController->startScanWorker(deviceName, colorMode, resolution,
                                              false, paperSource, duplexMode, 215.9, 279.4);
  }
}

void MainWindow::onMultiPageScanStarted() {
  m_isMultiPageScanning = true;
  m_multiPageImages.clear();
  m_statusBar->showMessage(tr("开始多页扫描..."));
  qDebug() << "多页扫描开始";
}

void MainWindow::onMultiPageScanFinished(int totalPages) {
  m_isScanning = false;
  m_progressBar->hide();
  m_isMultiPageScanning = false;

  m_statusBar->showMessage(tr("多页扫描完成，共扫描 %1 页").arg(totalPages),
                           5000);
  qDebug() << "多页扫描完成，总页数:" << totalPages;

  // 现在执行自动保存（如果启用）
  if (m_autoSaveCheck && m_autoSaveCheck->isChecked() &&
      !m_multiPageImages.isEmpty()) {
    performMultiPageAutoSave();
  }

  // 清空暂存的图像
  m_multiPageImages.clear();
  updateActions();
}

void MainWindow::performAutoSave(const QImage &image) {
  if (!m_autoSaveCheck || !m_autoSaveCheck->isChecked()) {
    return;
  }

  QString ext = m_formatCombo->currentData().toString();
  QString path = generateFileName(ext);

  // 确保目录存在
  QDir().mkpath(QFileInfo(path).absolutePath());

  if (saveImage(image, path)) {
    // 显示保存成功的消息对话框
    QMessageBox::information(this, tr("自动保存成功"),
                             tr("扫描结果已自动保存到:\n%1").arg(path));

    // 在状态栏显示更详细的保存信息
    QString format = m_formatCombo->currentText();
    QString message = tr("已自动保存为 %1 格式到: %2")
                          .arg(format)
                          .arg(QFileInfo(path).fileName());
    m_statusBar->showMessage(message, 5000);

    // 更新自动保存计数器显示
    updateAutoSaveCounter();
  } else {
    QMessageBox::warning(
        this, tr("自动保存失败"),
        tr("无法自动保存扫描结果到:\n%1\n请检查路径和权限设置").arg(path));
  }
}

void MainWindow::performMultiPageAutoSave() {
  QString ext = m_formatCombo->currentData().toString();

  if (ext == "pdf") {
    // 保存为单个PDF文件
    QString path = generateFileName(ext);
    QDir().mkpath(QFileInfo(path).absolutePath());

    if (saveMultiPageAsPDF(m_multiPageImages, path)) {
      QMessageBox::information(this, tr("多页自动保存成功"),
                               tr("多页文档已自动保存为PDF:\n%1").arg(path));
      m_statusBar->showMessage(tr("已自动保存 %1 页为PDF: %2")
                                   .arg(m_multiPageImages.size())
                                   .arg(QFileInfo(path).fileName()),
                               5000);
    } else {
      QMessageBox::warning(this, tr("多页自动保存失败"),
                           tr("无法保存多页PDF到:\n%1").arg(path));
    }
  } else {
    // 保存为多个图像文件
    int savedCount = 0;
    for (int i = 0; i < m_multiPageImages.size(); ++i) {
      QString path = generateFileNameWithIndex(ext, i + 1);
      QDir().mkpath(QFileInfo(path).absolutePath());

      if (saveImage(m_multiPageImages[i], path)) {
        savedCount++;
      }
    }

    if (savedCount > 0) {
      QMessageBox::information(this, tr("多页自动保存成功"),
                               tr("已自动保存 %1 个图像文件").arg(savedCount));
      m_statusBar->showMessage(
          tr("已自动保存 %1 页为 %2 格式").arg(savedCount).arg(ext.toUpper()),
          5000);

      // 更新保存计数器
      m_saveCounter += savedCount - 1; // -1因为generateFileName已经增加了一次
      updateAutoSaveCounter();
    } else {
      QMessageBox::warning(this, tr("多页自动保存失败"),
                           tr("无法保存任何图像文件"));
    }
  }
}

QString MainWindow::generateFileNameWithIndex(const QString &ext, int index) {
  QDateTime now = QDateTime::currentDateTime();
  // 使用年月日时分秒格式：yyyyMMdd_HHmmss，加上页码
  QString dateTimeStr = now.toString("yyyyMMdd_HHmmss");

  return QString("%1/%2_p%3.%4")
      .arg(m_pathEdit->text())
      .arg(dateTimeStr)
      .arg(index, 3, 10, QChar('0'))  // p001, p002, p003...
      .arg(ext);
}

bool MainWindow::saveMultiPageAsPDF(const QList<QImage> &images,
                                    const QString &path) {
  if (images.isEmpty()) {
    return false;
  }

  QPdfWriter writer(path);
  writer.setPageSize(QPageSize(QPageSize::A4));
  writer.setTitle(tr("多页扫描文档"));
  QPainter painter(&writer);

  for (int i = 0; i < images.size(); ++i) {
    if (i > 0) {
      writer.newPage();
    }

    // 将图像缩放到适合页面大小
    QRect pageRect = painter.viewport();
    QSize imageSize = images[i].size();
    imageSize.scale(pageRect.size(), Qt::KeepAspectRatio);

    QRect targetRect = QRect(QPoint(0, 0), imageSize);
    targetRect.moveCenter(pageRect.center());

    painter.drawImage(targetRect, images[i]);
  }

  painter.end();
  return true;
}


void MainWindow::onLayoutSettingsClicked() {
  // 创建界面布局设置对话框
  QDialog dialog(this);
  dialog.setWindowTitle(tr("界面布局设置"));
  dialog.setMinimumWidth(500);
  
  QVBoxLayout *layout = new QVBoxLayout(&dialog);
  
  QLabel *titleLabel = new QLabel(tr("<b>选择界面布局</b>"), &dialog);
  layout->addWidget(titleLabel);
  
  // 布局选项组
  QGroupBox *layoutGroup = new QGroupBox(tr("预设布局"), &dialog);
  QVBoxLayout *layoutLayout = new QVBoxLayout(layoutGroup);
  
  // 布局1：默认布局（工具栏在上，页面在左，扫描配置在右）
  QPushButton *layout1Btn = new QPushButton(tr("📐 默认布局"), layoutGroup);
  layout1Btn->setToolTip(tr("工具栏在上，页面列表在左，扫描配置在右"));
  layout1Btn->setStyleSheet("QPushButton { text-align: left; padding: 10px; }");
  connect(layout1Btn, &QPushButton::clicked, [this, &dialog]() {
    applyLayout1();
    m_statusBar->showMessage(tr("✓ 已切换到默认布局"), 3000);
    dialog.accept();
  });
  layoutLayout->addWidget(layout1Btn);
  
  QLabel *layout1Desc = new QLabel(tr("  • 工具栏：顶部\n"
                                      "  • 页面列表：左侧\n"
                                      "  • 扫描配置：右侧\n"
                                      "  • 适合：标准使用"), layoutGroup);
  layout1Desc->setStyleSheet("color: #666; font-size: 11px; margin-left: 20px; margin-bottom: 10px;");
  layoutLayout->addWidget(layout1Desc);
  
  // 布局2：反转布局（工具栏在上，扫描配置在左，页面在右）
  QPushButton *layout2Btn = new QPushButton(tr("🔄 反转布局"), layoutGroup);
  layout2Btn->setToolTip(tr("工具栏在上，扫描配置在左，页面列表在右"));
  layout2Btn->setStyleSheet("QPushButton { text-align: left; padding: 10px; }");
  connect(layout2Btn, &QPushButton::clicked, [this, &dialog]() {
    applyLayout2();
    m_statusBar->showMessage(tr("✓ 已切换到反转布局"), 3000);
    dialog.accept();
  });
  layoutLayout->addWidget(layout2Btn);
  
  QLabel *layout2Desc = new QLabel(tr("  • 工具栏：顶部\n"
                                      "  • 扫描配置：左侧\n"
                                      "  • 页面列表：右侧\n"
                                      "  • 适合：左手操作"), layoutGroup);
  layout2Desc->setStyleSheet("color: #666; font-size: 11px; margin-left: 20px; margin-bottom: 10px;");
  layoutLayout->addWidget(layout2Desc);
  
  layout->addWidget(layoutGroup);
  
  // 恢复默认按钮
  QPushButton *resetBtn = new QPushButton(tr("🔄 恢复默认布局"), &dialog);
  resetBtn->setStyleSheet("QPushButton { background-color: #FFA500; color: white; font-weight: bold; padding: 8px; }");
  connect(resetBtn, &QPushButton::clicked, [this, &dialog]() {
    applyLayout1();
    m_statusBar->showMessage(tr("✓ 已恢复默认布局"), 3000);
    dialog.accept();
  });
  layout->addWidget(resetBtn);
  
  // 说明信息
  QLabel *infoLabel = new QLabel(
      tr("<b>提示：</b><br>"
         "• 所有停靠窗口都可以自由拖动和调整大小<br>"
         "• 可以将停靠窗口拖出主窗口成为独立窗口<br>"
         "• 双击停靠窗口标题栏可以最大化/还原<br>"
         "• 布局设置会自动保存"), &dialog);
  infoLabel->setWordWrap(true);
  infoLabel->setStyleSheet("color: #444; font-size: 11px; padding: 12px; background-color: #e3f2fd; border-radius: 5px; border-left: 4px solid #2196F3;");
  layout->addWidget(infoLabel);
  
  // 关闭按钮
  QPushButton *closeBtn = new QPushButton(tr("关闭"), &dialog);
  connect(closeBtn, &QPushButton::clicked, &dialog, &QDialog::accept);
  layout->addWidget(closeBtn);
  
  dialog.exec();
}

// 布局1：默认布局（工具栏在上，页面在左，扫描配置在右）
void MainWindow::applyLayout1() {
  QDockWidget *pagesDock = m_pagesDock;
  QDockWidget *scanDock = findChild<QDockWidget *>("scanSettingsDock");
  
  if (!pagesDock || !scanDock) return;
  
  // 移除所有停靠窗口
  removeDockWidget(pagesDock);
  removeDockWidget(scanDock);
  
  // 重新添加到指定位置
  addDockWidget(Qt::LeftDockWidgetArea, pagesDock);
  addDockWidget(Qt::RightDockWidgetArea, scanDock);
  
  // 确保可见
  pagesDock->show();
  scanDock->show();
  
  qDebug() << "已应用布局1：默认布局";
}

// 布局2：反转布局（工具栏在上，扫描配置在左，页面在右）
void MainWindow::applyLayout2() {
  QDockWidget *pagesDock = m_pagesDock;
  QDockWidget *scanDock = findChild<QDockWidget *>("scanSettingsDock");
  
  if (!pagesDock || !scanDock) return;
  
  // 移除所有停靠窗口
  removeDockWidget(pagesDock);
  removeDockWidget(scanDock);
  
  // 重新添加到指定位置（反转）
  addDockWidget(Qt::LeftDockWidgetArea, scanDock);
  addDockWidget(Qt::RightDockWidgetArea, pagesDock);
  
  // 确保可见
  pagesDock->show();
  scanDock->show();
  
  qDebug() << "已应用布局2：反转布局";
}


