#ifndef PLATFORM_DETECTION_H
#define PLATFORM_DETECTION_H

// 平台和架构检测头文件
// 用于Simple-ScanX的跨平台适配

// 架构检测
#if defined(__x86_64__) || defined(__x86_64) || defined(__amd64__) || defined(__amd64) || defined(_M_X64)
    #define SCANX_ARCH_X86_64 1
    #define SCANX_ARCH_NAME "x86_64"
#elif defined(__aarch64__) || defined(_M_ARM64)
    #define SCANX_ARCH_ARM64 1
    #define SCANX_ARCH_NAME "arm64"
#elif defined(__loongarch64)
    #define SCANX_ARCH_LOONGARCH64 1
    #define SCANX_ARCH_NAME "loongarch64"
#elif defined(__arm__) || defined(__arm) || defined(_ARM_) || defined(_M_ARM)
    #define SCANX_ARCH_ARM32 1
    #define SCANX_ARCH_NAME "arm32"
#elif defined(__i386__) || defined(__i386) || defined(_M_IX86)
    #define SCANX_ARCH_X86_32 1
    #define SCANX_ARCH_NAME "x86_32"
#else
    #define SCANX_ARCH_UNKNOWN 1
    #define SCANX_ARCH_NAME "unknown"
#endif

// 操作系统检测
#if defined(__linux__)
    #define SCANX_OS_LINUX 1
    #define SCANX_OS_NAME "Linux"
#elif defined(_WIN32) || defined(_WIN64)
    #define SCANX_OS_WINDOWS 1
    #define SCANX_OS_NAME "Windows"
#elif defined(__APPLE__)
    #define SCANX_OS_MACOS 1
    #define SCANX_OS_NAME "macOS"
#elif defined(__FreeBSD__)
    #define SCANX_OS_FREEBSD 1
    #define SCANX_OS_NAME "FreeBSD"
#else
    #define SCANX_OS_UNKNOWN 1
    #define SCANX_OS_NAME "Unknown"
#endif

// 编译器检测
#if defined(__GNUC__)
    #define SCANX_COMPILER_GCC 1
    #define SCANX_COMPILER_NAME "GCC"
    #define SCANX_COMPILER_VERSION (__GNUC__ * 10000 + __GNUC_MINOR__ * 100 + __GNUC_PATCHLEVEL__)
#elif defined(__clang__)
    #define SCANX_COMPILER_CLANG 1
    #define SCANX_COMPILER_NAME "Clang"
    #define SCANX_COMPILER_VERSION (__clang_major__ * 10000 + __clang_minor__ * 100 + __clang_patchlevel__)
#elif defined(_MSC_VER)
    #define SCANX_COMPILER_MSVC 1
    #define SCANX_COMPILER_NAME "MSVC"
    #define SCANX_COMPILER_VERSION _MSC_VER
#else
    #define SCANX_COMPILER_UNKNOWN 1
    #define SCANX_COMPILER_NAME "Unknown"
    #define SCANX_COMPILER_VERSION 0
#endif

// 字节序检测
#include <QtGlobal>
#if Q_BYTE_ORDER == Q_LITTLE_ENDIAN
    #define SCANX_LITTLE_ENDIAN 1
#elif Q_BYTE_ORDER == Q_BIG_ENDIAN
    #define SCANX_BIG_ENDIAN 1
#endif

// 性能优化宏
#ifdef SCANX_ARCH_X86_64
    // x86_64 特定优化
    #define SCANX_CACHE_LINE_SIZE 64
    #define SCANX_SIMD_AVAILABLE 1
#elif defined(SCANX_ARCH_ARM64)
    // ARM64 特定优化
    #define SCANX_CACHE_LINE_SIZE 64
    #define SCANX_NEON_AVAILABLE 1
#elif defined(SCANX_ARCH_LOONGARCH64)
    // LoongArch64 特定优化
    #define SCANX_CACHE_LINE_SIZE 64
    #define SCANX_LSX_AVAILABLE 1  // LoongArch SIMD Extension
#else
    // 通用设置
    #define SCANX_CACHE_LINE_SIZE 64
#endif

// 内存对齐宏
#ifdef SCANX_COMPILER_GCC
    #define SCANX_ALIGNED(x) __attribute__((aligned(x)))
    #define SCANX_LIKELY(x) __builtin_expect(!!(x), 1)
    #define SCANX_UNLIKELY(x) __builtin_expect(!!(x), 0)
#elif defined(SCANX_COMPILER_CLANG)
    #define SCANX_ALIGNED(x) __attribute__((aligned(x)))
    #define SCANX_LIKELY(x) __builtin_expect(!!(x), 1)
    #define SCANX_UNLIKELY(x) __builtin_expect(!!(x), 0)
#elif defined(SCANX_COMPILER_MSVC)
    #define SCANX_ALIGNED(x) __declspec(align(x))
    #define SCANX_LIKELY(x) (x)
    #define SCANX_UNLIKELY(x) (x)
#else
    #define SCANX_ALIGNED(x)
    #define SCANX_LIKELY(x) (x)
    #define SCANX_UNLIKELY(x) (x)
#endif

// 调试宏
#ifdef QT_DEBUG
    #include <QDebug>
    #define SCANX_DEBUG(msg) qDebug() << "[" << SCANX_ARCH_NAME << "|" << SCANX_OS_NAME << "]" << msg
#else
    #define SCANX_DEBUG(msg) do {} while(0)
#endif

// 平台信息函数声明
#ifdef __cplusplus
extern "C" {
#endif

// 获取平台信息的函数
const char* scanx_get_arch_name();
const char* scanx_get_os_name();
const char* scanx_get_compiler_name();
int scanx_get_compiler_version();
bool scanx_is_little_endian();

#ifdef __cplusplus
}
#endif

#endif // PLATFORM_DETECTION_H