#ifndef SANE_H
#define SANE_H

#ifdef __cplusplus
extern "C" {
#endif

/* SANE 基本类型定义 */
typedef unsigned char SANE_Byte;
typedef int SANE_Int;
typedef double SANE_Fixed;
typedef char* SANE_String;
typedef const char* SANE_String_Const;
typedef void* SANE_Handle;
typedef int SANE_Bool;

#define SANE_TRUE  1
#define SANE_FALSE 0

/* SANE 状态码 */
typedef enum {
    SANE_STATUS_GOOD = 0,
    SANE_STATUS_UNSUPPORTED,
    SANE_STATUS_CANCELLED,
    SANE_STATUS_DEVICE_BUSY,
    SANE_STATUS_INVAL,
    SANE_STATUS_EOF,
    SANE_STATUS_JAMMED,
    SANE_STATUS_NO_DOCS,
    SANE_STATUS_COVER_OPEN,
    SANE_STATUS_IO_ERROR,
    SANE_STATUS_NO_MEM,
    SANE_STATUS_ACCESS_DENIED
} SANE_Status;

/* SANE 数据类型 */
typedef enum {
    SANE_TYPE_BOOL = 0,
    SANE_TYPE_INT,
    SANE_TYPE_FIXED,
    SANE_TYPE_STRING,
    SANE_TYPE_BUTTON,
    SANE_TYPE_GROUP
} SANE_Value_Type;

/* SANE 约束类型 */
typedef enum {
    SANE_CONSTRAINT_NONE = 0,
    SANE_CONSTRAINT_RANGE,
    SANE_CONSTRAINT_WORD_LIST,
    SANE_CONSTRAINT_STRING_LIST
} SANE_Constraint_Type;

/* SANE 动作类型 */
typedef enum {
    SANE_ACTION_GET_VALUE = 0,
    SANE_ACTION_SET_VALUE,
    SANE_ACTION_SET_AUTO
} SANE_Action;

/* SANE 帧类型 */
typedef enum {
    SANE_FRAME_GRAY = 0,
    SANE_FRAME_RGB,
    SANE_FRAME_RED,
    SANE_FRAME_GREEN,
    SANE_FRAME_BLUE
} SANE_Frame;

/* SANE 设备结构 */
typedef struct {
    SANE_String_Const name;
    SANE_String_Const vendor;
    SANE_String_Const model;
    SANE_String_Const type;
} SANE_Device;

/* SANE 参数结构 */
typedef struct {
    SANE_Frame format;
    SANE_Bool last_frame;
    SANE_Int bytes_per_line;
    SANE_Int pixels_per_line;
    SANE_Int lines;
    SANE_Int depth;
} SANE_Parameters;

/* SANE 选项描述符 */
typedef struct {
    SANE_String_Const name;
    SANE_String_Const title;
    SANE_String_Const desc;
    SANE_Value_Type type;
    SANE_Int unit;
    SANE_Int size;
    SANE_Int cap;
    SANE_Constraint_Type constraint_type;
    union {
        const SANE_String_Const *string_list;
        const SANE_Int *word_list;
        const void *range;
    } constraint;
} SANE_Option_Descriptor;

/* SANE 函数声明 */
SANE_Status sane_init(SANE_Int *version_code, void *authorize);
void sane_exit(void);
SANE_Status sane_get_devices(const SANE_Device ***device_list, SANE_Bool local_only);
SANE_Status sane_open(SANE_String_Const devicename, SANE_Handle *handle);
void sane_close(SANE_Handle handle);
const SANE_Option_Descriptor *sane_get_option_descriptor(SANE_Handle handle, SANE_Int option);
SANE_Status sane_control_option(SANE_Handle handle, SANE_Int option, SANE_Action action, void *value, SANE_Int *info);
SANE_Status sane_get_parameters(SANE_Handle handle, SANE_Parameters *params);
SANE_Status sane_start(SANE_Handle handle);
SANE_Status sane_read(SANE_Handle handle, SANE_Byte *data, SANE_Int max_length, SANE_Int *length);
void sane_cancel(SANE_Handle handle);
SANE_String_Const sane_strstatus(SANE_Status status);

/* SANE 版本宏 */
#define SANE_VERSION_MAJOR(code) ((code) >> 24)
#define SANE_VERSION_MINOR(code) (((code) >> 16) & 0xff)
#define SANE_VERSION_BUILD(code) (((code) >> 8) & 0xff)

/* SANE Fixed 转换宏 */
#define SANE_FIX(v) ((SANE_Fixed)((v) * (1 << 16)))
#define SANE_UNFIX(v) ((double)(v) / (1 << 16))

#ifdef __cplusplus
}
#endif

#endif /* SANE_H */