#include "scannerbrandprofile.h"
#include <QDebug>
#include <QDir>
#include <QStandardPaths>

ScannerBrandProfile::ScannerBrandProfile(QObject *parent)
    : QObject(parent)
    , m_currentBrand(ScannerBrand::Unknown)
{
    initializeDefaultProfiles();
}

ScannerBrandProfile::~ScannerBrandProfile()
{
}

ScannerBrand ScannerBrandProfile::detectBrand(const QString &vendor, const QString &model)
{
    QString vendorLower = vendor.toLower();
    QString modelLower = model.toLower();
    
    // 爱普生
    if (vendorLower.contains("epson")) {
        return ScannerBrand::Epson;
    }
    
    // 惠普
    if (vendorLower.contains("hp") || vendorLower.contains("hewlett")) {
        return ScannerBrand::HP;
    }
    
    // 佳能
    if (vendorLower.contains("canon")) {
        return ScannerBrand::Canon;
    }
    
    // 兄弟
    if (vendorLower.contains("brother")) {
        return ScannerBrand::Brother;
    }
    
    // 富士通
    if (vendorLower.contains("fujitsu")) {
        return ScannerBrand::Fujitsu;
    }
    
    // 奔图
    if (vendorLower.contains("pantum")) {
        return ScannerBrand::Pantum;
    }
    
    // 柯达
    if (vendorLower.contains("kodak")) {
        return ScannerBrand::Kodak;
    }
    
    // 中晶
    if (vendorLower.contains("microtek")) {
        return ScannerBrand::Microtek;
    }
    
    // 虹光
    if (vendorLower.contains("avision")) {
        return ScannerBrand::Avision;
    }
    
    // 紫光
    if (vendorLower.contains("unis")) {
        return ScannerBrand::Unis;
    }
    
    return ScannerBrand::Unknown;
}

QString ScannerBrandProfile::getBrandName(ScannerBrand brand)
{
    switch (brand) {
        case ScannerBrand::Epson: return "Epson";
        case ScannerBrand::HP: return "HP";
        case ScannerBrand::Canon: return "Canon";
        case ScannerBrand::Brother: return "Brother";
        case ScannerBrand::Fujitsu: return "Fujitsu";
        case ScannerBrand::Pantum: return "Pantum";
        case ScannerBrand::Kodak: return "Kodak";
        case ScannerBrand::Microtek: return "Microtek";
        case ScannerBrand::Avision: return "Avision";
        case ScannerBrand::Unis: return "Unis";
        default: return "Unknown";
    }
}

QString ScannerBrandProfile::getBrandNameChinese(ScannerBrand brand)
{
    switch (brand) {
        case ScannerBrand::Epson: return "爱普生";
        case ScannerBrand::HP: return "惠普";
        case ScannerBrand::Canon: return "佳能";
        case ScannerBrand::Brother: return "兄弟";
        case ScannerBrand::Fujitsu: return "富士通";
        case ScannerBrand::Pantum: return "奔图";
        case ScannerBrand::Kodak: return "柯达";
        case ScannerBrand::Microtek: return "中晶";
        case ScannerBrand::Avision: return "虹光";
        case ScannerBrand::Unis: return "紫光";
        default: return "未知";
    }
}

void ScannerBrandProfile::initializeDefaultProfiles()
{
    // 初始化所有品牌的默认配置
    createDefaultProfile(ScannerBrand::Epson);
    createDefaultProfile(ScannerBrand::HP);
    createDefaultProfile(ScannerBrand::Canon);
    createDefaultProfile(ScannerBrand::Brother);
    createDefaultProfile(ScannerBrand::Fujitsu);
    createDefaultProfile(ScannerBrand::Pantum);
}

void ScannerBrandProfile::createDefaultProfile(ScannerBrand brand)
{
    QJsonObject profile;
    
    switch (brand) {
        case ScannerBrand::Epson: {
            profile["brand"] = "Epson";
            profile["displayName"] = "爱普生";
            
            QJsonObject defaultSettings;
            defaultSettings["resolution"] = 300;
            defaultSettings["colorMode"] = "Color";
            profile["defaultSettings"] = defaultSettings;
            
            QJsonObject documentMode;
            documentMode["resolution"] = 300;
            documentMode["colorMode"] = "Gray";
            profile["documentMode"] = documentMode;
            
            QJsonObject photoMode;
            photoMode["resolution"] = 600;
            photoMode["colorMode"] = "Color";
            photoMode["enableColorCorrection"] = true;
            profile["photoMode"] = photoMode;
            
            QJsonObject quirks;
            quirks["sourceOptionInactive"] = true;
            quirks["requiresColorCorrectionTable"] = true;
            quirks["slowADFDetection"] = true;
            profile["quirks"] = quirks;
            
            QJsonObject recommendations;
            recommendations["document"] = "使用 300 DPI 灰度模式扫描文档";
            recommendations["photo"] = "使用 600 DPI 彩色模式扫描照片，启用色彩校正";
            profile["recommendations"] = recommendations;
            
            break;
        }
        
        case ScannerBrand::HP: {
            profile["brand"] = "HP";
            profile["displayName"] = "惠普";
            
            QJsonObject defaultSettings;
            defaultSettings["resolution"] = 200;
            defaultSettings["colorMode"] = "Gray";
            profile["defaultSettings"] = defaultSettings;
            
            QJsonObject documentMode;
            documentMode["resolution"] = 200;
            documentMode["colorMode"] = "Gray";
            documentMode["enableAutoDeskew"] = true;
            profile["documentMode"] = documentMode;
            
            QJsonObject batchMode;
            batchMode["resolution"] = 200;
            batchMode["colorMode"] = "Gray";
            batchMode["enableBlankPageDetection"] = true;
            profile["batchMode"] = batchMode;
            
            QJsonObject quirks;
            quirks["requiresSpecialDriver"] = true;
            profile["quirks"] = quirks;
            
            QJsonObject recommendations;
            recommendations["document"] = "使用 200 DPI 灰度模式，启用 ADF 批量扫描";
            recommendations["batch"] = "启用空白页检测和自动纠偏";
            profile["recommendations"] = recommendations;
            
            break;
        }
        
        case ScannerBrand::Canon: {
            profile["brand"] = "Canon";
            profile["displayName"] = "佳能";
            
            QJsonObject defaultSettings;
            defaultSettings["resolution"] = 300;
            defaultSettings["colorMode"] = "Color";
            defaultSettings["enableAutoDeskew"] = true;
            profile["defaultSettings"] = defaultSettings;
            
            QJsonObject recommendations;
            recommendations["document"] = "使用 300 DPI 彩色模式，启用自动纠偏";
            profile["recommendations"] = recommendations;
            
            break;
        }
        
        case ScannerBrand::Brother: {
            profile["brand"] = "Brother";
            profile["displayName"] = "兄弟";
            
            QJsonObject defaultSettings;
            defaultSettings["resolution"] = 200;
            defaultSettings["colorMode"] = "Gray";
            profile["defaultSettings"] = defaultSettings;
            
            QJsonObject recommendations;
            recommendations["document"] = "使用 200 DPI 灰度模式，适合商用文档";
            profile["recommendations"] = recommendations;
            
            break;
        }
        
        case ScannerBrand::Pantum: {
            profile["brand"] = "Pantum";
            profile["displayName"] = "奔图";
            
            QJsonObject defaultSettings;
            defaultSettings["resolution"] = 300;
            defaultSettings["colorMode"] = "Color";
            profile["defaultSettings"] = defaultSettings;
            
            QJsonObject recommendations;
            recommendations["document"] = "使用 300 DPI 彩色模式";
            profile["recommendations"] = recommendations;
            
            break;
        }
        
        default:
            return;
    }
    
    m_profiles[brand] = profile;
}

bool ScannerBrandProfile::loadProfile(ScannerBrand brand)
{
    m_currentBrand = brand;
    
    if (m_profiles.contains(brand)) {
        m_currentProfile = m_profiles[brand];
        qDebug() << "已加载品牌配置:" << getBrandNameChinese(brand);
        return true;
    }
    
    qDebug() << "未找到品牌配置:" << getBrandName(brand);
    return false;
}

ScanSettings ScannerBrandProfile::getDefaultSettings() const
{
    ScanSettings settings;
    
    if (m_currentProfile.contains("defaultSettings")) {
        QJsonObject defaults = m_currentProfile["defaultSettings"].toObject();
        settings.resolution = defaults["resolution"].toInt(300);
        settings.colorMode = defaults["colorMode"].toString("Color");
        settings.enableColorCorrection = defaults["enableColorCorrection"].toBool(false);
        settings.enableAutoDeskew = defaults["enableAutoDeskew"].toBool(false);
        settings.enableBlankPageDetection = defaults["enableBlankPageDetection"].toBool(false);
    }
    
    return settings;
}

ScanSettings ScannerBrandProfile::getSettingsForMode(ScanMode mode) const
{
    ScanSettings settings;
    QString modeKey;
    
    switch (mode) {
        case ScanMode::Document:
            modeKey = "documentMode";
            break;
        case ScanMode::Photo:
            modeKey = "photoMode";
            break;
        case ScanMode::Batch:
            modeKey = "batchMode";
            break;
        case ScanMode::Quick:
            modeKey = "quickMode";
            break;
        case ScanMode::HighQuality:
            modeKey = "highQualityMode";
            break;
    }
    
    if (m_currentProfile.contains(modeKey)) {
        QJsonObject modeSettings = m_currentProfile[modeKey].toObject();
        settings.resolution = modeSettings["resolution"].toInt(300);
        settings.colorMode = modeSettings["colorMode"].toString("Color");
        settings.enableColorCorrection = modeSettings["enableColorCorrection"].toBool(false);
        settings.enableAutoDeskew = modeSettings["enableAutoDeskew"].toBool(false);
        settings.enableBlankPageDetection = modeSettings["enableBlankPageDetection"].toBool(false);
    } else {
        // 如果没有特定模式配置，返回默认设置
        settings = getDefaultSettings();
    }
    
    return settings;
}

BrandQuirks ScannerBrandProfile::getQuirks() const
{
    BrandQuirks quirks;
    
    if (m_currentProfile.contains("quirks")) {
        QJsonObject quirksObj = m_currentProfile["quirks"].toObject();
        quirks.sourceOptionInactive = quirksObj["sourceOptionInactive"].toBool(false);
        quirks.requiresColorCorrectionTable = quirksObj["requiresColorCorrectionTable"].toBool(false);
        quirks.slowADFDetection = quirksObj["slowADFDetection"].toBool(false);
        quirks.needsUSBReset = quirksObj["needsUSBReset"].toBool(false);
        quirks.requiresSpecialDriver = quirksObj["requiresSpecialDriver"].toBool(false);
    }
    
    return quirks;
}

QString ScannerBrandProfile::getRecommendation(ScanMode mode) const
{
    if (!m_currentProfile.contains("recommendations")) {
        return QString();
    }
    
    QJsonObject recommendations = m_currentProfile["recommendations"].toObject();
    
    switch (mode) {
        case ScanMode::Document:
            return recommendations["document"].toString();
        case ScanMode::Photo:
            return recommendations["photo"].toString();
        case ScanMode::Batch:
            return recommendations["batch"].toString();
        default:
            return QString();
    }
}

bool ScannerBrandProfile::supportsFeature(const QString &feature) const
{
    if (!m_currentProfile.contains("features")) {
        return false;
    }
    
    QJsonObject features = m_currentProfile["features"].toObject();
    return features[feature].toBool(false);
}
