#ifndef SCANNERBRANDPROFILE_H
#define SCANNERBRANDPROFILE_H

#include <QObject>
#include <QString>
#include <QMap>
#include <QJsonObject>
#include <QJsonDocument>
#include <QFile>

// 扫描仪品牌枚举
enum class ScannerBrand {
    Unknown,
    Epson,      // 爱普生
    HP,         // 惠普
    Canon,      // 佳能
    Brother,    // 兄弟
    Fujitsu,    // 富士通
    Pantum,     // 奔图
    Kodak,      // 柯达
    Microtek,   // 中晶
    Avision,    // 虹光
    Unis        // 紫光
};

// 扫描模式
enum class ScanMode {
    Document,   // 文档扫描
    Photo,      // 照片扫描
    Batch,      // 批量扫描
    Quick,      // 快速扫描
    HighQuality // 高质量扫描
};

// 扫描设置
struct ScanSettings {
    int resolution;
    QString colorMode;
    bool enableColorCorrection;
    bool enableAutoDeskew;
    bool enableBlankPageDetection;
    QString paperSource;
    
    ScanSettings() 
        : resolution(300)
        , colorMode("Color")
        , enableColorCorrection(false)
        , enableAutoDeskew(false)
        , enableBlankPageDetection(false)
        , paperSource("Flatbed") {}
};

// 品牌特性
struct BrandQuirks {
    bool sourceOptionInactive;
    bool requiresColorCorrectionTable;
    bool slowADFDetection;
    bool needsUSBReset;
    bool requiresSpecialDriver;
    
    BrandQuirks()
        : sourceOptionInactive(false)
        , requiresColorCorrectionTable(false)
        , slowADFDetection(false)
        , needsUSBReset(false)
        , requiresSpecialDriver(false) {}
};

// 扫描仪品牌配置类
class ScannerBrandProfile : public QObject
{
    Q_OBJECT

public:
    explicit ScannerBrandProfile(QObject *parent = nullptr);
    ~ScannerBrandProfile();

    // 品牌检测
    static ScannerBrand detectBrand(const QString &vendor, const QString &model);
    static QString getBrandName(ScannerBrand brand);
    static QString getBrandNameChinese(ScannerBrand brand);
    
    // 加载配置
    bool loadProfile(ScannerBrand brand);
    bool loadProfileFromFile(const QString &filePath);
    
    // 获取设置
    ScanSettings getDefaultSettings() const;
    ScanSettings getSettingsForMode(ScanMode mode) const;
    BrandQuirks getQuirks() const;
    
    // 选项映射
    QString mapColorMode(const QString &mode) const;
    QString mapPaperSource(const QString &source) const;
    
    // 推荐信息
    QString getRecommendation(ScanMode mode) const;
    QStringList getSupportedDrivers() const;
    
    // 特性检查
    bool supportsFeature(const QString &feature) const;
    
    // 当前品牌
    ScannerBrand getCurrentBrand() const { return m_currentBrand; }

private:
    void initializeDefaultProfiles();
    void createDefaultProfile(ScannerBrand brand);
    
    ScannerBrand m_currentBrand;
    QJsonObject m_currentProfile;
    QMap<ScannerBrand, QJsonObject> m_profiles;
};

#endif // SCANNERBRANDPROFILE_H
