#!/bin/bash
# 多架构构建脚本
# 用于在不同架构上构建亮仔扫描

set -e

# 颜色定义
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# 打印带颜色的消息
print_info() {
    echo -e "${BLUE}[INFO]${NC} $1"
}

print_success() {
    echo -e "${GREEN}[SUCCESS]${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}[WARNING]${NC} $1"
}

print_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

# 检测当前架构
detect_arch() {
    local arch=$(uname -m)
    case "$arch" in
        x86_64|amd64)
            echo "x86_64"
            ;;
        aarch64|arm64)
            echo "arm64"
            ;;
        loongarch64)
            echo "loongarch64"
            ;;
        armv7l)
            echo "armv7l"
            ;;
        i686|i386)
            echo "i386"
            ;;
        *)
            echo "unknown"
            ;;
    esac
}

# 检查依赖
check_dependencies() {
    print_info "检查构建依赖..."
    
    local missing_deps=()
    
    # 检查必需的工具
    for cmd in cmake gcc g++ pkg-config; do
        if ! command -v $cmd &> /dev/null; then
            missing_deps+=($cmd)
        fi
    done
    
    # 检查 Qt5
    if ! pkg-config --exists Qt5Core; then
        missing_deps+=("qtbase5-dev")
    fi
    
    # 检查 SANE
    if ! pkg-config --exists sane-backends; then
        missing_deps+=("libsane-dev")
    fi
    
    if [ ${#missing_deps[@]} -gt 0 ]; then
        print_error "缺少以下依赖: ${missing_deps[*]}"
        print_info "请运行以下命令安装依赖:"
        echo "  sudo apt-get install cmake qtbase5-dev libsane-dev pkg-config build-essential"
        return 1
    fi
    
    print_success "所有依赖已满足"
    return 0
}

# 显示系统信息
show_system_info() {
    print_info "系统信息:"
    echo "  操作系统: $(uname -s)"
    echo "  内核版本: $(uname -r)"
    echo "  架构: $(detect_arch)"
    echo "  CPU: $(grep -m1 'model name' /proc/cpuinfo 2>/dev/null | cut -d: -f2 | xargs || echo 'Unknown')"
    echo "  内存: $(free -h | awk '/^Mem:/ {print $2}')"
    
    if command -v lsb_release &> /dev/null; then
        echo "  发行版: $(lsb_release -ds)"
    fi
    
    echo ""
}

# 构建项目
build_project() {
    local build_type=${1:-Release}
    local build_dir="build-$(detect_arch)"
    
    print_info "开始构建 (类型: $build_type, 架构: $(detect_arch))"
    
    # 创建构建目录
    if [ -d "$build_dir" ]; then
        print_warning "构建目录已存在，清理中..."
        rm -rf "$build_dir"
    fi
    
    mkdir -p "$build_dir"
    cd "$build_dir"
    
    # 配置 CMake
    print_info "配置 CMake..."
    cmake .. \
        -DCMAKE_BUILD_TYPE=$build_type \
        -DCMAKE_INSTALL_PREFIX=/usr \
        || { print_error "CMake 配置失败"; return 1; }
    
    # 编译
    print_info "编译中..."
    local cpu_cores=$(nproc)
    make -j$cpu_cores || { print_error "编译失败"; return 1; }
    
    print_success "构建完成！"
    print_info "可执行文件: $build_dir/lz-scan"
    
    cd ..
    return 0
}

# 运行测试
run_tests() {
    local build_dir="build-$(detect_arch)"
    
    if [ ! -f "$build_dir/lz-scan" ]; then
        print_error "找不到可执行文件，请先构建项目"
        return 1
    fi
    
    print_info "运行基本测试..."
    
    # 测试程序是否能启动
    timeout 5 "$build_dir/lz-scan" --version &> /dev/null || true
    
    # 检查 SANE
    if command -v scanimage &> /dev/null; then
        print_info "检测扫描仪..."
        scanimage -L || print_warning "未检测到扫描仪"
    else
        print_warning "未安装 sane-utils，跳过扫描仪检测"
    fi
    
    print_success "测试完成"
}

# 创建 DEB 包
create_deb() {
    print_info "创建 DEB 包..."
    
    if [ ! -d "debian" ]; then
        print_error "找不到 debian 目录"
        return 1
    fi
    
    # 检查 debhelper
    if ! command -v dpkg-buildpackage &> /dev/null; then
        print_error "未安装 dpkg-buildpackage"
        print_info "请运行: sudo apt-get install devscripts debhelper"
        return 1
    fi
    
    # 构建包
    dpkg-buildpackage -us -uc -b || { print_error "DEB 包构建失败"; return 1; }
    
    print_success "DEB 包创建完成"
    print_info "包文件位于上级目录"
    ls -lh ../*.deb 2>/dev/null || true
}

# 清理构建文件
clean() {
    print_info "清理构建文件..."
    
    rm -rf build-*
    rm -rf debian/.debhelper
    rm -f debian/files
    rm -f debian/*.log
    rm -f debian/*.substvars
    rm -rf debian/lz-scan
    
    print_success "清理完成"
}

# 显示帮助信息
show_help() {
    cat << EOF
亮仔扫描 - 多架构构建脚本

用法: $0 [选项]

选项:
  -h, --help          显示此帮助信息
  -i, --info          显示系统信息
  -c, --check         检查构建依赖
  -b, --build [TYPE]  构建项目 (TYPE: Release|Debug, 默认: Release)
  -t, --test          运行测试
  -d, --deb           创建 DEB 包
  -a, --all           执行完整构建流程（检查+构建+测试+打包）
  --clean             清理构建文件

示例:
  $0 --info           # 显示系统信息
  $0 --check          # 检查依赖
  $0 --build          # 构建项目（Release）
  $0 --build Debug    # 构建项目（Debug）
  $0 --all            # 完整构建流程
  $0 --clean          # 清理构建文件

支持的架构:
  - x86_64 (AMD64)
  - ARM64 (AArch64)
  - LoongArch64 (龙芯)
  - ARMv7
  - i386

EOF
}

# 主函数
main() {
    echo "========================================"
    echo "  亮仔扫描 - 多架构构建脚本"
    echo "========================================"
    echo ""
    
    # 如果没有参数，显示帮助
    if [ $# -eq 0 ]; then
        show_help
        exit 0
    fi
    
    # 解析参数
    case "$1" in
        -h|--help)
            show_help
            ;;
        -i|--info)
            show_system_info
            ;;
        -c|--check)
            check_dependencies
            ;;
        -b|--build)
            check_dependencies || exit 1
            build_project "${2:-Release}"
            ;;
        -t|--test)
            run_tests
            ;;
        -d|--deb)
            create_deb
            ;;
        -a|--all)
            show_system_info
            check_dependencies || exit 1
            build_project "Release" || exit 1
            run_tests || exit 1
            create_deb || exit 1
            print_success "完整构建流程完成！"
            ;;
        --clean)
            clean
            ;;
        *)
            print_error "未知选项: $1"
            echo ""
            show_help
            exit 1
            ;;
    esac
}

# 运行主函数
main "$@"
