#include "imageprocessor.h"

#include <QDebug>
#include <QPainter>
#include <QTransform>
#include <QVector>
#include <QtMath>
#include <QApplication>

ImageProcessor::ImageProcessor(QObject *parent)
    : QObject(parent)
{
}

ImageProcessor::~ImageProcessor()
{
}

QImage ImageProcessor::rotateLeft(const QImage &image)
{
    if (image.isNull()) {
        qDebug() << "无法旋转空图像";
        return QImage();
    }
    
    QTransform transform;
    transform.rotate(-90);
    
    QImage rotatedImage = image.transformed(transform);
    
    qDebug() << "图像已向左旋转90度";
    return rotatedImage;
}

QImage ImageProcessor::rotateRight(const QImage &image)
{
    if (image.isNull()) {
        qDebug() << "无法旋转空图像";
        return QImage();
    }
    
    QTransform transform;
    transform.rotate(90);
    
    QImage rotatedImage = image.transformed(transform);
    
    qDebug() << "图像已向右旋转90度";
    return rotatedImage;
}

QImage ImageProcessor::crop(const QImage &image, const QRect &rect)
{
    if (image.isNull()) {
        qDebug() << "无法裁剪空图像";
        return QImage();
    }
    
    if (!rect.isValid() || !image.rect().contains(rect)) {
        qDebug() << "无效的裁剪区域";
        return image;
    }
    
    QImage croppedImage = image.copy(rect);
    
    qDebug() << "图像已裁剪至: " << rect;
    return croppedImage;
}

QImage ImageProcessor::adjustBrightnessContrast(const QImage &image, int brightness, int contrast)
{
    if (image.isNull()) {
        qDebug() << "无法调整空图像";
        return QImage();
    }
    
    // 将亮度和对比度范围限制在-100到100之间
    brightness = qBound(-100, brightness, 100);
    contrast = qBound(-100, contrast, 100);
    
    // 创建一个新图像
    QImage adjustedImage = image.convertToFormat(QImage::Format_ARGB32);
    
    // 计算对比度因子
    qreal contrastFactor = 1.0;
    if (contrast > 0) {
        contrastFactor = 1.0 + contrast / 100.0;
    } else if (contrast < 0) {
        contrastFactor = 1.0 / (1.0 - contrast / 100.0);
    }
    
    // 计算亮度偏移
    int brightnessOffset = brightness * 255 / 100;
    
    // 处理每个像素
    for (int y = 0; y < adjustedImage.height(); ++y) {
        QRgb *line = reinterpret_cast<QRgb*>(adjustedImage.scanLine(y));
        
        for (int x = 0; x < adjustedImage.width(); ++x) {
            int r = qRed(line[x]);
            int g = qGreen(line[x]);
            int b = qBlue(line[x]);
            int a = qAlpha(line[x]);
            
            // 应用对比度
            r = qBound(0, static_cast<int>((r - 127) * contrastFactor + 127), 255);
            g = qBound(0, static_cast<int>((g - 127) * contrastFactor + 127), 255);
            b = qBound(0, static_cast<int>((b - 127) * contrastFactor + 127), 255);
            
            // 应用亮度
            r = qBound(0, r + brightnessOffset, 255);
            g = qBound(0, g + brightnessOffset, 255);
            b = qBound(0, b + brightnessOffset, 255);
            
            line[x] = qRgba(r, g, b, a);
        }
        
        // 更新进度
        if (y % 20 == 0) {
            emit processingProgress(y * 100 / adjustedImage.height());
            QApplication::processEvents();
        }
    }
    
    emit processingProgress(100);
    qDebug() << "已调整亮度: " << brightness << ", 对比度: " << contrast;
    
    return adjustedImage;
}

QImage ImageProcessor::adjustHueSaturation(const QImage &image, int hue, int saturation)
{
    if (image.isNull()) {
        qDebug() << "无法调整空图像";
        return QImage();
    }
    
    // 将色度范围限制在-180到180之间，饱和度限制在-100到100之间
    hue = qBound(-180, hue, 180);
    saturation = qBound(-100, saturation, 100);
    
    emit processingProgress(0);
    qDebug() << "开始调整色度: " << hue << ", 饱和度: " << saturation;
    
    // 创建一个新图像
    QImage adjustedImage = image.convertToFormat(QImage::Format_ARGB32);
    
    // 计算饱和度因子
    qreal saturationFactor = 1.0 + saturation / 100.0;
    
    // 处理每个像素
    for (int y = 0; y < adjustedImage.height(); ++y) {
        QRgb *line = reinterpret_cast<QRgb*>(adjustedImage.scanLine(y));
        
        for (int x = 0; x < adjustedImage.width(); ++x) {
            QColor color(line[x]);
            
            // 转换到HSV色彩空间
            int h, s, v;
            color.getHsv(&h, &s, &v);
            
            // 调整色度（只对有颜色的像素调整，灰色像素h为-1）
            if (h >= 0) {
                h = (h + hue + 360) % 360;
            }
            
            // 调整饱和度
            s = qBound(0, static_cast<int>(s * saturationFactor), 255);
            
            // 转换回RGB
            color.setHsv(h, s, v);
            line[x] = color.rgba();
        }
        
        // 更新进度
        if (y % 20 == 0) {
            emit processingProgress(y * 100 / adjustedImage.height());
            QApplication::processEvents();
        }
    }
    
    emit processingProgress(100);
    qDebug() << "已调整色度: " << hue << ", 饱和度: " << saturation;
    
    return adjustedImage;
}

QImage ImageProcessor::autoLevel(const QImage &image)
{
    if (image.isNull()) {
        qDebug() << "无法自动调整空图像";
        return QImage();
    }
    
    QImage adjustedImage = image.convertToFormat(QImage::Format_ARGB32);
    
    // 找出最小和最大亮度值
    int minR = 255, minG = 255, minB = 255;
    int maxR = 0, maxG = 0, maxB = 0;
    
    for (int y = 0; y < adjustedImage.height(); ++y) {
        QRgb *line = reinterpret_cast<QRgb*>(adjustedImage.scanLine(y));
        
        for (int x = 0; x < adjustedImage.width(); ++x) {
            int r = qRed(line[x]);
            int g = qGreen(line[x]);
            int b = qBlue(line[x]);
            
            minR = qMin(minR, r);
            minG = qMin(minG, g);
            minB = qMin(minB, b);
            
            maxR = qMax(maxR, r);
            maxG = qMax(maxG, g);
            maxB = qMax(maxB, b);
        }
        
        // 更新进度
        if (y % 20 == 0) {
            emit processingProgress(y * 50 / adjustedImage.height());
            QApplication::processEvents();
        }
    }
    
    // 如果最小值等于最大值，则不需要调整
    if (minR == maxR && minG == maxG && minB == maxB) {
        qDebug() << "图像已经是最佳水平，无需自动调整";
        return image;
    }
    
    // 计算缩放因子
    qreal scaleR = (maxR > minR) ? 255.0 / (maxR - minR) : 1.0;
    qreal scaleG = (maxG > minG) ? 255.0 / (maxG - minG) : 1.0;
    qreal scaleB = (maxB > minB) ? 255.0 / (maxB - minB) : 1.0;
    
    // 应用自动调整
    for (int y = 0; y < adjustedImage.height(); ++y) {
        QRgb *line = reinterpret_cast<QRgb*>(adjustedImage.scanLine(y));
        
        for (int x = 0; x < adjustedImage.width(); ++x) {
            int r = qRed(line[x]);
            int g = qGreen(line[x]);
            int b = qBlue(line[x]);
            int a = qAlpha(line[x]);
            
            r = qBound(0, static_cast<int>((r - minR) * scaleR), 255);
            g = qBound(0, static_cast<int>((g - minG) * scaleG), 255);
            b = qBound(0, static_cast<int>((b - minB) * scaleB), 255);
            
            line[x] = qRgba(r, g, b, a);
        }
        
        // 更新进度
        if (y % 20 == 0) {
            emit processingProgress(50 + y * 50 / adjustedImage.height());
            QApplication::processEvents();
        }
    }
    
    emit processingProgress(100);
    qDebug() << "已自动调整图像水平";
    
    return adjustedImage;
}

QImage ImageProcessor::sharpen(const QImage &image, int amount)
{
    if (image.isNull()) {
        qDebug() << "无法锐化空图像";
        return QImage();
    }
    
    // 将锐化量限制在0到100之间
    amount = qBound(0, amount, 100);
    
    // 如果锐化量为0，则返回原图像
    if (amount == 0) {
        return image;
    }
    
    // 计算锐化因子
    qreal factor = amount / 100.0;
    
    // 创建锐化卷积核
    QVector<qreal> kernel = {
        -factor, -factor, -factor,
        -factor, 1 + 8 * factor, -factor,
        -factor, -factor, -factor
    };
    
    return applyFilter(image, kernel, 3);
}

QImage ImageProcessor::denoise(const QImage &image, int level)
{
    if (image.isNull()) {
        qDebug() << "无法去噪空图像";
        return QImage();
    }
    
    // 将去噪级别限制在0到100之间
    level = qBound(0, level, 100);
    
    // 如果去噪级别为0，则返回原图像
    if (level == 0) {
        return image;
    }
    
    // 计算高斯模糊卷积核大小
    int kernelSize = 3 + 2 * (level / 20); // 3, 5, 7, 9, 11
    
    // 创建高斯模糊卷积核
    QVector<qreal> kernel;
    kernel.resize(kernelSize * kernelSize);
    
    qreal sigma = 0.3 * ((kernelSize - 1) * 0.5 - 1) + 0.8;
    qreal sum = 0;
    
    for (int y = 0; y < kernelSize; ++y) {
        for (int x = 0; x < kernelSize; ++x) {
            int dx = x - (kernelSize / 2);
            int dy = y - (kernelSize / 2);
            qreal g = qExp(-(dx * dx + dy * dy) / (2 * sigma * sigma));
            kernel[y * kernelSize + x] = g;
            sum += g;
        }
    }
    
    // 归一化卷积核
    for (int i = 0; i < kernel.size(); ++i) {
        kernel[i] /= sum;
    }
    
    return applyFilter(image, kernel, kernelSize);
}

QImage ImageProcessor::convertToGrayscale(const QImage &image)
{
    if (image.isNull()) {
        qDebug() << "无法转换空图像";
        return QImage();
    }
    
    QImage grayImage = image.convertToFormat(QImage::Format_Grayscale8);
    
    qDebug() << "图像已转换为灰度";
    return grayImage;
}

QImage ImageProcessor::convertToBinary(const QImage &image, int threshold)
{
    if (image.isNull()) {
        qDebug() << "无法转换空图像";
        return QImage();
    }
    
    // 如果阈值为-1，则自动计算阈值
    if (threshold == -1) {
        threshold = calculateThreshold(image);
    } else {
        threshold = qBound(0, threshold, 255);
    }
    
    return applyThreshold(image, threshold);
}

QImage ImageProcessor::deskew(const QImage &image)
{
    if (image.isNull()) {
        qDebug() << "无法进行倾斜校正：空图像";
        return QImage();
    }

    emit processingProgress(0);
    qDebug() << "开始倾斜校正处理（增强版）...";

    // 转换为灰度图像以便处理
    QImage grayImage = convertToGrayscale(image);
    emit processingProgress(10);

    // 方法1：基于线条检测倾斜角度
    double angleByLines = detectSkewAngleByLines(grayImage);
    emit processingProgress(30);
    qDebug() << "基于线条检测的倾斜角度: " << angleByLines << " 度";

    // 方法2：基于文字检测倾斜角度
    double angleByText = detectSkewAngleByText(grayImage);
    emit processingProgress(50);
    qDebug() << "基于文字检测的倾斜角度: " << angleByText << " 度";

    // 综合两种方法的结果
    double skewAngle = 0.0;
    if (qAbs(angleByLines) > 0.1 && qAbs(angleByText) > 0.1) {
        // 两种方法都检测到倾斜，取平均值
        skewAngle = (angleByLines + angleByText) / 2.0;
        qDebug() << "综合两种方法，最终倾斜角度: " << skewAngle << " 度";
    } else if (qAbs(angleByLines) > 0.1) {
        // 只有线条检测有效
        skewAngle = angleByLines;
        qDebug() << "使用线条检测结果: " << skewAngle << " 度";
    } else if (qAbs(angleByText) > 0.1) {
        // 只有文字检测有效
        skewAngle = angleByText;
        qDebug() << "使用文字检测结果: " << skewAngle << " 度";
    } else {
        qDebug() << "图像倾斜角度很小，无需校正";
        emit processingProgress(100);
        return image;
    }

    // 如果倾斜角度很小，则不需要校正
    if (qAbs(skewAngle) < 0.1) {
        qDebug() << "图像倾斜角度很小，无需校正";
        emit processingProgress(100);
        return image;
    }

    // 应用旋转校正
    QTransform transform;
    transform.rotate(-skewAngle); // 负角度用于纠正倾斜

    QImage deskewedImage = image.transformed(transform, Qt::SmoothTransformation);
    emit processingProgress(90);

    // 裁剪掉旋转后的黑边
    deskewedImage = cropRotatedImage(deskewedImage, skewAngle);
    emit processingProgress(100);

    qDebug() << "倾斜校正完成，校正角度: " << skewAngle << " 度";
    return deskewedImage;
}

QImage ImageProcessor::rotateByAngle(const QImage &image, double angle)
{
    if (image.isNull()) {
        qDebug() << "无法旋转空图像";
        return QImage();
    }

    if (qAbs(angle) < 0.01) {
        qDebug() << "旋转角度为0，无需旋转";
        return image;
    }

    emit processingProgress(0);
    qDebug() << "开始按指定角度旋转: " << angle << " 度";

    // 应用旋转
    QTransform transform;
    transform.rotate(angle);

    QImage rotatedImage = image.transformed(transform, Qt::SmoothTransformation);
    emit processingProgress(80);

    // 裁剪掉旋转后的黑边
    if (qAbs(angle) > 0.5) {
        rotatedImage = cropRotatedImage(rotatedImage, angle);
    }
    
    emit processingProgress(100);
    qDebug() << "角度旋转完成: " << angle << " 度";
    
    return rotatedImage;
}

QImage ImageProcessor::removeBorder(const QImage &image, int borderWidth)
{
    if (image.isNull()) {
        qDebug() << "无法处理空图像";
        return QImage();
    }
    
    // 确保边框宽度有效
    borderWidth = qBound(0, borderWidth, qMin(image.width(), image.height()) / 4);
    
    if (borderWidth == 0) {
        return image;
    }
    
    QRect cropRect(borderWidth, borderWidth, 
                  image.width() - 2 * borderWidth, 
                  image.height() - 2 * borderWidth);
    
    return crop(image, cropRect);
}

QImage ImageProcessor::enhanceText(const QImage &image)
{
    if (image.isNull()) {
        qDebug() << "无法增强空图像";
        return QImage();
    }
    
    emit processingProgress(0);
    qDebug() << "开始文档增强处理（保留彩色，突出内容）...";
    
    // 彩色文档增强
    QImage enhancedImage = enhanceDocumentColor(image);
    emit processingProgress(50);
    
    // 去除背景，保留内容
    enhancedImage = removeBackgroundKeepContent(enhancedImage);
    emit processingProgress(100);
    
    qDebug() << "文档增强完成";
    return enhancedImage;
}

QImage ImageProcessor::enhanceDocumentColor(const QImage &image)
{
    // 彩色文档增强：提高对比度和饱和度，保留彩色信息
    qDebug() << "开始彩色文档增强...";
    
    QImage enhancedImage = image.convertToFormat(QImage::Format_ARGB32);
    int width = enhancedImage.width();
    int height = enhancedImage.height();
    
    for (int y = 0; y < height; ++y) {
        QRgb *line = reinterpret_cast<QRgb*>(enhancedImage.scanLine(y));
        
        for (int x = 0; x < width; ++x) {
            QColor color(line[x]);
            
            // 转换到HSV色彩空间
            int h, s, v;
            color.getHsv(&h, &s, &v);
            
            // 增加饱和度（让颜色更鲜艳）
            s = qMin(255, (int)(s * 1.3));
            
            // 增加明度对比（让暗的更暗，亮的更亮）
            if (v < 128) {
                v = qMax(0, (int)(v * 0.8));  // 暗部压暗
            } else {
                v = qMin(255, (int)(v * 1.1));  // 亮部提亮
            }
            
            color.setHsv(h, s, v);
            line[x] = color.rgba();
        }
        
        if (y % 20 == 0) {
            emit processingProgress(y * 50 / height);
            QApplication::processEvents();
        }
    }
    
    // 锐化处理，让文字更清晰
    enhancedImage = sharpen(enhancedImage, 40);
    
    qDebug() << "彩色文档增强完成";
    return enhancedImage;
}

QImage ImageProcessor::removeBackgroundKeepContent(const QImage &image)
{
    // 去除背景，保留内容（基于亮度和颜色分析）
    qDebug() << "开始去除背景，保留内容...";
    
    QImage resultImage = image.convertToFormat(QImage::Format_ARGB32);
    int width = resultImage.width();
    int height = resultImage.height();
    
    // 第一步：分析图像，确定背景亮度阈值
    QVector<int> brightnessHistogram(256, 0);
    
    for (int y = 0; y < height; y += 2) {
        for (int x = 0; x < width; x += 2) {
            QColor color(resultImage.pixel(x, y));
            int brightness = (color.red() + color.green() + color.blue()) / 3;
            brightnessHistogram[brightness]++;
        }
    }
    
    // 找到最常见的亮度值（通常是背景）
    int maxCount = 0;
    int backgroundBrightness = 255;
    for (int i = 200; i < 256; ++i) {  // 背景通常是亮色
        if (brightnessHistogram[i] > maxCount) {
            maxCount = brightnessHistogram[i];
            backgroundBrightness = i;
        }
    }
    
    qDebug() << "检测到背景亮度:" << backgroundBrightness;
    
    // 第二步：去除背景
    int threshold = backgroundBrightness - 30;  // 比背景暗30以上的认为是内容
    
    for (int y = 0; y < height; ++y) {
        QRgb *line = reinterpret_cast<QRgb*>(resultImage.scanLine(y));
        
        for (int x = 0; x < width; ++x) {
            QColor color(line[x]);
            int brightness = (color.red() + color.green() + color.blue()) / 3;
            
            if (brightness > threshold) {
                // 这是背景区域，将其变白
                int whitenFactor = (brightness - threshold) * 255 / (255 - threshold);
                int newR = qMin(255, color.red() + (255 - color.red()) * whitenFactor / 255);
                int newG = qMin(255, color.green() + (255 - color.green()) * whitenFactor / 255);
                int newB = qMin(255, color.blue() + (255 - color.blue()) * whitenFactor / 255);
                
                line[x] = qRgba(newR, newG, newB, color.alpha());
            } else {
                // 这是内容区域，增强对比度
                int newR = qMax(0, qMin(255, (int)((color.red() - 127) * 1.3 + 127)));
                int newG = qMax(0, qMin(255, (int)((color.green() - 127) * 1.3 + 127)));
                int newB = qMax(0, qMin(255, (int)((color.blue() - 127) * 1.3 + 127)));
                
                line[x] = qRgba(newR, newG, newB, color.alpha());
            }
        }
        
        if (y % 20 == 0) {
            emit processingProgress(50 + y * 50 / height);
            QApplication::processEvents();
        }
    }
    
    qDebug() << "背景去除完成";
    return resultImage;
}

QImage ImageProcessor::bleachBackground(const QImage &image, int threshold, int strength)
{
    if (image.isNull()) {
        qDebug() << "无法漂白空图像";
        return QImage();
    }
    
    emit processingProgress(0);
    
    QImage bleachedImage = image.convertToFormat(QImage::Format_ARGB32);
    int width = bleachedImage.width();
    int height = bleachedImage.height();
    
    qDebug() << "开始图片漂白处理，阈值:" << threshold << "强度:" << strength;
    
    // 第一步：分析图像，找到主要的背景色
    QMap<QRgb, int> colorCount;
    int totalPixels = width * height;
    
    // 采样分析背景色（为了性能，可以降采样）
    int sampleStep = qMax(1, qMin(width, height) / 100);
    for (int y = 0; y < height; y += sampleStep) {
        for (int x = 0; x < width; x += sampleStep) {
            QRgb pixel = bleachedImage.pixel(x, y);
            colorCount[pixel]++;
        }
        
        // 更新进度
        if (y % (height / 10) == 0) {
            emit processingProgress(10 + (y * 20) / height);
            QApplication::processEvents();
        }
    }
    
    // 找到最常见的颜色作为背景色（通常是最亮的颜色）
    QRgb backgroundColor = 0;
    int maxCount = 0;
    int maxBrightness = 0;
    
    for (auto it = colorCount.begin(); it != colorCount.end(); ++it) {
        QColor color(it.key());
        int brightness = (color.red() + color.green() + color.blue()) / 3;
        
        // 优先选择亮度高且出现频率高的颜色作为背景色
        if (brightness > threshold && (it.value() > maxCount || 
            (it.value() > maxCount * 0.8 && brightness > maxBrightness))) {
            backgroundColor = it.key();
            maxCount = it.value();
            maxBrightness = brightness;
        }
    }
    
    QColor bgColor(backgroundColor);
    qDebug() << "检测到背景色 RGB:" << bgColor.red() << bgColor.green() << bgColor.blue() 
             << "出现次数:" << maxCount;
    
    emit processingProgress(30);
    
    // 第二步：漂白处理
    for (int y = 0; y < height; ++y) {
        QRgb *line = reinterpret_cast<QRgb*>(bleachedImage.scanLine(y));
        
        for (int x = 0; x < width; ++x) {
            QColor pixelColor(line[x]);
            
            // 计算与背景色的相似度
            int redDiff = qAbs(pixelColor.red() - bgColor.red());
            int greenDiff = qAbs(pixelColor.green() - bgColor.green());
            int blueDiff = qAbs(pixelColor.blue() - bgColor.blue());
            int totalDiff = redDiff + greenDiff + blueDiff;
            
            // 如果颜色相似度高且亮度足够，进行漂白处理
            int pixelBrightness = (pixelColor.red() + pixelColor.green() + pixelColor.blue()) / 3;
            
            if (totalDiff < (765 - threshold * 3) && pixelBrightness > threshold - 50) {
                // 计算漂白强度（距离背景色越近，漂白越强）
                float similarity = 1.0f - (float)totalDiff / (765 - threshold * 3);
                float bleachFactor = similarity * strength / 100.0f;
                
                // 向白色漂白
                int newRed = qMin(255, (int)(pixelColor.red() + (255 - pixelColor.red()) * bleachFactor));
                int newGreen = qMin(255, (int)(pixelColor.green() + (255 - pixelColor.green()) * bleachFactor));
                int newBlue = qMin(255, (int)(pixelColor.blue() + (255 - pixelColor.blue()) * bleachFactor));
                
                line[x] = qRgba(newRed, newGreen, newBlue, pixelColor.alpha());
            }
        }
        
        // 更新进度
        if (y % (height / 50) == 0) {
            emit processingProgress(30 + (y * 70) / height);
            QApplication::processEvents();
        }
    }
    
    emit processingProgress(100);
    qDebug() << "图片漂白处理完成";
    
    return bleachedImage;
}

QImage ImageProcessor::applyFilter(const QImage &image, const QVector<qreal> &kernel, int kernelSize)
{
    if (image.isNull() || kernel.isEmpty() || kernelSize <= 0) {
        return image;
    }
    
    QImage resultImage = image.convertToFormat(QImage::Format_ARGB32);
    QImage sourceImage = image.convertToFormat(QImage::Format_ARGB32);
    
    int offset = kernelSize / 2;
    
    for (int y = 0; y < resultImage.height(); ++y) {
        QRgb *destLine = reinterpret_cast<QRgb*>(resultImage.scanLine(y));
        
        for (int x = 0; x < resultImage.width(); ++x) {
            qreal sumR = 0, sumG = 0, sumB = 0;
            qreal sumA = 0;
            
            // 应用卷积核
            for (int ky = 0; ky < kernelSize; ++ky) {
                int sourceY = qBound(0, y + ky - offset, sourceImage.height() - 1);
                
                QRgb *sourceLine = reinterpret_cast<QRgb*>(sourceImage.scanLine(sourceY));
                
                for (int kx = 0; kx < kernelSize; ++kx) {
                    int sourceX = qBound(0, x + kx - offset, sourceImage.width() - 1);
                    QRgb pixel = sourceLine[sourceX];
                    
                    qreal kernelValue = kernel[ky * kernelSize + kx];
                    
                    sumR += qRed(pixel) * kernelValue;
                    sumG += qGreen(pixel) * kernelValue;
                    sumB += qBlue(pixel) * kernelValue;
                    sumA += qAlpha(pixel) * kernelValue;
                }
            }
            
            // 将结果限制在0-255范围内
            int r = qBound(0, static_cast<int>(sumR), 255);
            int g = qBound(0, static_cast<int>(sumG), 255);
            int b = qBound(0, static_cast<int>(sumB), 255);
            int a = qBound(0, static_cast<int>(sumA), 255);
            
            destLine[x] = qRgba(r, g, b, a);
        }
        
        // 更新进度
        if (y % 20 == 0) {
            emit processingProgress(y * 100 / resultImage.height());
            QApplication::processEvents();
        }
    }
    
    emit processingProgress(100);
    return resultImage;
}

int ImageProcessor::calculateThreshold(const QImage &image)
{
    // 使用Otsu算法自动计算二值化阈值
    QImage grayImage = convertToGrayscale(image);
    
    // 计算直方图
    QVector<int> histogram(256, 0);
    
    for (int y = 0; y < grayImage.height(); ++y) {
        const uchar *line = grayImage.constScanLine(y);
        
        for (int x = 0; x < grayImage.width(); ++x) {
            histogram[line[x]]++;
        }
    }
    
    // 计算总像素数
    int totalPixels = grayImage.width() * grayImage.height();
    
    // 计算Otsu阈值
    qreal sum = 0;
    for (int i = 0; i < 256; ++i) {
        sum += i * histogram[i];
    }
    
    qreal sumB = 0;
    int wB = 0;
    int wF = 0;
    
    qreal maxVariance = 0;
    int threshold = 0;
    
    for (int i = 0; i < 256; ++i) {
        wB += histogram[i];
        if (wB == 0) continue;
        
        wF = totalPixels - wB;
        if (wF == 0) break;
        
        sumB += i * histogram[i];
        
        qreal mB = sumB / wB;
        qreal mF = (sum - sumB) / wF;
        
        qreal variance = wB * wF * (mB - mF) * (mB - mF);
        
        if (variance > maxVariance) {
            maxVariance = variance;
            threshold = i;
        }
    }
    
    qDebug() << "自动计算的阈值: " << threshold;
    return threshold;
}

QImage ImageProcessor::applyThreshold(const QImage &image, int threshold)
{
    QImage grayImage = convertToGrayscale(image);
    QImage binaryImage(grayImage.size(), QImage::Format_Mono);
    
    for (int y = 0; y < grayImage.height(); ++y) {
        const uchar *srcLine = grayImage.constScanLine(y);
        
        for (int x = 0; x < grayImage.width(); ++x) {
            bool isWhite = srcLine[x] > threshold;
            binaryImage.setPixel(x, y, isWhite ? 1 : 0);
        }
        
        // 更新进度
        if (y % 20 == 0) {
            emit processingProgress(y * 100 / grayImage.height());
            QApplication::processEvents();
        }
    }
    
    emit processingProgress(100);
    qDebug() << "图像已转换为二值图像，阈值: " << threshold;
    
    return binaryImage;
}

QImage ImageProcessor::applyGamma(const QImage &image, qreal gamma)
{
    if (image.isNull()) {
        return QImage();
    }
    
    // 创建伽马校正查找表
    QVector<uchar> gammaTable(256);
    for (int i = 0; i < 256; ++i) {
        gammaTable[i] = qBound(0, static_cast<int>(qPow(i / 255.0, 1.0 / gamma) * 255), 255);
    }
    
    QImage resultImage = image.convertToFormat(QImage::Format_ARGB32);
    
    for (int y = 0; y < resultImage.height(); ++y) {
        QRgb *line = reinterpret_cast<QRgb*>(resultImage.scanLine(y));
        
        for (int x = 0; x < resultImage.width(); ++x) {
            int r = gammaTable[qRed(line[x])];
            int g = gammaTable[qGreen(line[x])];
            int b = gammaTable[qBlue(line[x])];
            int a = qAlpha(line[x]);
            
            line[x] = qRgba(r, g, b, a);
        }
        
        // 更新进度
        if (y % 20 == 0) {
            emit processingProgress(y * 100 / resultImage.height());
            QApplication::processEvents();
        }
    }
    
    emit processingProgress(100);
    qDebug() << "已应用伽马校正: " << gamma;
    
    return resultImage;
}

double ImageProcessor::detectSkewAngle(const QImage &grayImage)
{
    // 使用Hough变换检测倾斜角度
    // 这是一个简化的实现，实际应用中可能需要更复杂的算法
    
    // 首先进行边缘检测
    QImage edgeImage = detectEdges(grayImage);
    
    // 使用Hough变换检测直线
    QVector<double> angles;
    int width = edgeImage.width();
    int height = edgeImage.height();
    
    // 扫描角度范围 -45度到45度
    for (double angle = -45.0; angle <= 45.0; angle += 0.5) {
        int lineCount = 0;
        double radians = angle * M_PI / 180.0;
        double cosAngle = qCos(radians);
        double sinAngle = qSin(radians);
        
        // 在图像中寻找符合该角度的直线
        for (int y = 0; y < height; y += 5) {
            for (int x = 0; x < width; x += 5) {
                if (edgeImage.pixelIndex(x, y) > 0) {
                    // 检查该点是否在一条直线上
                    int linePixels = 0;
                    for (int i = -20; i <= 20; i++) {
                        int newX = x + i * cosAngle;
                        int newY = y + i * sinAngle;
                        
                        if (newX >= 0 && newX < width && newY >= 0 && newY < height) {
                            if (edgeImage.pixelIndex(newX, newY) > 0) {
                                linePixels++;
                            }
                        }
                    }
                    
                    if (linePixels > 15) { // 如果有足够多的点在直线上
                        lineCount++;
                    }
                }
            }
        }
        
        if (lineCount > 10) { // 如果找到足够多的直线
            angles.append(angle);
        }
    }
    
    // 计算最可能的倾斜角度
    if (angles.isEmpty()) {
        return 0.0; // 没有检测到明显的倾斜
    }
    
    // 计算角度的平均值
    double sum = 0.0;
    for (double angle : angles) {
        sum += angle;
    }
    
    return sum / angles.size();
}

QImage ImageProcessor::detectEdges(const QImage &grayImage)
{
    // 使用Sobel算子进行边缘检测
    QVector<qreal> sobelX = {
        -1, 0, 1,
        -2, 0, 2,
        -1, 0, 1
    };
    
    QVector<qreal> sobelY = {
        -1, -2, -1,
         0,  0,  0,
         1,  2,  1
    };
    
    QImage edgeX = applyFilter(grayImage, sobelX, 3);
    QImage edgeY = applyFilter(grayImage, sobelY, 3);
    
    // 合并X和Y方向的边缘
    QImage edgeImage(grayImage.size(), QImage::Format_Grayscale8);
    
    for (int y = 0; y < edgeImage.height(); ++y) {
        uchar *edgeLine = edgeImage.scanLine(y);
        const uchar *edgeXLine = edgeX.constScanLine(y);
        const uchar *edgeYLine = edgeY.constScanLine(y);
        
        for (int x = 0; x < edgeImage.width(); ++x) {
            int gx = edgeXLine[x];
            int gy = edgeYLine[x];
            int magnitude = qSqrt(gx * gx + gy * gy);
            edgeLine[x] = qBound(0, magnitude, 255);
        }
    }
    
    // 应用阈值，只保留强边缘
    return applyThreshold(edgeImage, 100);
}

double ImageProcessor::detectSkewAngleByLines(const QImage &grayImage)
{
    // 基于线条检测倾斜角度（使用Hough变换）
    qDebug() << "开始基于线条检测倾斜角度...";
    
    // 边缘检测
    QImage edgeImage = detectEdges(grayImage);
    
    int width = edgeImage.width();
    int height = edgeImage.height();
    
    // Hough变换累加器
    QMap<double, int> angleVotes;
    
    // 扫描角度范围 -15度到15度（文档通常不会倾斜太多）
    for (double angle = -15.0; angle <= 15.0; angle += 0.2) {
        int voteCount = 0;
        double radians = angle * M_PI / 180.0;
        double cosAngle = qCos(radians);
        double sinAngle = qSin(radians);
        
        // 在图像中寻找符合该角度的直线
        for (int y = 10; y < height - 10; y += 8) {
            for (int x = 10; x < width - 10; x += 8) {
                if (edgeImage.pixelIndex(x, y) > 0) {
                    // 检查该点是否在一条直线上
                    int linePixels = 0;
                    int checkLength = qMin(50, qMin(width, height) / 10);
                    
                    for (int i = -checkLength; i <= checkLength; i += 2) {
                        int newX = x + i * cosAngle;
                        int newY = y + i * sinAngle;
                        
                        if (newX >= 0 && newX < width && newY >= 0 && newY < height) {
                            if (edgeImage.pixelIndex(newX, newY) > 0) {
                                linePixels++;
                            }
                        }
                    }
                    
                    if (linePixels > checkLength / 3) {
                        voteCount++;
                    }
                }
            }
        }
        
        if (voteCount > 0) {
            angleVotes[angle] = voteCount;
        }
    }
    
    // 找到得票最多的角度
    double bestAngle = 0.0;
    int maxVotes = 0;
    
    for (auto it = angleVotes.begin(); it != angleVotes.end(); ++it) {
        if (it.value() > maxVotes) {
            maxVotes = it.value();
            bestAngle = it.key();
        }
    }
    
    qDebug() << "线条检测完成，最佳角度:" << bestAngle << "得票:" << maxVotes;
    
    // 如果得票数太少，说明没有明显的线条
    if (maxVotes < 10) {
        return 0.0;
    }
    
    return bestAngle;
}

double ImageProcessor::detectSkewAngleByText(const QImage &grayImage)
{
    // 基于文字检测倾斜角度（检测文字的水平和垂直对齐）
    qDebug() << "开始基于文字检测倾斜角度...";
    
    int width = grayImage.width();
    int height = grayImage.height();
    
    // 二值化图像以便检测文字
    QImage binaryImage = convertToBinary(grayImage, -1);
    
    // 统计每行的黑色像素数量（文字行会有较多黑色像素）
    QVector<int> rowDensity(height, 0);
    for (int y = 0; y < height; ++y) {
        for (int x = 0; x < width; ++x) {
            if (binaryImage.pixelIndex(x, y) == 0) { // 黑色像素
                rowDensity[y]++;
            }
        }
    }
    
    // 找到文字行（密度较高的行）
    QVector<int> textRows;
    int avgDensity = 0;
    for (int density : rowDensity) {
        avgDensity += density;
    }
    avgDensity /= height;
    
    for (int y = 0; y < height; ++y) {
        if (rowDensity[y] > avgDensity * 1.5) {
            textRows.append(y);
        }
    }
    
    if (textRows.size() < 3) {
        qDebug() << "检测到的文字行太少，无法判断倾斜";
        return 0.0;
    }
    
    // 分析文字行的倾斜趋势
    QVector<double> angles;
    
    for (int i = 0; i < textRows.size() - 1; ++i) {
        int y1 = textRows[i];
        int y2 = textRows[i + 1];
        
        // 找到这两行的中心点
        int x1 = 0, count1 = 0;
        int x2 = 0, count2 = 0;
        
        for (int x = 0; x < width; ++x) {
            if (binaryImage.pixelIndex(x, y1) == 0) {
                x1 += x;
                count1++;
            }
            if (binaryImage.pixelIndex(x, y2) == 0) {
                x2 += x;
                count2++;
            }
        }
        
        if (count1 > 0 && count2 > 0) {
            x1 /= count1;
            x2 /= count2;
            
            // 计算倾斜角度
            double angle = qAtan2(x2 - x1, y2 - y1) * 180.0 / M_PI;
            angles.append(angle);
        }
    }
    
    if (angles.isEmpty()) {
        return 0.0;
    }
    
    // 计算平均角度
    double sumAngle = 0.0;
    for (double angle : angles) {
        sumAngle += angle;
    }
    double avgAngle = sumAngle / angles.size();
    
    qDebug() << "文字检测完成，平均倾斜角度:" << avgAngle;
    
    return avgAngle;
}

QImage ImageProcessor::cropRotatedImage(const QImage &rotatedImage, double angle)
{
    if (qAbs(angle) < 0.1) {
        return rotatedImage; // 角度太小，不需要裁剪
    }
    
    // 计算旋转后需要裁剪的边界
    double radians = qAbs(angle) * M_PI / 180.0;
    double cosAngle = qCos(radians);
    double sinAngle = qSin(radians);
    
    int originalWidth = rotatedImage.width();
    int originalHeight = rotatedImage.height();
    
    // 计算裁剪区域，去除旋转产生的黑边
    int cropWidth = originalWidth * cosAngle - originalHeight * sinAngle;
    int cropHeight = originalHeight * cosAngle - originalWidth * sinAngle;
    
    // 确保裁剪尺寸有效
    cropWidth = qMax(1, qMin(cropWidth, originalWidth));
    cropHeight = qMax(1, qMin(cropHeight, originalHeight));
    
    // 计算裁剪起始位置（居中裁剪）
    int startX = (originalWidth - cropWidth) / 2;
    int startY = (originalHeight - cropHeight) / 2;
    
    QRect cropRect(startX, startY, cropWidth, cropHeight);
    
    return rotatedImage.copy(cropRect);
}

bool ImageProcessor::isBlankPage(const QImage &image, int threshold, double blankRatio)
{
    if (image.isNull()) {
        qDebug() << "无法检测空图像";
        return false;
    }
    
    // 转换为灰度图像以简化处理
    QImage grayImage = image.convertToFormat(QImage::Format_Grayscale8);
    int width = grayImage.width();
    int height = grayImage.height();
    int totalPixels = width * height;
    
    if (totalPixels == 0) {
        return false;
    }
    
    // 计算忽略边缘的范围（忽略边缘1%）
    int marginX = qMax(1, (int)(width * 0.01));   // 左右各忽略1%
    int marginY = qMax(1, (int)(height * 0.01));  // 上下各忽略1%
    
    // 有效检测区域
    int startX = marginX;
    int endX = width - marginX;
    int startY = marginY;
    int endY = height - marginY;
    
    qDebug() << "空白页检测: 图像尺寸" << width << "x" << height 
             << ", 忽略边缘" << marginX << "x" << marginY 
             << ", 检测区域" << (endX - startX) << "x" << (endY - startY);
    
    // 为了提高性能，可以采样检测（每隔几个像素检测一次）
    int sampleStep = qMax(1, qMin(width, height) / 200);  // 采样步长
    
    // 定义区域边界（上中下三等分）
    int regionHeight = (endY - startY) / 3;
    int topBoundary = startY + regionHeight;
    int bottomBoundary = endY - regionHeight;
    
    // 定义区域权重（上下部分权重更高，因为标题和页码通常在这些位置）
    double topWeight = 2.0;      // 上部权重（标题区域）
    double middleWeight = 1.0;   // 中部权重（正文区域）
    double bottomWeight = 2.0;   // 下部权重（页脚/页码区域）
    
    // 分区域统计
    struct RegionStats {
        int blankPixels = 0;
        int totalPixels = 0;
        double weight = 1.0;
    };
    
    RegionStats topRegion, middleRegion, bottomRegion;
    topRegion.weight = topWeight;
    middleRegion.weight = middleWeight;
    bottomRegion.weight = bottomWeight;
    
    // 只在有效区域内检测（忽略边缘）
    for (int y = startY; y < endY; y += sampleStep) {
        const uchar *line = grayImage.constScanLine(y);
        
        // 确定当前行属于哪个区域
        RegionStats *currentRegion;
        if (y < topBoundary) {
            currentRegion = &topRegion;
        } else if (y < bottomBoundary) {
            currentRegion = &middleRegion;
        } else {
            currentRegion = &bottomRegion;
        }
        
        for (int x = startX; x < endX; x += sampleStep) {
            uchar grayValue = line[x];
            
            // 如果像素值大于阈值，认为是空白像素
            if (grayValue >= threshold) {
                currentRegion->blankPixels++;
            }
            
            currentRegion->totalPixels++;
        }
    }
    
    // 计算各区域的空白比例
    double topRatio = (topRegion.totalPixels > 0) ? 
                      (double)topRegion.blankPixels / topRegion.totalPixels : 1.0;
    double middleRatio = (middleRegion.totalPixels > 0) ? 
                         (double)middleRegion.blankPixels / middleRegion.totalPixels : 1.0;
    double bottomRatio = (bottomRegion.totalPixels > 0) ? 
                         (double)bottomRegion.blankPixels / bottomRegion.totalPixels : 1.0;
    
    // 计算加权平均空白比例
    double totalWeight = topRegion.weight + middleRegion.weight + bottomRegion.weight;
    double weightedRatio = (topRatio * topRegion.weight + 
                           middleRatio * middleRegion.weight + 
                           bottomRatio * bottomRegion.weight) / totalWeight;
    
    // 判断逻辑：加权平均满足条件，且关键区域（上下）也要基本满足
    // 关键区域使用稍微宽松的阈值（95%），避免误判有少量内容的页面
    double keyRegionThreshold = blankRatio * 0.95;
    bool isBlank = (weightedRatio >= blankRatio) &&
                   (topRatio >= keyRegionThreshold) &&
                   (bottomRatio >= keyRegionThreshold);
    
    qDebug() << "空白页检测（区域加权）:"
             << "上部=" << QString::number(topRatio * 100, 'f', 1) << "%"
             << ", 中部=" << QString::number(middleRatio * 100, 'f', 1) << "%"
             << ", 下部=" << QString::number(bottomRatio * 100, 'f', 1) << "%"
             << ", 加权平均=" << QString::number(weightedRatio * 100, 'f', 1) << "%"
             << " (阈值:" << QString::number(blankRatio * 100, 'f', 1) << "%)"
             << ", 结果:" << (isBlank ? "空白页" : "非空白页");
    
    return isBlank;
}

QImage ImageProcessor::mergeImagesHorizontal(const QImage &image1, const QImage &image2, int spacing)
{
    if (image1.isNull() || image2.isNull()) {
        qDebug() << "无法合并空图像";
        return QImage();
    }
    
    emit processingProgress(0);
    
    int width1 = image1.width();
    int height1 = image1.height();
    int width2 = image2.width();
    int height2 = image2.height();
    
    // 计算合并后的尺寸
    int maxHeight = qMax(height1, height2);
    int totalWidth = width1 + width2 + spacing;
    
    qDebug() << "开始左右合并图片，尺寸:" << width1 << "x" << height1 << "+" << width2 << "x" << height2;
    
    // 创建结果图像（白色背景）
    QImage result(totalWidth, maxHeight, QImage::Format_RGB32);
    result.fill(Qt::white);
    
    emit processingProgress(20);
    
    // 使用 QPainter 绘制图片
    QPainter painter(&result);
    painter.setRenderHint(QPainter::SmoothPixmapTransform, true);
    
    // 绘制第一张图片（垂直居中对齐）
    int y1 = (maxHeight - height1) / 2;
    painter.drawImage(0, y1, image1);
    
    emit processingProgress(60);
    
    // 绘制第二张图片（垂直居中对齐）
    int y2 = (maxHeight - height2) / 2;
    painter.drawImage(width1 + spacing, y2, image2);
    
    emit processingProgress(90);
    
    painter.end();
    
    emit processingProgress(100);
    qDebug() << "左右合并完成，结果尺寸:" << totalWidth << "x" << maxHeight;
    
    return result;
}

QImage ImageProcessor::mergeImagesVertical(const QImage &image1, const QImage &image2, int spacing)
{
    if (image1.isNull() || image2.isNull()) {
        qDebug() << "无法合并空图像";
        return QImage();
    }
    
    emit processingProgress(0);
    
    int width1 = image1.width();
    int height1 = image1.height();
    int width2 = image2.width();
    int height2 = image2.height();
    
    // 计算合并后的尺寸
    int maxWidth = qMax(width1, width2);
    int totalHeight = height1 + height2 + spacing;
    
    qDebug() << "开始上下合并图片，尺寸:" << width1 << "x" << height1 << "+" << width2 << "x" << height2;
    
    // 创建结果图像（白色背景）
    QImage result(maxWidth, totalHeight, QImage::Format_RGB32);
    result.fill(Qt::white);
    
    emit processingProgress(20);
    
    // 使用 QPainter 绘制图片
    QPainter painter(&result);
    painter.setRenderHint(QPainter::SmoothPixmapTransform, true);
    
    // 绘制第一张图片（水平居中对齐）
    int x1 = (maxWidth - width1) / 2;
    painter.drawImage(x1, 0, image1);
    
    emit processingProgress(60);
    
    // 绘制第二张图片（水平居中对齐）
    int x2 = (maxWidth - width2) / 2;
    painter.drawImage(x2, height1 + spacing, image2);
    
    emit processingProgress(90);
    
    painter.end();
    
    emit processingProgress(100);
    qDebug() << "上下合并完成，结果尺寸:" << maxWidth << "x" << totalHeight;
    
    return result;
}


QRect ImageProcessor::detectContentBounds(const QImage &image, int threshold)
{
    if (image.isNull()) {
        qDebug() << "无法检测空图像的内容边界";
        return QRect();
    }
    
    QImage grayImage = image.convertToFormat(QImage::Format_Grayscale8);
    int width = grayImage.width();
    int height = grayImage.height();
    
    qDebug() << "开始检测内容边界，阈值:" << threshold;
    
    // 初始化边界
    int minX = width;
    int maxX = 0;
    int minY = height;
    int maxY = 0;
    
    bool foundContent = false;
    
    // 扫描图像，找到非白色区域的边界
    for (int y = 0; y < height; ++y) {
        const uchar *line = grayImage.constScanLine(y);
        for (int x = 0; x < width; ++x) {
            // 如果像素值小于阈值，说明不是白色背景
            if (line[x] < threshold) {
                foundContent = true;
                if (x < minX) minX = x;
                if (x > maxX) maxX = x;
                if (y < minY) minY = y;
                if (y > maxY) maxY = y;
            }
        }
    }
    
    if (!foundContent) {
        qDebug() << "未检测到内容，返回整个图像区域";
        return image.rect();
    }
    
    // 添加一些边距（5像素）
    int margin = 5;
    minX = qMax(0, minX - margin);
    minY = qMax(0, minY - margin);
    maxX = qMin(width - 1, maxX + margin);
    maxY = qMin(height - 1, maxY + margin);
    
    QRect contentRect(minX, minY, maxX - minX + 1, maxY - minY + 1);
    
    qDebug() << "检测到内容边界:" << contentRect 
             << "原始尺寸:" << width << "x" << height
             << "内容占比:" << (contentRect.width() * contentRect.height() * 100.0 / (width * height)) << "%";
    
    return contentRect;
}

QImage ImageProcessor::autoTrimWhitespace(const QImage &image, int threshold)
{
    if (image.isNull()) {
        qDebug() << "无法裁剪空图像";
        return QImage();
    }
    
    emit processingProgress(0);
    
    // 检测内容边界
    QRect contentRect = detectContentBounds(image, threshold);
    
    emit processingProgress(50);
    
    if (contentRect.isEmpty() || contentRect == image.rect()) {
        qDebug() << "无需裁剪";
        emit processingProgress(100);
        return image;
    }
    
    // 裁剪图像
    QImage trimmedImage = image.copy(contentRect);
    
    emit processingProgress(100);
    qDebug() << "自动裁剪完成，从" << image.size() << "裁剪到" << trimmedImage.size();
    
    return trimmedImage;
}


QVector<QRect> ImageProcessor::detectMultipleContentBounds(const QImage &image, bool isHorizontal, int threshold)
{
    if (image.isNull()) {
        qDebug() << "无法检测空图像的内容边界";
        return QVector<QRect>();
    }
    
    QImage grayImage = image.convertToFormat(QImage::Format_Grayscale8);
    int width = grayImage.width();
    int height = grayImage.height();
    
    qDebug() << "开始检测多个内容区域，方向:" << (isHorizontal ? "左右" : "上下");
    
    QVector<QRect> contentRects;
    
    if (isHorizontal) {
        // 左右拆分：从中间分割
        int midX = width / 2;
        
        // 检测左半部分
        QRect leftBounds;
        int minX = width, maxX = 0, minY = height, maxY = 0;
        bool foundLeft = false;
        
        for (int y = 0; y < height; ++y) {
            const uchar *line = grayImage.constScanLine(y);
            for (int x = 0; x < midX; ++x) {
                if (line[x] < threshold) {
                    foundLeft = true;
                    minX = qMin(minX, x);
                    maxX = qMax(maxX, x);
                    minY = qMin(minY, y);
                    maxY = qMax(maxY, y);
                }
            }
        }
        
        if (foundLeft) {
            int margin = 5;
            leftBounds = QRect(qMax(0, minX - margin), 
                              qMax(0, minY - margin),
                              qMin(midX - minX, maxX - minX + 2 * margin),
                              maxY - minY + 2 * margin);
            contentRects.append(leftBounds);
            qDebug() << "检测到左侧内容:" << leftBounds;
        }
        
        // 检测右半部分
        QRect rightBounds;
        minX = width; maxX = 0; minY = height; maxY = 0;
        bool foundRight = false;
        
        for (int y = 0; y < height; ++y) {
            const uchar *line = grayImage.constScanLine(y);
            for (int x = midX; x < width; ++x) {
                if (line[x] < threshold) {
                    foundRight = true;
                    minX = qMin(minX, x);
                    maxX = qMax(maxX, x);
                    minY = qMin(minY, y);
                    maxY = qMax(maxY, y);
                }
            }
        }
        
        if (foundRight) {
            int margin = 5;
            rightBounds = QRect(qMax(midX, minX - margin),
                               qMax(0, minY - margin),
                               maxX - minX + 2 * margin,
                               maxY - minY + 2 * margin);
            contentRects.append(rightBounds);
            qDebug() << "检测到右侧内容:" << rightBounds;
        }
        
    } else {
        // 上下拆分：从中间分割
        int midY = height / 2;
        
        // 检测上半部分
        QRect topBounds;
        int minX = width, maxX = 0, minY = height, maxY = 0;
        bool foundTop = false;
        
        for (int y = 0; y < midY; ++y) {
            const uchar *line = grayImage.constScanLine(y);
            for (int x = 0; x < width; ++x) {
                if (line[x] < threshold) {
                    foundTop = true;
                    minX = qMin(minX, x);
                    maxX = qMax(maxX, x);
                    minY = qMin(minY, y);
                    maxY = qMax(maxY, y);
                }
            }
        }
        
        if (foundTop) {
            int margin = 5;
            topBounds = QRect(qMax(0, minX - margin),
                             qMax(0, minY - margin),
                             maxX - minX + 2 * margin,
                             qMin(midY - minY, maxY - minY + 2 * margin));
            contentRects.append(topBounds);
            qDebug() << "检测到上部内容:" << topBounds;
        }
        
        // 检测下半部分
        QRect bottomBounds;
        minX = width; maxX = 0; minY = height; maxY = 0;
        bool foundBottom = false;
        
        for (int y = midY; y < height; ++y) {
            const uchar *line = grayImage.constScanLine(y);
            for (int x = 0; x < width; ++x) {
                if (line[x] < threshold) {
                    foundBottom = true;
                    minX = qMin(minX, x);
                    maxX = qMax(maxX, x);
                    minY = qMin(minY, y);
                    maxY = qMax(maxY, y);
                }
            }
        }
        
        if (foundBottom) {
            int margin = 5;
            bottomBounds = QRect(qMax(0, minX - margin),
                                qMax(midY, minY - margin),
                                maxX - minX + 2 * margin,
                                maxY - minY + 2 * margin);
            contentRects.append(bottomBounds);
            qDebug() << "检测到下部内容:" << bottomBounds;
        }
    }
    
    qDebug() << "共检测到" << contentRects.size() << "个内容区域";
    return contentRects;
}

QVector<QImage> ImageProcessor::splitImageHorizontal(const QImage &image, int threshold)
{
    if (image.isNull()) {
        qDebug() << "无法拆分空图像";
        return QVector<QImage>();
    }
    
    emit processingProgress(0);
    
    qDebug() << "开始左右拆分图片";
    
    // 检测左右两部分的内容边界
    QVector<QRect> contentRects = detectMultipleContentBounds(image, true, threshold);
    
    emit processingProgress(50);
    
    QVector<QImage> splitImages;
    
    for (const QRect &rect : contentRects) {
        if (!rect.isEmpty()) {
            QImage part = image.copy(rect);
            // 裁剪每部分的空白
            part = autoTrimWhitespace(part, threshold);
            splitImages.append(part);
        }
    }
    
    emit processingProgress(100);
    qDebug() << "左右拆分完成，得到" << splitImages.size() << "个部分";
    
    return splitImages;
}

QVector<QImage> ImageProcessor::splitImageVertical(const QImage &image, int threshold)
{
    if (image.isNull()) {
        qDebug() << "无法拆分空图像";
        return QVector<QImage>();
    }
    
    emit processingProgress(0);
    
    qDebug() << "开始上下拆分图片";
    
    // 检测上下两部分的内容边界
    QVector<QRect> contentRects = detectMultipleContentBounds(image, false, threshold);
    
    emit processingProgress(50);
    
    QVector<QImage> splitImages;
    
    for (const QRect &rect : contentRects) {
        if (!rect.isEmpty()) {
            QImage part = image.copy(rect);
            // 裁剪每部分的空白
            part = autoTrimWhitespace(part, threshold);
            splitImages.append(part);
        }
    }
    
    emit processingProgress(100);
    qDebug() << "上下拆分完成，得到" << splitImages.size() << "个部分";
    
    return splitImages;
}
