#ifndef IMAGEPROCESSOR_H
#define IMAGEPROCESSOR_H

#include <QObject>
#include <QImage>
#include <QRect>

class ImageProcessor : public QObject
{
    Q_OBJECT

public:
    explicit ImageProcessor(QObject *parent = nullptr);
    ~ImageProcessor();

    // 基本图像处理函数
    QImage rotateLeft(const QImage &image);
    QImage rotateRight(const QImage &image);
    QImage crop(const QImage &image, const QRect &rect);
    QImage adjustBrightnessContrast(const QImage &image, int brightness, int contrast);
    QImage adjustHueSaturation(const QImage &image, int hue, int saturation);  // 新增：色度饱和度调整
    
    // 高级图像处理函数
    QImage autoLevel(const QImage &image);
    QImage sharpen(const QImage &image, int amount);
    QImage denoise(const QImage &image, int level);
    QImage convertToGrayscale(const QImage &image);
    QImage convertToBinary(const QImage &image, int threshold);
    
    // 文档增强函数
    QImage deskew(const QImage &image);
    QImage rotateByAngle(const QImage &image, double angle);  // 按指定角度旋转
    QImage removeBorder(const QImage &image, int borderWidth);
    QImage enhanceText(const QImage &image);
    QImage bleachBackground(const QImage &image, int threshold = 240, int strength = 80);
    
    // 空白页检测
    bool isBlankPage(const QImage &image, int threshold = 250, double blankRatio = 0.98);  // 检测是否为空白页
    
    // 图片合并
    QImage mergeImagesHorizontal(const QImage &image1, const QImage &image2, int spacing = 20);  // 左右合并
    QImage mergeImagesVertical(const QImage &image1, const QImage &image2, int spacing = 20);    // 上下合并
    
    // 智能裁剪
    QImage autoTrimWhitespace(const QImage &image, int threshold = 240);  // 自动裁剪空白边缘
    QRect detectContentBounds(const QImage &image, int threshold = 240);  // 检测内容边界
    
    // 智能拆分
    QVector<QImage> splitImageHorizontal(const QImage &image, int threshold = 240);  // 左右拆分
    QVector<QImage> splitImageVertical(const QImage &image, int threshold = 240);    // 上下拆分
    QVector<QRect> detectMultipleContentBounds(const QImage &image, bool isHorizontal, int threshold = 240);  // 检测多个内容区域

signals:
    void processingProgress(int percentage);
    void processingComplete(const QImage &processedImage);
    void processingError(const QString &errorMessage);

private:
    // 辅助函数
    QImage applyFilter(const QImage &image, const QVector<qreal> &kernel, int kernelSize);
    int calculateThreshold(const QImage &image);
    QImage applyThreshold(const QImage &image, int threshold);
    QImage applyGamma(const QImage &image, qreal gamma);
    
    // 倾斜校正相关辅助函数（增强版）
    double detectSkewAngle(const QImage &grayImage);
    double detectSkewAngleByLines(const QImage &grayImage);  // 基于线条检测
    double detectSkewAngleByText(const QImage &grayImage);   // 基于文字检测
    QImage cropRotatedImage(const QImage &rotatedImage, double angle);
    QImage detectEdges(const QImage &grayImage);
    
    // 文档增强相关辅助函数
    QImage enhanceDocumentColor(const QImage &image);  // 彩色文档增强
    QImage removeBackgroundKeepContent(const QImage &image);  // 去除背景保留内容
};

#endif // IMAGEPROCESSOR_H