#ifndef IMAGEPROCESSOR_H
#define IMAGEPROCESSOR_H

#include <QObject>
#include <QImage>
#include <QRect>

class ImageProcessor : public QObject
{
    Q_OBJECT

public:
    explicit ImageProcessor(QObject *parent = nullptr);
    ~ImageProcessor();

    // 基本图像处理函数
    QImage rotateLeft(const QImage &image);
    QImage rotateRight(const QImage &image);
    QImage crop(const QImage &image, const QRect &rect);
    QImage adjustBrightnessContrast(const QImage &image, int brightness, int contrast);
    
    // 高级图像处理函数
    QImage autoLevel(const QImage &image);
    QImage sharpen(const QImage &image, int amount);
    QImage denoise(const QImage &image, int level);
    QImage convertToGrayscale(const QImage &image);
    QImage convertToBinary(const QImage &image, int threshold);
    
    // 文档增强函数
    QImage deskew(const QImage &image);  // 倾斜校正（增强版）
    QImage removeBorder(const QImage &image, int borderWidth);
    QImage enhanceText(const QImage &image);  // 文档增强（保留彩色，突出内容）
    QImage bleachBackground(const QImage &image, int threshold = 240, int strength = 80);

signals:
    void processingProgress(int percentage);
    void processingComplete(const QImage &processedImage);
    void processingError(const QString &errorMessage);

private:
    // 辅助函数
    QImage applyFilter(const QImage &image, const QVector<qreal> &kernel, int kernelSize);
    int calculateThreshold(const QImage &image);
    QImage applyThreshold(const QImage &image, int threshold);
    QImage applyGamma(const QImage &image, qreal gamma);
    
    // 倾斜校正相关辅助函数（增强版）
    double detectSkewAngle(const QImage &grayImage);
    double detectSkewAngleByLines(const QImage &grayImage);  // 基于线条检测
    double detectSkewAngleByText(const QImage &grayImage);   // 基于文字检测
    QImage cropRotatedImage(const QImage &rotatedImage, double angle);
    QImage detectEdges(const QImage &grayImage);
    
    // 文档增强相关辅助函数
    QImage enhanceDocumentColor(const QImage &image);  // 彩色文档增强
    QImage removeBackgroundKeepContent(const QImage &image);  // 去除背景保留内容
};

#endif // IMAGEPROCESSOR_H
