#ifndef MAINWINDOW_H
#define MAINWINDOW_H

#include <QMainWindow>
#include <QImage>
#include <QLabel>
#include <QScrollArea>
#include <QToolBar>
#include <QAction>
#include <QComboBox>
#include <QSpinBox>
#include <QStatusBar>
#include <QProgressBar>
#include <QListWidget>
#include <QSplitter>
#include <QDockWidget>
#include <QCheckBox>
#include <QLineEdit>
#include <QPdfWriter>
#include <QPainter>
#include <QIcon>
#include <QPageSize>
#include <QPushButton>
#include <QGroupBox>
#include <QInputDialog>
#include <QTimer>
#include <QStack>

#include "scannercontroller.h"
#include "imageprocessor.h"
#include "documentmanager.h"
#include "croppableimagewidget.h"

#include <chrono>

// 合并模板结构
struct MergeTemplate {
    QString name;           // 模板名称
    QString description;    // 模板描述
    bool isHorizontal;      // 是否左右合并
    int spacing;            // 图片间距
    bool autoTrim;          // 是否自动裁剪
    int targetWidth;        // 目标宽度（0表示不调整）
    int targetHeight;       // 目标高度（0表示不调整）
    bool scaleToFit;        // 是否缩放以适应目标尺寸
};

class MainWindow : public QMainWindow
{
    Q_OBJECT

public:
    std::chrono::high_resolution_clock::time_point m_startTime;
    explicit MainWindow(QWidget *parent = nullptr);
    ~MainWindow();

protected:
    void closeEvent(QCloseEvent *event) override;
    void showEvent(QShowEvent *event) override;
    void resizeEvent(QResizeEvent *event) override;

private slots:
    // 扫描操作
    void onScanButtonClicked();
    void onPreviewButtonClicked();
    void onDelayScanButtonClicked();  // 新增：延时扫描按钮槽函数
    void onFinishDelayScanClicked();  // 新增：完成延时扫描按钮槽函数
    void onScannerSelected(int index);
    void onScanProgress(int percentage);
    void onScanComplete(const QImage &image);
    void onScanError(const QString &errorMessage);
    
    // 图像处理操作
    void onRotateLeftClicked();
    void onRotateRightClicked();
    void onCropClicked();
    void onCropCompleted(const QRect &rect);
    void onCropCancelled();
    void onAdjustBrightnessContrastClicked();
    void onAutoLevelClicked();
    void onConvertToGrayscaleClicked();
    void onConvertToBinaryClicked();
    
    // 高级设置操作
    void onAdvancedSettingsClicked();
    void onAutoDeskewClicked();  // 改为角度调整
    void onEnhanceTextClicked();
    void onRemoveBorderClicked();
    void onBleachBackgroundClicked();
    void onAdjustBrightnessContrastDialogClicked();  // 新增：亮度对比度调整
    void onAdjustHueSaturationClicked();  // 新增：色度饱和度调整
    void onUndoClicked();  // 新增：撤销操作
    void onRemoveBlankPagesToggled(bool enabled);  // 新增：去除空白页开关
    void onBlankPageSettingsClicked();  // 新增：空白页检测参数设置
    void onScanSettingsClicked();  // 新增：扫描设置
    void onLayoutSettingsClicked();  // 新增：界面布局设置
    
    // 布局管理
    void applyLayout1();  // 默认布局
    void applyLayout2();  // 反转布局
    
    // 文档操作
    void onNewDocument();
    void onOpenDocument();
    void onSaveDocument();
    void onSaveAsDocument();
    void onSaveMultiPageDocument();
    void onExportDocument();
    void onPageAdded(int index);
    void onPageRemoved(int index);
    void onPageMoved(int oldIndex, int newIndex);
    void onCurrentPageChanged(int index);
    void onPageSelected(int index);
    
    // 其他操作
    void onAboutClicked();
    void onClearAllClicked();
    void onPreferencesClicked();
    void updateStatusBar();
    void onSaveCurrentImage();
    void updateAutoSaveCounter();
    
    // ADF相关槽函数
    void onADFStatusChanged(bool hasDocument);
    void onADFEmptyWarning();
    void forceScanWithADF();
    
    // 多页扫描槽函数
    void onMultiPageScanStarted();
    void onMultiPageScanFinished(int totalPages);
    void onCurrentPageScanning(int pageNumber);
    
    // 右键菜单槽函数
    void showPagesListContextMenu(const QPoint &pos);
    void onAddImageFromFile();
    void onInsertImageFromFile();
    void onDeleteCurrentPage();
    void onClearAllPages();
    void onMovePageUp();
    void onMovePageDown();
    void onMergeImagesClicked();
    void onSplitImageHorizontal();
    void onSplitImageVertical();

private:
    QPushButton *m_findDevicesBtn; // 查找设备按钮引用
    
    // 初始化函数
    void createActions();
    void createMenus();
    void createToolbars();
    void createStatusBar();
    void createDockWidgets();
    void createCentralWidget();
    void connectSignals();
    void loadSettings();
    void saveSettings();
    
    // 辅助函数
    bool maybeSave();
    void updateWindowTitle();
    void updateActions();
    void displayImage(const QImage &image);
    void refreshScannerList();
    void selectSavePath();
    void updatePaperSourceOptions();
    
    // 延迟初始化函数
    bool ensureScannerControllerInitialized();
    void initializeScannerController();
    
    // UI组件
    QScrollArea *m_imageScrollArea;
    CroppableImageWidget *m_imageLabel;
    QToolBar *m_mainToolBar;
    QToolBar *m_scanToolBar;
    QToolBar *m_imageToolBar;
    QStatusBar *m_statusBar;
    QProgressBar *m_progressBar;
    QDockWidget *m_pagesDock;
    QListWidget *m_pagesList;
    QComboBox *m_scannerComboBox;
    QComboBox *m_resolutionComboBox;
    QComboBox *m_colorModeComboBox;
    QComboBox *m_pageSizeComboBox;
    QComboBox *m_paperSourceComboBox;
    QComboBox *m_duplexModeComboBox;
    QDoubleSpinBox *m_leftSpinBox;
    QDoubleSpinBox *m_topSpinBox;
    QDoubleSpinBox *m_widthSpinBox;
    QDoubleSpinBox *m_heightSpinBox;
    QSlider *m_qualitySlider;
    
    // 核心组件
    ScannerController *m_scannerController;
    ImageProcessor *m_imageProcessor;
    DocumentManager *m_documentManager;
    
    // 状态变量
    bool m_isScanning;
    QImage m_currentImage;
    QRect m_cropRect;
    bool m_isCropping;
    
    // 撤销功能相关
    QStack<QImage> m_undoStack;  // 撤销栈
    static const int MAX_UNDO_STEPS = 10;  // 最大撤销步数
    void saveToUndoStack(const QImage &image);  // 保存到撤销栈
    
    // 延时扫描相关
    QTimer *m_delayScanTimer;
    int m_delayScanRemaining;
    int m_delayScanInterval;  // 延时间隔（秒）
    bool m_isDelayScanMode;   // 是否处于延时扫描模式
    int m_delayScanCount;     // 已扫描的张数
    QPushButton *m_finishDelayScanBtn;  // 完成延时扫描按钮
    
    // 多页扫描状态
    bool m_isMultiPageScanning;
    QList<QImage> m_multiPageImages;
    
    // 保存设置相关
    QComboBox *m_formatCombo;
    QLineEdit *m_pathEdit;
    QCheckBox *m_autoSaveCheck;
    int m_saveCounter;
    
    // 空白页检测设置
    bool m_removeBlankPagesEnabled;  // 是否启用去除空白页功能
    int m_blankPageThreshold;        // 空白页检测亮度阈值（0-255）
    int m_blankPageRatio;            // 空白页检测比例阈值（0-100，表示百分比）
    
    // 扫描设置
    bool m_searchNetworkDevices;     // 是否搜索网络设备（false=仅USB，true=USB+局域网）
    
    // 保存相关方法
    QString generateFileName(const QString &ext);
    QString generateFileNameWithIndex(const QString &ext, int index);
    bool saveImage(const QImage &image, const QString &path);
    bool saveMultiPageAsPDF(const QList<QImage> &images, const QString &path);
    void performAutoSave(const QImage &image);
    void performMultiPageAutoSave();
    
    // 图像处理相关方法
    QImage performAutoDeskew(const QImage &image);
    bool isBlankPage(const QImage &image);  // 检测是否为空白页
};

#endif // MAINWINDOW_H