#include "scannercontroller.h"

#include <QApplication>
#include <QBuffer>
#include <QByteArray>
#include <QCoreApplication>
#include <QDebug>
#include <QImage>
#include <QMutexLocker>
#include <QPainter>
#include <QThread>
#include <QtConcurrent/QtConcurrentRun>
#include <cstring>

// SANE 回调函数
static SANE_Status sane_callback(SANE_Handle handle, SANE_Frame format,
                                 SANE_Byte *data, SANE_Int *length,
                                 void *userdata) {
  Q_UNUSED(handle);
  Q_UNUSED(format);
  Q_UNUSED(data);
  Q_UNUSED(length);
  Q_UNUSED(userdata);

  // 这个函数在SANE API中用于异步扫描，但我们使用同步方式，所以这里不需要实现
  return SANE_STATUS_GOOD;
}

ScannerController::ScannerController(QObject *parent)
    : QObject(parent), m_resolution(300), m_colorMode("Color"), m_left(0.0),
      m_top(0.0), m_width(215.9) // A4宽度，单位mm
      ,
      m_height(279.4) // A4高度，单位mm
      ,
      m_pageSize("A4"), m_saneInitialized(false), m_initState(NotInitialized),
      m_supportsADF(false), m_supportsDuplex(false), m_duplexMode("Simplex"),
      m_isMultiPageScanning(false), m_currentPageCount(0) {
  // 创建扫描工作线程
  m_scanWorker = new ScanWorker();
  m_scanWorker->moveToThread(&m_workerThread);

  // 连接信号和槽
  connect(this, &ScannerController::startScanWorker, m_scanWorker,
          &ScanWorker::doScan);
  connect(this, &ScannerController::cancelScanWorker, m_scanWorker,
          &ScanWorker::cancelScan);
  connect(m_scanWorker, &ScanWorker::scanProgress, this,
          &ScannerController::scanProgress);
  connect(m_scanWorker, &ScanWorker::scanComplete, this,
          &ScannerController::scanComplete);
  connect(m_scanWorker, &ScanWorker::scanError, this,
          &ScannerController::scanError);
  connect(m_scanWorker, &ScanWorker::multiPageScanStarted, this,
          &ScannerController::multiPageScanStarted);
  connect(m_scanWorker, &ScanWorker::multiPageScanFinished, this,
          &ScannerController::multiPageScanFinished);
  connect(m_scanWorker, &ScanWorker::currentPageScanning, this,
          &ScannerController::currentPageScanning);

  // 启动工作线程
  m_workerThread.start();

  qDebug() << "ScannerController创建完成，延迟初始化模式";
}

ScannerController::~ScannerController() {
  // 先关闭SANE，避免在线程运行时关闭
  closeSane();

  // 停止工作线程
  m_workerThread.quit();
  if (!m_workerThread.wait(3000)) {
    // 如果3秒内没有退出，强制终止
    m_workerThread.terminate();
    m_workerThread.wait(1000);
  }

  // 释放资源
  delete m_scanWorker;
  m_scanWorker = nullptr;

  qDebug() << "ScannerController已销毁";
}

bool ScannerController::initSane() {
  SANE_Int version_code;
  SANE_Status status = sane_init(&version_code, nullptr);

  if (status != SANE_STATUS_GOOD) {
    qDebug() << "SANE初始化失败: " << sane_strstatus(status);
    return false;
  }

  qDebug() << "SANE初始化成功，版本: " << SANE_VERSION_MAJOR(version_code)
           << "." << SANE_VERSION_MINOR(version_code) << "."
           << SANE_VERSION_BUILD(version_code);

  // 检查是否有可用的扫描仪设备
  const SANE_Device **device_list;
  status = sane_get_devices(&device_list, SANE_FALSE);

  if (status != SANE_STATUS_GOOD) {
    qDebug() << "获取扫描仪列表失败: " << sane_strstatus(status);
    return false;
  }

  if (!device_list || !device_list[0]) {
    qDebug() << "未找到扫描仪设备";
    return false;
  }

  return true;
}

void ScannerController::closeSane() {
  if (m_saneInitialized) {
    sane_exit();
    m_saneInitialized = false;
    qDebug() << "SANE已关闭";
  }
}

QStringList ScannerController::getAvailableScanners() {
  if (!m_cachedScanners.isEmpty()) {
    return m_cachedScanners;
  }

  return refreshScanners();
}

QStringList ScannerController::refreshScanners(bool includeNetwork) {
  // 如果SANE未初始化，返回空列表（异步初始化会处理）
  if (!m_saneInitialized) {
    qDebug() << "SANE未初始化，等待异步初始化完成...";
    return QStringList();
  }
  
  // 清除缓存，强制重新搜索（确保切换模式时能正确搜索）
  static bool lastIncludeNetwork = false;
  bool modeChanged = (lastIncludeNetwork != includeNetwork);
  
  if (modeChanged) {
    qDebug() << "搜索模式改变，清除缓存并重新初始化 SANE";
    m_cachedScanners.clear();
    m_displayNameToDeviceName.clear();
    
    // 重新初始化 SANE 以确保读取最新的设备列表
    // 这对于网络设备特别重要，因为 SANE 可能需要重新扫描网络
    if (m_saneInitialized) {
      sane_exit();
      SANE_Int version_code;
      SANE_Status init_status = sane_init(&version_code, nullptr);
      if (init_status == SANE_STATUS_GOOD) {
        m_saneInitialized = true;
        qDebug() << "SANE 重新初始化成功";
      } else {
        qDebug() << "SANE 重新初始化失败:" << sane_strstatus(init_status);
        m_saneInitialized = false;
        return QStringList();
      }
    }
  }
  lastIncludeNetwork = includeNetwork;

  QStringList scanners;

  const SANE_Device **device_list;
  // 参考 deepin-scanner：总是搜索所有设备（包括网络），然后在应用层过滤
  // 这样可以确保 SANE 返回所有可用设备
  qDebug() << "搜索扫描仪设备，模式:" << (includeNetwork ? "USB+局域网" : "仅USB");
  SANE_Status status = sane_get_devices(&device_list, SANE_FALSE);

  if (status != SANE_STATUS_GOOD) {
    qDebug() << "获取扫描仪列表失败: " << sane_strstatus(status);
    return scanners;
  }

  if (!device_list || !device_list[0]) {
    qDebug() << "未找到扫描仪设备";
    return scanners;
  }

  int totalDevices = 0;
  for (int i = 0; device_list[i] != nullptr; ++i) {
    totalDevices++;
  }
  qDebug() << "========================================";
  qDebug() << "SANE返回设备总数:" << totalDevices;
  qDebug() << "搜索模式:" << (includeNetwork ? "USB+局域网" : "仅USB");
  qDebug() << "========================================";

  for (int i = 0; device_list[i] != nullptr; ++i) {
    QString name = QString::fromUtf8(device_list[i]->name);
    QString model = QString::fromUtf8(device_list[i]->model);
    QString vendor = QString::fromUtf8(device_list[i]->vendor);
    QString type = QString::fromUtf8(device_list[i]->type);

    qDebug() << "";
    qDebug() << "========== 设备" << (i+1) << "/" << totalDevices << "==========";
    qDebug() << "原始名称:" << name;
    qDebug() << "型号:" << model;
    qDebug() << "厂商:" << vendor;

    // 判断是否为网络设备
    // 注意：localhost 的 eSCL 设备实际上是本地USB设备，不算网络设备
    bool isNetwork = false;
    if (name.contains(":tcp ") || name.contains("net:")) {
      // TCP 连接或 net: 前缀，肯定是网络设备
      isNetwork = true;
      qDebug() << "判断为网络设备（TCP/net）";
    } else if (name.contains("http:") || name.contains("https:")) {
      // HTTP/HTTPS 连接，但需要排除 localhost
      if (!name.contains("localhost") && !name.contains("127.0.0.1")) {
        isNetwork = true;
        qDebug() << "判断为网络设备（HTTP/HTTPS，非localhost）";
      } else {
        qDebug() << "判断为本地设备（localhost）";
      }
    } else if (name.contains("airscan:")) {
      // AirScan 设备，需要检查是否是 localhost
      if (!name.contains("localhost") && !name.contains("127.0.0.1")) {
        isNetwork = true;
        qDebug() << "判断为网络设备（AirScan，非localhost）";
      } else {
        qDebug() << "判断为本地设备（AirScan localhost）";
      }
    } else {
      qDebug() << "判断为本地设备（其他）";
    }
    
    // 如果不搜索网络设备，则跳过
    if (isNetwork && !includeNetwork) {
      qDebug() << "❌ 跳过网络设备（当前模式不包括网络）";
      continue;
    }
    
    if (isNetwork) {
      qDebug() << "✓ 添加网络设备";
    } else {
      qDebug() << "✓ 添加本地设备";
    }

    // 格式化显示名称
    QString displayName;
    
    // 根据 vendor 和 model 构建显示名称
    if (!vendor.isEmpty() && !model.isEmpty()) {
      if (model.startsWith(vendor)) {
        // 型号已包含品牌名，直接使用型号
        displayName = model;
      } else {
        // 组合品牌和型号
        displayName = QString("%1 %2").arg(vendor, model);
      }
    } else if (!model.isEmpty()) {
      // 只有型号，使用型号
      displayName = model;
    } else {
      // 都没有，使用设备名称
      displayName = name;
    }
    
    // 为网络设备添加标识
    if (isNetwork) {
      displayName = QString("[网络] %1").arg(displayName);
    }

    // 清理显示名称
    QString cleanDisplayName = displayName;
    
    // 对于本地设备，清理不必要的技术细节（如 libusb 地址）
    // 对于网络设备，保留完整的连接信息（IP、端口等）
    if (!isNetwork) {
      // 清理本地设备的 libusb 地址等技术细节
      // 例如：Pantum M6600 series (libusb:001:007) -> Pantum M6600 series
      if (cleanDisplayName.contains("(libusb:") || 
          cleanDisplayName.contains("(USB)")) {
        int firstParenIndex = cleanDisplayName.indexOf('(');
        if (firstParenIndex > 0) {
          cleanDisplayName = cleanDisplayName.left(firstParenIndex).trimmed();
        }
      }
    }
    // 网络设备保留所有信息，不做清理

    // 如果清理后名称为空，使用原始显示名称
    if (cleanDisplayName.isEmpty()) {
      cleanDisplayName = displayName;
    }

    // 检查是否已存在相同的显示名称
    if (m_displayNameToDeviceName.contains(cleanDisplayName)) {
      qDebug() << "⚠️ 警告：发现重复的显示名称:" << cleanDisplayName;
      qDebug() << "  旧设备: " << m_displayNameToDeviceName[cleanDisplayName];
      qDebug() << "  新设备: " << name;
      // 为重复的设备添加序号
      int count = 2;
      QString uniqueName = QString("%1 (%2)").arg(cleanDisplayName).arg(count);
      while (m_displayNameToDeviceName.contains(uniqueName)) {
        count++;
        uniqueName = QString("%1 (%2)").arg(cleanDisplayName).arg(count);
      }
      cleanDisplayName = uniqueName;
      qDebug() << "  使用唯一名称: " << cleanDisplayName;
    }
    
    // 只显示清理后的名称，不显示技术细节
    scanners.append(cleanDisplayName);
    // 建立显示名称到设备名称的映射
    m_displayNameToDeviceName[cleanDisplayName] = name;
    qDebug() << "找到扫描仪:";
    qDebug() << "  显示名称: \"" << cleanDisplayName << "\"";
    qDebug() << "  原始名称: \"" << displayName << "\"";
    qDebug() << "  设备名称: \"" << name << "\"";
    qDebug() << "  类型: \"" << type << "\"";
  }

  // 缓存结果
  m_availableScanners = scanners;
  m_cachedScanners = scanners;
  
  qDebug() << "";
  qDebug() << "========================================";
  qDebug() << "扫描仪列表已更新并缓存:" << scanners.size() << "个设备";
  qDebug() << "最终设备列表:";
  for (int i = 0; i < scanners.size(); ++i) {
    qDebug() << "  " << (i+1) << ". " << scanners[i];
  }
  qDebug() << "========================================";
  return scanners;
}

bool ScannerController::isScannerAvailable(const QString &deviceName) {
  // 检查设备名称是否在映射表的值中
  return m_displayNameToDeviceName.values().contains(deviceName);
}

bool ScannerController::selectScanner(const QString &displayName) {
  qDebug() << "尝试选择扫描仪:" << displayName;
  qDebug() << "可用扫描仪列表:" << m_availableScanners;
  qDebug() << "设备名称映射:" << m_displayNameToDeviceName;
  
  if (!m_availableScanners.contains(displayName)) {
    qDebug() << "无效的扫描仪名称: " << displayName;
    return false;
  }
  
  // 使用显示名称查找实际设备名称
  QString deviceName = m_displayNameToDeviceName.value(displayName, displayName);
  qDebug() << "映射的设备名称:" << deviceName;
  
  // 尝试打开设备来验证可用性
  SANE_Handle testHandle = nullptr;
  SANE_Status status = sane_open(deviceName.toUtf8().constData(), &testHandle);
  
  if (status != SANE_STATUS_GOOD || !testHandle) {
    qDebug() << "无法打开设备:" << deviceName << "错误:" << sane_strstatus(status);
    
    // 尝试使用显示名称作为设备名称
    if (deviceName != displayName) {
      qDebug() << "尝试使用显示名称作为设备名称:" << displayName;
      status = sane_open(displayName.toUtf8().constData(), &testHandle);
      if (status == SANE_STATUS_GOOD && testHandle) {
        deviceName = displayName;
        qDebug() << "使用显示名称成功打开设备";
      }
    }
    
    if (status != SANE_STATUS_GOOD || !testHandle) {
      qDebug() << "设备完全不可用:" << displayName;
      return false;
    }
  }
  
  // 关闭测试句柄
  if (testHandle) {
    sane_close(testHandle);
  }
  
  // 设置当前扫描仪
  m_currentScanner = deviceName;
  qDebug() << "成功选择扫描仪:" << displayName << "(设备:" << deviceName << ")";

  // 检测ADF支持
  detectADFSupport();
  
  // 强制刷新设备参数
  updateScannerParameters();
  
  qDebug() << "ADF支持:" << (m_supportsADF ? "是" : "否");
  qDebug() << "双面支持:" << (m_supportsDuplex ? "是" : "否");
  
  return true;
}

void ScannerController::detectADFSupport() {
  m_supportsADF = false;
  m_supportsDuplex = false;
  m_availablePaperSources.clear();
  m_availableDuplexModes.clear();
  m_availablePaperSources << "Flatbed"; // 默认支持平板
  m_availableDuplexModes << "Simplex"; // 默认单面

  if (!m_saneInitialized || m_currentScanner.isEmpty()) {
    return;
  }

  // 提取设备名称
  QString deviceName;
  int startPos = m_currentScanner.lastIndexOf('(');
  int endPos = m_currentScanner.lastIndexOf(')');
  if (startPos >= 0 && endPos > startPos) {
    deviceName = m_currentScanner.mid(startPos + 1, endPos - startPos - 1);
  } else {
    deviceName = m_currentScanner;
  }

  // 打开设备检查支持的纸张来源
  SANE_Handle deviceHandle = nullptr;
  SANE_Status status =
      sane_open(deviceName.toUtf8().constData(), &deviceHandle);

  if (status != SANE_STATUS_GOOD || !deviceHandle) {
    qDebug() << "无法打开设备进行ADF检测: " << sane_strstatus(status);
    return;
  }

  // 获取选项数量
  SANE_Int num_options = 0;
  status = sane_control_option(deviceHandle, 0, SANE_ACTION_GET_VALUE,
                               &num_options, nullptr);

  if (status == SANE_STATUS_GOOD && num_options > 0) {
    qDebug() << "ADF检测: 设备有" << num_options << "个选项";
    
    // 遍历选项查找纸张来源和双面扫描选项
    for (SANE_Int i = 1; i < num_options; ++i) {
      const SANE_Option_Descriptor *opt =
          sane_get_option_descriptor(deviceHandle, i);

      if (!opt || !opt->name)
        continue;

      QString optName;
      try {
        optName = QString::fromUtf8(opt->name);
      } catch (...) {
        qDebug() << "ADF检测: 无法读取选项" << i << "的名称";
        continue;
      }

      // 检查纸张来源选项
      if ((optName == "source" || optName == "adf" || optName == "feeder") &&
          opt->type == SANE_TYPE_STRING &&
          opt->constraint_type == SANE_CONSTRAINT_STRING_LIST) {

        // 检查选项是否活动（某些设备的选项可能是不活动的）
        // SANE_CAP_INACTIVE = (1 << 5) = 32
        bool isActive = !((opt->cap) & (1 << 5));
        qDebug() << "纸张来源选项" << optName << "活动状态:" << isActive;

        const SANE_String_Const *string_list = opt->constraint.string_list;
        if (string_list) {
          // 限制最大循环次数，防止无限循环或越界访问
          for (int j = 0; j < 50 && string_list[j] != nullptr; ++j) {
            QString source;
            try {
              source = QString::fromUtf8(string_list[j]);
            } catch (...) {
              qDebug() << "读取纸张来源选项时出错，跳过";
              continue;
            }
            
            qDebug() << "发现纸张来源选项: " << source << "(活动:" << isActive << ")";

            if (source.contains("ADF", Qt::CaseInsensitive) ||
                source.contains("Feeder", Qt::CaseInsensitive) ||
                source.contains("Automatic", Qt::CaseInsensitive)) {
              // 只有当选项活动时才标记为支持ADF
              if (isActive) {
                m_supportsADF = true;
              }
              if (!m_availablePaperSources.contains(source)) {
                m_availablePaperSources << source;
              }
            } else if (source.contains("Flatbed", Qt::CaseInsensitive)) {
              if (!m_availablePaperSources.contains(source)) {
                m_availablePaperSources.removeAll("Flatbed");
                m_availablePaperSources.prepend(source);
              }
            }
          }
        }
      }
      
      // 检查双面扫描选项
      if ((optName == "duplex" || optName == "adf-duplex" || optName == "duplex-mode" || 
           optName == "double-sided" || optName == "both-sides") &&
          opt->type == SANE_TYPE_STRING &&
          opt->constraint_type == SANE_CONSTRAINT_STRING_LIST) {

        // 检查选项是否活动
        bool isActive = !((opt->cap) & (1 << 5));
        qDebug() << "双面扫描选项" << optName << "活动状态:" << isActive;

        const SANE_String_Const *string_list = opt->constraint.string_list;
        if (string_list && isActive) {  // 只处理活动的选项
          m_supportsDuplex = true;
          m_availableDuplexModes.clear(); // 清除默认值
          
          // 限制最大循环次数，防止无限循环或越界访问
          for (int j = 0; j < 50 && string_list[j] != nullptr; ++j) {
            QString duplexMode;
            try {
              duplexMode = QString::fromUtf8(string_list[j]);
            } catch (...) {
              qDebug() << "读取双面扫描选项时出错，跳过";
              continue;
            }
            
            qDebug() << "发现双面扫描选项: " << duplexMode;
            m_availableDuplexModes << duplexMode;
          }
        }
      }
      
      // 检查布尔型双面扫描选项
      if ((optName == "duplex" || optName == "adf-duplex" || optName == "double-sided") &&
          opt->type == SANE_TYPE_BOOL) {
        // 检查选项是否活动
        bool isActive = !((opt->cap) & (1 << 5));
        qDebug() << "布尔型双面扫描选项" << optName << "活动状态:" << isActive;
        
        if (isActive) {
          m_supportsDuplex = true;
          m_availableDuplexModes.clear();
          m_availableDuplexModes << "False" << "True";
          qDebug() << "发现布尔型双面扫描选项: " << optName;
        }
      }
    }
  }

  sane_close(deviceHandle);

  qDebug() << "检测到的纸张来源: " << m_availablePaperSources;
  qDebug() << "ADF支持状态: " << m_supportsADF;
  qDebug() << "检测到的双面扫描模式: " << m_availableDuplexModes;
  qDebug() << "双面扫描支持状态: " << m_supportsDuplex;
}

QString ScannerController::getCurrentScanner() const {
  return m_currentScanner;
}

void ScannerController::setResolution(int dpi) {
  m_resolution = dpi;
  qDebug() << "分辨率设置为: " << dpi << " dpi";
}

void ScannerController::setColorMode(const QString &mode) {
  m_colorMode = mode;
  qDebug() << "颜色模式设置为: " << mode;
}

void ScannerController::setScanArea(double left, double top, double width,
                                    double height) {
  m_left = left;
  m_top = top;
  m_width = width;
  m_height = height;
  qDebug() << "扫描区域设置为: " << left << "," << top << "," << width << ","
           << height;
}

void ScannerController::setPageSize(const QString &size) {
  m_pageSize = size;

  // 根据页面大小设置扫描区域
  if (size == "A4") {
    m_width = 215.9;  // mm
    m_height = 279.4; // mm
  } else if (size == "A5") {
    m_width = 148.0;  // mm
    m_height = 210.0; // mm
  } else if (size == "Letter") {
    m_width = 215.9;  // mm
    m_height = 279.4; // mm
  } else if (size == "Legal") {
    m_width = 215.9;  // mm
    m_height = 355.6; // mm
  } else {
    qDebug() << "无效的纸张大小: " << size;
    return;
  }

  qDebug() << "页面大小设置为: " << size << ", 尺寸: " << m_width << "x"
           << m_height << " mm";
}

void ScannerController::setPaperSource(const QString &source) {
  m_paperSource = source;
  qDebug() << "纸张来源设置为: " << source;
}

bool ScannerController::startScan(const QString &scannerName,
                                  const QString &colorMode, int resolution,
                                  const QString &paperSource,
                                  const QString &duplexMode) {
  // 确保SANE已初始化
  if (!m_saneInitialized) {
    qDebug() << "SANE未初始化，尝试初始化...";
    ensureInitialized();
    if (!m_saneInitialized) {
      emit scanError(tr("扫描仪初始化失败"));
      return false;
    }
  }

  // 如果提供了扫描仪名称，则选择该扫描仪
  if (!scannerName.isEmpty()) {
    selectScanner(scannerName);
  }

  // 如果没有选择扫描仪，则使用第一个可用的扫描仪
  if (m_currentScanner.isEmpty() && !m_availableScanners.isEmpty()) {
    m_currentScanner = m_availableScanners.first();
  }

  // 如果仍然没有扫描仪，则返回错误
  if (m_currentScanner.isEmpty()) {
    emit scanError(tr("未选择扫描仪"));
    return false;
  }

  // 设置颜色模式和分辨率（如果提供）
  QString actualColorMode = colorMode.isEmpty() ? m_colorMode : colorMode;
  int actualResolution = resolution > 0 ? resolution : m_resolution;

  // 设置纸张来源（如果提供）
  QString actualPaperSource =
      paperSource.isEmpty() ? m_paperSource : paperSource;
      
  // 设置双面扫描模式（如果提供）
  QString actualDuplexMode =
      duplexMode.isEmpty() ? m_duplexMode : duplexMode;
      
  // 如果用户选择了双面自动进纸器，自动启用双面扫描
  if (actualPaperSource == tr("双面自动进纸器") && 
      (actualDuplexMode.isEmpty() || actualDuplexMode == "Simplex")) {
    actualDuplexMode = "Duplex";
    qDebug() << "检测到双面自动进纸器，自动启用双面扫描模式";
  }

  // 检查ADF使用情况
  bool isUsingADF = (actualPaperSource == tr("自动进纸器") ||
                     actualPaperSource == tr("双面自动进纸器") ||
                     actualPaperSource.contains("ADF", Qt::CaseInsensitive) ||
                     actualPaperSource.contains("Feeder", Qt::CaseInsensitive));

  if (isUsingADF) {
    if (!m_supportsADF) {
      emit scanError(tr("当前扫描仪不支持自动进纸器功能"));
      return false;
    }

    qDebug() << "使用ADF模式，检查纸张状态...";

    // 检查ADF中是否有纸张
    bool hasDocument = hasDocumentInADF();
    if (!hasDocument) {
      qDebug() << "ADF纸张检测失败，但继续尝试扫描（可能是检测不准确）";
      // 不直接返回错误，而是发出警告信号，让用户决定
      emit adfStatusChanged(false);
    } else {
      qDebug() << "检测到ADF中有纸张，开始扫描";
      emit adfStatusChanged(true);
    }
  }

  // 提取设备名称（从显示名称中）
  QString deviceName;
  int startPos = m_currentScanner.lastIndexOf('(');
  int endPos = m_currentScanner.lastIndexOf(')');
  if (startPos >= 0 && endPos > startPos) {
    deviceName = m_currentScanner.mid(startPos + 1, endPos - startPos - 1);
  } else {
    deviceName = m_currentScanner;
  }

  qDebug() << "使用的设备名称:" << deviceName;
  qDebug() << "使用的纸张来源:" << actualPaperSource;

  // 启动扫描工作线程
  emit startScanWorker(deviceName, actualColorMode, actualResolution, false,
                       actualPaperSource, actualDuplexMode, m_width, m_height);

  return true;
}

bool ScannerController::startPreview(const QString &scannerName) {
  // 确保SANE已初始化
  if (!m_saneInitialized) {
    ensureInitialized();
    emit scanError(tr("扫描仪正在初始化中，请稍后再试"));
    return false;
  }

  // 预览使用较低的分辨率
  int previewResolution = 75;

  // 如果提供了扫描仪名称，则选择该扫描仪
  if (!scannerName.isEmpty()) {
    selectScanner(scannerName);
  }

  // 如果没有选择扫描仪，则使用第一个可用的扫描仪
  if (m_currentScanner.isEmpty() && !m_availableScanners.isEmpty()) {
    m_currentScanner = m_availableScanners.first();
  }

  // 如果仍然没有扫描仪，则返回错误
  if (m_currentScanner.isEmpty()) {
    emit scanError(tr("未选择扫描仪"));
    return false;
  }

  // 提取设备名称（从显示名称中）
  QString deviceName;
  int startPos = m_currentScanner.lastIndexOf('(');
  int endPos = m_currentScanner.lastIndexOf(')');
  if (startPos >= 0 && endPos > startPos) {
    deviceName = m_currentScanner.mid(startPos + 1, endPos - startPos - 1);
  } else {
    deviceName = m_currentScanner;
  }

  // 启动预览扫描工作线程
  emit startScanWorker(deviceName, m_colorMode, previewResolution, true,
                       QString(), QString(), m_width, m_height);

  return true;
}

void ScannerController::cancelScan() { emit cancelScanWorker(); }

bool ScannerController::supportsADF() const { return m_supportsADF; }

bool ScannerController::hasDocumentInADF() const {
  if (!m_supportsADF || !m_saneInitialized || m_currentScanner.isEmpty()) {
    qDebug() << "ADF检查失败: supportsADF=" << m_supportsADF
             << ", saneInitialized=" << m_saneInitialized
             << ", currentScanner=" << m_currentScanner;
    return false;
  }

  // 提取设备名称
  QString deviceName;
  int startPos = m_currentScanner.lastIndexOf('(');
  int endPos = m_currentScanner.lastIndexOf(')');
  if (startPos >= 0 && endPos > startPos) {
    deviceName = m_currentScanner.mid(startPos + 1, endPos - startPos - 1);
  } else {
    deviceName = m_currentScanner;
  }

  qDebug() << "检查ADF纸张状态，设备名称: " << deviceName;

  // 打开设备检查ADF状态
  SANE_Handle deviceHandle = nullptr;
  SANE_Status status =
      sane_open(deviceName.toUtf8().constData(), &deviceHandle);

  if (status != SANE_STATUS_GOOD || !deviceHandle) {
    qDebug() << "无法打开设备进行ADF纸张检测: " << sane_strstatus(status);
    // 如果无法检测纸张状态，假设有纸张（避免阻止用户扫描）
    return true;
  }

  bool hasDocument = false;
  bool foundPaperSensor = false;

  // 获取选项数量
  SANE_Int num_options;
  status = sane_control_option(deviceHandle, 0, SANE_ACTION_GET_VALUE,
                               &num_options, nullptr);

  if (status == SANE_STATUS_GOOD) {
    qDebug() << "设备选项数量: " << num_options;

    // 遍历选项查找ADF状态
    for (SANE_Int i = 1; i < num_options; ++i) {
      const SANE_Option_Descriptor *opt =
          sane_get_option_descriptor(deviceHandle, i);

      if (!opt || !opt->name)
        continue;

      QString optName = QString::fromUtf8(opt->name);

      // 检查多种可能的纸张检测选项
      if (optName.contains("adf", Qt::CaseInsensitive) ||
          optName.contains("feeder", Qt::CaseInsensitive) ||
          optName.contains("document", Qt::CaseInsensitive) ||
          optName.contains("paper", Qt::CaseInsensitive) ||
          optName.contains("loaded", Qt::CaseInsensitive) ||
          optName.contains("ready", Qt::CaseInsensitive) ||
          optName.contains("sensor", Qt::CaseInsensitive)) {

        foundPaperSensor = true;
        qDebug() << "找到纸张检测选项: " << optName << ", 类型: " << opt->type;

        if (opt->type == SANE_TYPE_BOOL) {
          SANE_Bool value;
          status = sane_control_option(deviceHandle, i, SANE_ACTION_GET_VALUE,
                                       &value, nullptr);
          if (status == SANE_STATUS_GOOD) {
            qDebug() << "选项 " << optName
                     << " 值: " << (value == SANE_TRUE ? "TRUE" : "FALSE");
            if (value == SANE_TRUE) {
              hasDocument = true;
              break;
            }
          } else {
            qDebug() << "读取选项失败: " << sane_strstatus(status);
          }
        } else if (opt->type == SANE_TYPE_INT) {
          SANE_Int value;
          status = sane_control_option(deviceHandle, i, SANE_ACTION_GET_VALUE,
                                       &value, nullptr);
          if (status == SANE_STATUS_GOOD) {
            qDebug() << "选项 " << optName << " 值: " << value;
            if (value > 0) {
              hasDocument = true;
              break;
            }
          }
        }
      }
    }
  }

  sane_close(deviceHandle);

  // 如果没有找到纸张传感器，假设有纸张（兼容性处理）
  if (!foundPaperSensor) {
    qDebug() << "未找到纸张传感器，假设ADF中有纸张";
    hasDocument = true;
  }

  qDebug() << "ADF纸张检测结果: " << (hasDocument ? "有纸张" : "无纸张");
  return hasDocument;
}

QStringList ScannerController::getAvailablePaperSources() const {
  return m_availablePaperSources;
}

bool ScannerController::supportsDuplex() const {
  return m_supportsDuplex;
}

QStringList ScannerController::getAvailableDuplexModes() const {
  return m_availableDuplexModes;
}

void ScannerController::setDuplexMode(const QString &mode) {
  m_duplexMode = mode;
  qDebug() << "双面扫描模式设置为: " << mode;
}

void ScannerController::debugScannerOptions() const {
  if (!m_saneInitialized || m_currentScanner.isEmpty()) {
    qDebug() << "无法调试扫描仪选项：SANE未初始化或未选择扫描仪";
    return;
  }

  // 提取设备名称
  QString deviceName;
  int startPos = m_currentScanner.lastIndexOf('(');
  int endPos = m_currentScanner.lastIndexOf(')');
  if (startPos >= 0 && endPos > startPos) {
    deviceName = m_currentScanner.mid(startPos + 1, endPos - startPos - 1);
  } else {
    deviceName = m_currentScanner;
  }

  // 打开设备
  SANE_Handle deviceHandle = nullptr;
  SANE_Status status =
      sane_open(deviceName.toUtf8().constData(), &deviceHandle);

  if (status != SANE_STATUS_GOOD || !deviceHandle) {
    qDebug() << "无法打开设备进行调试: " << sane_strstatus(status);
    return;
  }

  qDebug() << "=== 扫描仪选项调试信息 ===";
  qDebug() << "设备: " << deviceName;

  // 获取选项数量
  SANE_Int num_options = 0;
  status = sane_control_option(deviceHandle, 0, SANE_ACTION_GET_VALUE,
                               &num_options, nullptr);

  if (status == SANE_STATUS_GOOD && num_options > 0) {
    qDebug() << "总选项数: " << num_options;

    for (SANE_Int i = 1; i < num_options; ++i) {
      const SANE_Option_Descriptor *opt =
          sane_get_option_descriptor(deviceHandle, i);

      if (!opt || !opt->name)
        continue;

      // 安全地获取选项名称
      QString optName;
      try {
        optName = QString::fromUtf8(opt->name);
      } catch (...) {
        qDebug() << QString("选项 %1: 无法读取名称").arg(i);
        continue;
      }

      QString optTitle = opt->title ? QString::fromUtf8(opt->title) : "无标题";
      QString optDesc = opt->desc ? QString::fromUtf8(opt->desc) : "无描述";

      qDebug() << QString("选项 %1: %2").arg(i).arg(optName);
      qDebug() << QString("  标题: %1").arg(optTitle);
      qDebug() << QString("  描述: %1").arg(optDesc);
      qDebug() << QString("  类型: %1").arg(opt->type);
      qDebug() << QString("  约束类型: %1").arg(opt->constraint_type);

      // 如果是字符串列表，显示可用选项
      if (opt->type == SANE_TYPE_STRING &&
          opt->constraint_type == SANE_CONSTRAINT_STRING_LIST &&
          opt->constraint.string_list != nullptr) {
        const SANE_String_Const *string_list = opt->constraint.string_list;
        qDebug() << "  可用选项:";
        try {
          for (int j = 0; j < 100 && string_list[j] != nullptr; ++j) {
            QString value = QString::fromUtf8(string_list[j]);
            qDebug() << QString("    - %1").arg(value);
          }
        } catch (...) {
          qDebug() << "    (读取选项列表时出错)";
        }
      }

      qDebug() << "";
    }
  }

  sane_close(deviceHandle);
  qDebug() << "=== 调试信息结束 ===";
}

// ScanWorker 实现

ScanWorker::ScanWorker(QObject *parent)
    : QObject(parent), m_deviceHandle(nullptr), m_scanning(false),
      m_cancelled(false) {}

ScanWorker::~ScanWorker() { closeDevice(); }

void ScanWorker::doScan(const QString &deviceName, const QString &colorMode,
                        int resolution, bool isPreview,
                        const QString &paperSource, const QString &duplexMode,
                        double width, double height) {
  QMutexLocker locker(&m_mutex);

  // 重置取消标志
  m_cancelled = false;

  // 打开设备
  if (!openDevice(deviceName)) {
    emit scanError(tr("无法打开扫描仪设备: %1").arg(deviceName));
    return;
  }

  // 设置扫描参数
  if (!setOptions(colorMode, resolution, isPreview, paperSource, duplexMode, width,
                  height)) {
    emit scanError(tr("无法设置扫描参数"));
    closeDevice();
    return;
  }

  // 检查是否使用ADF
  bool isUsingADF =
      (paperSource.contains("自动进纸器", Qt::CaseInsensitive) ||
       paperSource.contains("双面自动进纸器", Qt::CaseInsensitive) ||
       paperSource.contains("ADF", Qt::CaseInsensitive) ||
       paperSource.contains("Feeder", Qt::CaseInsensitive) ||
       paperSource.contains("Automatic", Qt::CaseInsensitive));

  m_scanning = true;

  if (isUsingADF && !isPreview) {
    // ADF多页扫描模式
    performMultiPageScan();
  } else {
    // 单页扫描模式（平板或预览）
    QImage scannedImage = performScan();

    if (!m_cancelled && !scannedImage.isNull()) {
      emit scanComplete(scannedImage);
    } else if (!m_cancelled) {
      emit scanError(tr("扫描失败或被取消"));
    }
  }

  m_scanning = false;

  // 关闭设备
  closeDevice();
}

void ScanWorker::cancelScan() {
  QMutexLocker locker(&m_mutex);

  if (m_scanning) {
    m_cancelled = true;
    // 唤醒可能在等待的线程
    m_condition.wakeAll();
  }
}

bool ScanWorker::openDevice(const QString &deviceName) {
  // 确保设备已关闭
  closeDevice();

  // 打开设备
  SANE_Status status =
      sane_open(deviceName.toUtf8().constData(), &m_deviceHandle);

  if (status != SANE_STATUS_GOOD) {
    qDebug() << "打开设备失败: " << sane_strstatus(status);
    m_deviceHandle = nullptr;
    return false;
  }

  qDebug() << "成功打开设备: " << deviceName;
  return true;
}

bool ScanWorker::setOptions(const QString &colorMode, int resolution,
                            bool isPreview, const QString &paperSource,
                            const QString &duplexMode, double width, double height) {
  if (!m_deviceHandle) {
    qDebug() << "设备未打开，无法设置选项";
    return false;
  }

  // 使用传入的纸张来源参数
  QString actualPaperSource = paperSource.isEmpty() ? "Flatbed" : paperSource;

  // 获取选项描述符
  SANE_Int num_options;
  SANE_Status status = sane_control_option(
      m_deviceHandle, 0, SANE_ACTION_GET_VALUE, &num_options, nullptr);

  if (status != SANE_STATUS_GOOD) {
    qDebug() << "获取选项数量失败: " << sane_strstatus(status);
    return false;
  }

  qDebug() << "设备选项数量: " << num_options;
  qDebug() << "目标纸张来源: " << actualPaperSource;

  // 遍历所有选项
  for (SANE_Int i = 1; i < num_options; ++i) {
    const SANE_Option_Descriptor *opt =
        sane_get_option_descriptor(m_deviceHandle, i);

    if (!opt) {
      continue;
    }

    QString optName = opt->name ? QString::fromUtf8(opt->name) : QString();

    // 调试：输出所有选项名称
    if (!optName.isEmpty()) {
      qDebug() << "发现选项:" << optName << "类型:" << opt->type;
    }

    // 设置分辨率
    if (optName == "resolution" && opt->type == SANE_TYPE_INT) {
      SANE_Int dpi = resolution;
      SANE_Status res = sane_control_option(
          m_deviceHandle, i, SANE_ACTION_SET_VALUE, &dpi, nullptr);

      if (res == SANE_STATUS_GOOD) {
        qDebug() << "分辨率设置为: " << dpi << " dpi";
      } else {
        qDebug() << "设置分辨率失败: " << sane_strstatus(res);
      }
    }

    // 设置颜色模式 - 修复Brother和Canon扫描仪兼容性
    if ((optName == "mode" || optName == "color-mode" || 
         optName == "scan-mode" || optName == "colour-mode") &&
        opt->type == SANE_TYPE_STRING) {
      SANE_String mode = nullptr;

      // 首先检查设备支持的颜色模式
      if (opt->constraint_type == SANE_CONSTRAINT_STRING_LIST) {
        const SANE_String_Const *string_list = opt->constraint.string_list;
        if (string_list) {
          // 根据用户选择查找匹配的模式
          if (colorMode == "彩色" || colorMode == "Color") {
            // 查找彩色模式
            for (int j = 0; string_list[j] != nullptr; ++j) {
              QString availableMode = QString::fromUtf8(string_list[j]);
              if (availableMode.contains("Color", Qt::CaseInsensitive) ||
                  availableMode.contains("24bit", Qt::CaseInsensitive)) {
                mode = const_cast<SANE_String>(string_list[j]);
                qDebug() << "找到彩色模式: " << availableMode;
                break;
              }
            }
          } else if (colorMode == "灰度" || colorMode == "Gray") {
            // 查找灰度模式 - 扩展匹配规则以支持更多扫描仪
            for (int j = 0; string_list[j] != nullptr; ++j) {
              QString availableMode = QString::fromUtf8(string_list[j]);
              qDebug() << "检查灰度模式选项: " << availableMode;
              if (availableMode.contains("Gray", Qt::CaseInsensitive) ||
                  availableMode.contains("Grey", Qt::CaseInsensitive) ||
                  availableMode.contains("Grayscale", Qt::CaseInsensitive) ||
                  availableMode.contains("Monochrome", Qt::CaseInsensitive) ||
                  availableMode.contains("8bit", Qt::CaseInsensitive)) {
                mode = const_cast<SANE_String>(string_list[j]);
                qDebug() << "找到灰度模式: " << availableMode;
                break;
              }
            }
          } else if (colorMode == "黑白" || colorMode == "Lineart") {
            // 查找黑白模式
            for (int j = 0; string_list[j] != nullptr; ++j) {
              QString availableMode = QString::fromUtf8(string_list[j]);
              if (availableMode.contains("Black", Qt::CaseInsensitive) ||
                  availableMode.contains("Lineart", Qt::CaseInsensitive) ||
                  availableMode.contains("Mono", Qt::CaseInsensitive)) {
                mode = const_cast<SANE_String>(string_list[j]);
                qDebug() << "找到黑白模式: " << availableMode;
                break;
              }
            }
          }
          
          // 如果没有找到匹配的模式，使用第一个可用模式
          if (!mode && string_list[0]) {
            mode = const_cast<SANE_String>(string_list[0]);
            qDebug() << "使用默认颜色模式: " << string_list[0];
          }
        }
      } else {
        // 如果没有约束列表，使用标准名称
        if (colorMode == "彩色" || colorMode == "Color") {
          mode = const_cast<SANE_String>("Color");
        } else if (colorMode == "灰度" || colorMode == "Gray") {
          mode = const_cast<SANE_String>("Gray");
        } else if (colorMode == "黑白" || colorMode == "Lineart") {
          mode = const_cast<SANE_String>("Lineart");
        } else {
          mode = const_cast<SANE_String>("Color"); // 默认彩色
        }
      }

      if (mode) {
        SANE_Status res = sane_control_option(
            m_deviceHandle, i, SANE_ACTION_SET_VALUE, mode, nullptr);

        if (res == SANE_STATUS_GOOD) {
          qDebug() << "颜色模式设置为: " << mode;
          
          // 验证设置是否生效（特别是对于佳能扫描仪）
          char currentMode[256];
          SANE_Status verify = sane_control_option(
              m_deviceHandle, i, SANE_ACTION_GET_VALUE, currentMode, nullptr);
          
          if (verify == SANE_STATUS_GOOD) {
            qDebug() << "验证当前颜色模式: " << currentMode;
            if (QString::fromUtf8(currentMode) != QString::fromUtf8(mode)) {
              qDebug() << "⚠️ 警告: 颜色模式设置未生效! 期望:" << mode << " 实际:" << currentMode;
            }
          }
        } else {
          qDebug() << "设置颜色模式失败: " << sane_strstatus(res);
        }
      }
    }

    // 设置纸张来源 - 扩展选项名称以支持更多扫描仪
    if ((optName == "source" || optName == "adf" || optName == "feeder" ||
         optName == "doc-source" || optName == "document-source" ||
         optName == "paper-source") &&
        opt->type == SANE_TYPE_STRING) {

      // 智能映射纸张来源 - 修复Brother扫描仪兼容性
      SANE_String source = nullptr;
      
      if (actualPaperSource == "自动进纸器" ||
          actualPaperSource == "双面自动进纸器" ||
          actualPaperSource.contains("ADF", Qt::CaseInsensitive) ||
          actualPaperSource.contains("Feeder", Qt::CaseInsensitive)) {

        // 检查设备支持的ADF选项
        if (opt->constraint_type == SANE_CONSTRAINT_STRING_LIST) {
          const SANE_String_Const *string_list = opt->constraint.string_list;
          bool foundADF = false;

          if (string_list) {
            // 如果用户选择了双面自动进纸器，优先查找双面ADF选项
            if (actualPaperSource == "双面自动进纸器") {
              for (int j = 0; string_list[j] != nullptr; ++j) {
                QString availableSource = QString::fromUtf8(string_list[j]);
                if (availableSource.contains("Duplex", Qt::CaseInsensitive) &&
                    (availableSource.contains("ADF", Qt::CaseInsensitive) ||
                     availableSource.contains("Feeder", Qt::CaseInsensitive) ||
                     availableSource.contains("Automatic", Qt::CaseInsensitive))) {
                  source = const_cast<SANE_String>(string_list[j]);
                  foundADF = true;
                  qDebug() << "找到双面ADF选项: " << availableSource;
                  break;
                }
              }
            }
            
            // 如果没有找到双面ADF选项，或者用户选择的是普通自动进纸器，查找普通ADF选项
            if (!foundADF) {
              for (int j = 0; string_list[j] != nullptr; ++j) {
                QString availableSource = QString::fromUtf8(string_list[j]);
                qDebug() << "检查纸张来源选项: " << availableSource;
                if (availableSource.contains("ADF", Qt::CaseInsensitive) ||
                    availableSource.contains("Feeder", Qt::CaseInsensitive) ||
                    availableSource.contains("Automatic", Qt::CaseInsensitive) ||
                    availableSource.contains("Document", Qt::CaseInsensitive) ||
                    availableSource.contains("Auto", Qt::CaseInsensitive)) {
                  source = const_cast<SANE_String>(string_list[j]);
                  foundADF = true;
                  qDebug() << "找到ADF选项: " << availableSource;
                  break;
                }
              }
            }
          }

          if (!foundADF) {
            qDebug() << "设备不支持ADF，强制使用平板模式";
            // 查找FlatBed选项
            for (int j = 0; string_list[j] != nullptr; ++j) {
              QString availableSource = QString::fromUtf8(string_list[j]);
              if (availableSource.contains("FlatBed", Qt::CaseInsensitive) ||
                  availableSource.contains("Flatbed", Qt::CaseInsensitive)) {
                source = const_cast<SANE_String>(string_list[j]);
                qDebug() << "找到平板选项: " << availableSource;
                break;
              }
            }
            if (!source && string_list[0]) {
              source = const_cast<SANE_String>(string_list[0]);
              qDebug() << "使用第一个可用选项: " << string_list[0];
            }
          }
        } else {
          source = const_cast<SANE_String>("ADF");
        }
      } else {
        // 默认使用平板模式，但要检查设备实际支持的选项
        if (opt->constraint_type == SANE_CONSTRAINT_STRING_LIST) {
          const SANE_String_Const *string_list = opt->constraint.string_list;
          if (string_list) {
            // 查找FlatBed选项
            for (int j = 0; string_list[j] != nullptr; ++j) {
              QString availableSource = QString::fromUtf8(string_list[j]);
              if (availableSource.contains("FlatBed", Qt::CaseInsensitive) ||
                  availableSource.contains("Flatbed", Qt::CaseInsensitive)) {
                source = const_cast<SANE_String>(string_list[j]);
                qDebug() << "找到平板选项: " << availableSource;
                break;
              }
            }
            // 如果没有找到FlatBed，使用第一个选项
            if (!source && string_list[0]) {
              source = const_cast<SANE_String>(string_list[0]);
              qDebug() << "未找到FlatBed，使用第一个选项: " << string_list[0];
            }
          }
        } else {
          source = const_cast<SANE_String>("FlatBed");
        }
      }

      SANE_Status res = sane_control_option(
          m_deviceHandle, i, SANE_ACTION_SET_VALUE, source, nullptr);

      if (res == SANE_STATUS_GOOD) {
        qDebug() << "纸张来源设置为: " << source;
      } else {
        qDebug() << "设置纸张来源失败: " << sane_strstatus(res);
      }
    }

    // 设置双面扫描模式
    if ((optName == "duplex" || optName == "adf-duplex" || optName == "duplex-mode" || 
         optName == "double-sided" || optName == "both-sides") &&
        !duplexMode.isEmpty()) {

      if (opt->type == SANE_TYPE_STRING && 
          opt->constraint_type == SANE_CONSTRAINT_STRING_LIST) {
        
        // 字符串列表类型的双面扫描选项
        const SANE_String_Const *string_list = opt->constraint.string_list;
        SANE_String selectedMode = nullptr;
        
        if (string_list) {
          // 查找匹配的双面扫描模式
          for (int j = 0; string_list[j] != nullptr; ++j) {
            QString availableMode = QString::fromUtf8(string_list[j]);
            qDebug() << "可用双面扫描模式: " << availableMode;
            
            // 智能匹配双面扫描模式
            if ((duplexMode.contains("Duplex", Qt::CaseInsensitive) || 
                 duplexMode.contains("True", Qt::CaseInsensitive) ||
                 duplexMode.contains("Both", Qt::CaseInsensitive)) &&
                (availableMode.contains("Duplex", Qt::CaseInsensitive) ||
                 availableMode.contains("Both", Qt::CaseInsensitive) ||
                 availableMode.contains("True", Qt::CaseInsensitive))) {
              selectedMode = const_cast<SANE_String>(string_list[j]);
              break;
            } else if ((duplexMode.contains("Simplex", Qt::CaseInsensitive) ||
                       duplexMode.contains("False", Qt::CaseInsensitive) ||
                       duplexMode.contains("Single", Qt::CaseInsensitive)) &&
                      (availableMode.contains("Simplex", Qt::CaseInsensitive) ||
                       availableMode.contains("False", Qt::CaseInsensitive) ||
                       availableMode.contains("Single", Qt::CaseInsensitive))) {
              selectedMode = const_cast<SANE_String>(string_list[j]);
              break;
            }
          }
          
          // 如果没有找到匹配的，使用第一个选项作为默认值
          if (!selectedMode && string_list[0]) {
            selectedMode = const_cast<SANE_String>(string_list[0]);
          }
        }

        if (selectedMode) {
          SANE_Status res = sane_control_option(
              m_deviceHandle, i, SANE_ACTION_SET_VALUE, selectedMode, nullptr);

          if (res == SANE_STATUS_GOOD) {
            qDebug() << "双面扫描模式设置为: " << selectedMode;
          } else {
            qDebug() << "设置双面扫描模式失败: " << sane_strstatus(res);
          }
        }
        
      } else if (opt->type == SANE_TYPE_BOOL) {
        
        // 布尔类型的双面扫描选项
        SANE_Bool enableDuplex = SANE_FALSE;
        
        if (duplexMode.contains("True", Qt::CaseInsensitive) ||
            duplexMode.contains("Duplex", Qt::CaseInsensitive) ||
            duplexMode.contains("Both", Qt::CaseInsensitive)) {
          enableDuplex = SANE_TRUE;
        }

        SANE_Status res = sane_control_option(
            m_deviceHandle, i, SANE_ACTION_SET_VALUE, &enableDuplex, nullptr);

        if (res == SANE_STATUS_GOOD) {
          qDebug() << "双面扫描设置为: " << (enableDuplex ? "启用" : "禁用");
        } else {
          qDebug() << "设置双面扫描失败: " << sane_strstatus(res);
        }
      }
    }

    // 如果是预览模式，设置较低的分辨率和较大的扫描区域
    if (isPreview) {
      // 设置预览区域（通常是整个扫描区域）
      if (optName == "tl-x" && opt->type == SANE_TYPE_FIXED) {
        SANE_Fixed value = SANE_FIX(0.0);
        sane_control_option(m_deviceHandle, i, SANE_ACTION_SET_VALUE, &value,
                            nullptr);
      }
      if (optName == "tl-y" && opt->type == SANE_TYPE_FIXED) {
        SANE_Fixed value = SANE_FIX(0.0);
        sane_control_option(m_deviceHandle, i, SANE_ACTION_SET_VALUE, &value,
                            nullptr);
      }
      if (optName == "br-x" && opt->type == SANE_TYPE_FIXED) {
        SANE_Fixed value = SANE_FIX(width); // 使用设置的页面宽度
        sane_control_option(m_deviceHandle, i, SANE_ACTION_SET_VALUE, &value,
                            nullptr);
        qDebug() << "设置扫描宽度为:" << width << "mm";
      }
      if (optName == "br-y" && opt->type == SANE_TYPE_FIXED) {
        SANE_Fixed value = SANE_FIX(height); // 使用设置的页面高度
        sane_control_option(m_deviceHandle, i, SANE_ACTION_SET_VALUE, &value,
                            nullptr);
        qDebug() << "设置扫描高度为:" << height << "mm";
      }
    }
  }

  return true;
}

QImage ScanWorker::performScan() {
  if (!m_deviceHandle) {
    qDebug() << "设备未打开，无法扫描";
    return QImage();
  }

  // 开始扫描
  qDebug() << "调用 sane_start...";
  SANE_Status status = sane_start(m_deviceHandle);

  if (status != SANE_STATUS_GOOD) {
    qDebug() << "开始扫描失败: " << sane_strstatus(status);
    return QImage();
  }
  
  qDebug() << "sane_start 成功";

  // 获取扫描参数
  SANE_Parameters params;
  qDebug() << "调用 sane_get_parameters...";
  status = sane_get_parameters(m_deviceHandle, &params);

  if (status != SANE_STATUS_GOOD) {
    qDebug() << "获取扫描参数失败: " << sane_strstatus(status);
    return QImage();
  }

  qDebug() << "扫描参数:";
  qDebug() << "  格式: " << params.format;
  qDebug() << "  最后一帧: " << params.last_frame;
  qDebug() << "  字节数/行: " << params.bytes_per_line;
  qDebug() << "  像素数/行: " << params.pixels_per_line;
  qDebug() << "  行数: " << params.lines;
  qDebug() << "  深度: " << params.depth;

  // 创建图像
  QImage::Format imageFormat;

  switch (params.format) {
  case SANE_FRAME_RGB:
    imageFormat = QImage::Format_RGB888;
    break;
  case SANE_FRAME_GRAY:
    if (params.depth == 1) {
      imageFormat = QImage::Format_Mono;
    } else {
      imageFormat = QImage::Format_Grayscale8;
    }
    break;
  default:
    qDebug() << "不支持的图像格式: " << params.format;
    return QImage();
  }

  // 创建图像缓冲区
  QImage image(params.pixels_per_line, params.lines > 0 ? params.lines : 1,
               imageFormat);

  if (image.isNull()) {
    qDebug() << "创建图像缓冲区失败";
    return QImage();
  }

  // 读取扫描数据
  SANE_Int totalBytes = 0;
  SANE_Int expectedBytes = params.bytes_per_line * params.lines;

  if (expectedBytes <= 0) {
    // 如果扫描仪不知道总行数，我们使用一个估计值
    expectedBytes = params.bytes_per_line * 3000; // 假设最多3000行
  }

  // 使用更大的缓冲区以提高性能
  const int BUFFER_SIZE = 32768; // 32KB 缓冲区
  QByteArray buffer;
  buffer.resize(BUFFER_SIZE);
  
  QByteArray lineBuffer; // 用于累积一行的数据
  lineBuffer.reserve(params.bytes_per_line);

  int currentLine = 0;
  
  qDebug() << "开始读取扫描数据，预期字节数:" << expectedBytes;
  qDebug() << "缓冲区大小:" << BUFFER_SIZE << "每行字节数:" << params.bytes_per_line;

  while (true) {
    // 检查是否取消
    if (m_cancelled) {
      qDebug() << "扫描被取消";
      return QImage();
    }

    // 读取数据块
    SANE_Int readBytes = 0;
    status =
        sane_read(m_deviceHandle, reinterpret_cast<SANE_Byte *>(buffer.data()),
                  buffer.size(), &readBytes);

    if (currentLine < 5 || (currentLine % 100 == 0)) {
      qDebug() << "行" << currentLine << ": sane_read 返回 status=" << status << " readBytes=" << readBytes;
    }

    if (status != SANE_STATUS_GOOD && status != SANE_STATUS_EOF) {
      qDebug() << "读取扫描数据失败: " << sane_strstatus(status);
      return QImage();
    }

    if (status == SANE_STATUS_EOF) {
      qDebug() << "收到 EOF，当前已读取" << totalBytes << "字节，" << currentLine << "行";
      // 处理剩余的行缓冲区数据
      if (!lineBuffer.isEmpty()) {
        qDebug() << "处理剩余的" << lineBuffer.size() << "字节";
      }
      break; // 扫描完成
    }
    
    if (readBytes == 0) {
      if (currentLine < 5) {
        qDebug() << "读取0字节，继续...";
      }
      continue; // 继续读取，不要立即退出
    }

    // 处理读取的数据
    totalBytes += readBytes;
    
    // 将读取的数据添加到行缓冲区
    lineBuffer.append(buffer.constData(), readBytes);
    
    // 处理完整的行
    while (lineBuffer.size() >= params.bytes_per_line) {
      // 提取一行数据
      const char* lineData = lineBuffer.constData();
      
      // 将数据复制到图像中
      if (params.format == SANE_FRAME_RGB) {
        // RGB格式
        for (int x = 0; x < params.pixels_per_line; ++x) {
          if (x * 3 + 2 < params.bytes_per_line) {
            QRgb pixel = qRgb(static_cast<uchar>(lineData[x * 3]),
                              static_cast<uchar>(lineData[x * 3 + 1]),
                              static_cast<uchar>(lineData[x * 3 + 2]));
            image.setPixel(x, currentLine, pixel);
          }
        }
      } else if (params.format == SANE_FRAME_GRAY) {
        // 灰度格式
        if (params.depth == 8) {
          for (int x = 0; x < params.pixels_per_line && x < params.bytes_per_line; ++x) {
            uchar gray = static_cast<uchar>(lineData[x]);
            image.setPixel(x, currentLine, qRgb(gray, gray, gray));
          }
        } else if (params.depth == 1) {
          // 黑白格式
          for (int x = 0; x < params.pixels_per_line; ++x) {
            int byteIdx = x / 8;
            int bitIdx = 7 - (x % 8);

            if (byteIdx < params.bytes_per_line) {
              bool isBlack = (lineData[byteIdx] & (1 << bitIdx)) == 0;
              image.setPixel(x, currentLine, isBlack ? 0 : 1);
            }
          }
        }
      }

      currentLine++;
      
      // 从缓冲区移除已处理的行
      lineBuffer.remove(0, params.bytes_per_line);

      // 如果图像高度不够，则调整大小
      if (currentLine >= image.height()) {
        QImage newImage(image.width(), image.height() + 100, image.format());
        QPainter painter(&newImage);
        painter.drawImage(0, 0, image);
        painter.end();
        image = newImage;
      }
    }

    // 更新进度
    int progress =
        (expectedBytes > 0) ? (totalBytes * 100 / expectedBytes) : 50;
    progress = qBound(0, progress, 99); // 最多显示99%，留1%给处理阶段
    emit scanProgress(progress);

    // 处理事件，让UI保持响应
    QCoreApplication::processEvents();
  }

  // 裁剪图像到实际大小
  if (currentLine > 0 && currentLine < image.height()) {
    image = image.copy(0, 0, image.width(), currentLine);
  }

  // 扫描完成
  emit scanProgress(100);

  qDebug() << "扫描完成，总字节数: " << totalBytes;

  return image;
}

void ScanWorker::performMultiPageScan() {
  if (!m_deviceHandle) {
    emit scanError(tr("设备未打开，无法执行多页扫描"));
    return;
  }

  int pageCount = 0;
  int consecutiveErrors = 0;  // 添加连续错误计数器
  const int MAX_CONSECUTIVE_ERRORS = 3;  // 最多允许3次连续错误
  const int MAX_PAGES = 999;  // 最大页数限制，防止无限循环
  
  qDebug() << "开始ADF多页扫描...";

  // 发送多页扫描开始信号
  emit multiPageScanStarted();

  while (!m_cancelled && pageCount < MAX_PAGES) {
    pageCount++;

    // 发出当前页面扫描信号
    emit currentPageScanning(pageCount);

    // 重置进度为0%，开始新页面扫描
    emit scanProgress(0);

    // 开始扫描当前页
    SANE_Status status = sane_start(m_deviceHandle);

    if (status == SANE_STATUS_NO_DOCS) {
      // 没有更多文档，扫描完成
      pageCount--; // 减去这次没有成功的页面
      qDebug() << "ADF中没有更多文档，扫描完成。总共扫描了" << pageCount
               << "页";
      break;
    }

    if (status != SANE_STATUS_GOOD) {
      consecutiveErrors++;  // 增加错误计数
      qDebug() << "扫描第" << pageCount << "页失败:" << sane_strstatus(status) 
               << "连续错误次数:" << consecutiveErrors;
      
      if (pageCount == 1) {
        // 第一页就失败，报告错误
        emit scanError(tr("开始扫描失败: %1").arg(sane_strstatus(status)));
        return;
      } else {
        // 后续页面失败
        pageCount--; // 减去这次没有成功的页面
        
        // 检查是否连续错误过多
        if (consecutiveErrors >= MAX_CONSECUTIVE_ERRORS) {
          qDebug() << "连续错误次数过多，停止扫描";
          emit scanError(tr("ADF扫描失败：连续%1次无法读取文档，可能输稿器中没有纸张").arg(consecutiveErrors));
          break;
        }
        
        qDebug() << "扫描第" << (pageCount + 1)
                 << "页失败，可能没有更多文档:" << sane_strstatus(status);
        break;
      }
    } else {
      // 扫描成功，重置错误计数器
      consecutiveErrors = 0;
    }

    // 处理多帧扫描（双面扫描可能产生多帧）
    bool hasMoreFrames = true;
    int frameCount = 0;
    
    while (hasMoreFrames && !m_cancelled) {
      frameCount++;
      
      // 获取扫描参数
      SANE_Parameters params;
      status = sane_get_parameters(m_deviceHandle, &params);

      if (status != SANE_STATUS_GOOD) {
        qDebug() << "获取第" << pageCount << "页第" << frameCount 
                 << "帧扫描参数失败:" << sane_strstatus(status);
        break;
      }

      qDebug() << "扫描第" << pageCount << "页第" << frameCount << "帧";
      qDebug() << "  格式: " << params.format;
      qDebug() << "  最后一帧: " << params.last_frame;
      qDebug() << "  像素数/行: " << params.pixels_per_line;
      qDebug() << "  行数: " << params.lines;

      // 读取当前帧数据
      QImage frameImage = readScanData(params);

      if (!frameImage.isNull() && !m_cancelled) {
        qDebug() << "成功扫描第" << pageCount << "页第" << frameCount << "帧";

        // 发送当前帧图像
        emit scanComplete(frameImage);
        
        // 更新页面计数（每个帧都算作一页）
        if (frameCount > 1) {
          pageCount++;
          emit currentPageScanning(pageCount);
        }
      } else {
        qDebug() << "第" << pageCount << "页第" << frameCount << "帧扫描失败或被取消";
        break;
      }

      // 检查是否是最后一帧
      if (params.last_frame == SANE_TRUE) {
        hasMoreFrames = false;
        qDebug() << "第" << pageCount << "页扫描完成，共" << frameCount << "帧";
      } else {
        // 继续读取下一帧
        qDebug() << "准备读取第" << pageCount << "页的下一帧...";
      }
    }

    // 发送当前页面完成进度
    emit scanProgress(100);

    // 检查是否还有更多页面
    // 对于某些扫描仪，需要调用sane_start来检查下一页
    // 这里我们继续循环，让sane_start来判断是否还有文档
  }

  // 检查退出原因
  if (m_cancelled) {
    qDebug() << "ADF扫描被用户取消，已扫描" << pageCount << "页";
  } else if (pageCount >= MAX_PAGES) {
    qDebug() << "达到最大页数限制:" << MAX_PAGES;
    emit scanError(tr("已达到最大页数限制 (%1 页)，扫描停止").arg(MAX_PAGES));
  }

  if (pageCount > 0) {
    qDebug() << "ADF多页扫描完成，总共扫描了" << pageCount << "页";
    // 发送多页扫描完成信号
    emit multiPageScanFinished(pageCount);
  } else {
    emit scanError(tr("ADF扫描失败，没有扫描到任何页面"));
  }
}

QImage ScanWorker::readScanData(const SANE_Parameters &params) {
  // 创建图像格式
  QImage::Format imageFormat;

  switch (params.format) {
  case SANE_FRAME_RGB:
    imageFormat = QImage::Format_RGB888;
    break;
  case SANE_FRAME_GRAY:
    if (params.depth == 1) {
      imageFormat = QImage::Format_Mono;
    } else {
      imageFormat = QImage::Format_Grayscale8;
    }
    break;
  default:
    qDebug() << "不支持的图像格式: " << params.format;
    return QImage();
  }

  // 创建图像缓冲区
  QImage image(params.pixels_per_line, params.lines > 0 ? params.lines : 1,
               imageFormat);

  if (image.isNull()) {
    qDebug() << "创建图像缓冲区失败";
    return QImage();
  }

  // 读取扫描数据
  SANE_Int totalBytes = 0;
  SANE_Int expectedBytes = params.bytes_per_line * params.lines;

  if (expectedBytes <= 0) {
    expectedBytes = params.bytes_per_line * 3000; // 估计值
  }

  // 使用更大的缓冲区以提高性能
  const int BUFFER_SIZE = 32768; // 32KB 缓冲区
  QByteArray buffer;
  buffer.resize(BUFFER_SIZE);
  
  QByteArray lineBuffer; // 用于累积一行的数据
  lineBuffer.reserve(params.bytes_per_line);

  int currentLine = 0;
  
  qDebug() << "readScanData: 开始读取，预期字节数:" << expectedBytes;

  while (!m_cancelled) {
    SANE_Int readBytes = 0;
    SANE_Status status =
        sane_read(m_deviceHandle, reinterpret_cast<SANE_Byte *>(buffer.data()),
                  buffer.size(), &readBytes);

    if (currentLine < 5 || (currentLine % 100 == 0)) {
      qDebug() << "readScanData 行" << currentLine << ": status=" << status << " readBytes=" << readBytes;
    }

    if (status != SANE_STATUS_GOOD && status != SANE_STATUS_EOF) {
      qDebug() << "读取扫描数据失败: " << sane_strstatus(status);
      return QImage();
    }

    if (status == SANE_STATUS_EOF) {
      qDebug() << "readScanData: 收到 EOF，当前已读取" << totalBytes << "字节，" << currentLine << "行";
      // 处理剩余的行缓冲区数据
      if (!lineBuffer.isEmpty()) {
        qDebug() << "readScanData: 处理剩余的" << lineBuffer.size() << "字节";
      }
      break; // 当前页面扫描完成
    }
    
    if (readBytes == 0) {
      if (currentLine < 5) {
        qDebug() << "readScanData: 读取0字节，继续...";
      }
      continue; // 继续读取，不要立即退出
    }

    totalBytes += readBytes;
    
    // 将读取的数据添加到行缓冲区
    lineBuffer.append(buffer.constData(), readBytes);
    
    // 处理完整的行
    while (lineBuffer.size() >= params.bytes_per_line) {
      // 添加安全检查
      if (currentLine >= image.height()) {
        qDebug() << "警告: 当前行超出图像高度，调整图像大小";
        QImage newImage(image.width(), image.height() + 100, image.format());
        QPainter painter(&newImage);
        painter.drawImage(0, 0, image);
        painter.end();
        image = newImage;
      }
      
      // 提取一行数据
      const char* lineData = lineBuffer.constData();
      
      // 将数据复制到图像中
      if (params.format == SANE_FRAME_RGB) {
        // RGB格式
        for (int x = 0; x < params.pixels_per_line; ++x) {
          if (x * 3 + 2 < params.bytes_per_line && x < image.width()) {
            QRgb pixel = qRgb(static_cast<uchar>(lineData[x * 3]),
                              static_cast<uchar>(lineData[x * 3 + 1]),
                              static_cast<uchar>(lineData[x * 3 + 2]));
            image.setPixel(x, currentLine, pixel);
          }
        }
      } else if (params.format == SANE_FRAME_GRAY) {
        // 灰度格式
        if (params.depth == 8) {
          for (int x = 0; x < params.pixels_per_line && x < params.bytes_per_line && x < image.width(); ++x) {
            uchar gray = static_cast<uchar>(lineData[x]);
            image.setPixel(x, currentLine, qRgb(gray, gray, gray));
          }
        } else if (params.depth == 1) {
          // 黑白格式
          for (int x = 0; x < params.pixels_per_line && x < image.width(); ++x) {
            int byteIdx = x / 8;
            int bitIdx = 7 - (x % 8);
            if (byteIdx < params.bytes_per_line) {
              bool isBlack = (lineData[byteIdx] & (1 << bitIdx)) == 0;
              image.setPixel(x, currentLine, isBlack ? qRgb(0,0,0) : qRgb(255,255,255));
            }
          }
        }
      }

      currentLine++;
      
      // 从缓冲区移除已处理的行
      lineBuffer.remove(0, params.bytes_per_line);
    }

    // 更新当前页面的扫描进度
    int progress =
        (expectedBytes > 0) ? (totalBytes * 100 / expectedBytes) : 50;
    progress = qBound(0, progress, 99); // 最多显示99%，留1%给处理阶段
    emit scanProgress(progress);

    // 处理事件，让UI保持响应
    QCoreApplication::processEvents();
  }

  // 裁剪图像到实际大小
  if (currentLine > 0 && currentLine < image.height()) {
    image = image.copy(0, 0, image.width(), currentLine);
  }

  qDebug() << "页面扫描完成，总字节数: " << totalBytes;
  return image;
}

void ScanWorker::closeDevice() {
  if (m_deviceHandle) {
    sane_close(m_deviceHandle);
    m_deviceHandle = nullptr;
    qDebug() << "设备已关闭";
  }
}

void ScannerController::initializeAsync() {
  if (m_initState == Initialized) {
    // 已经初始化，直接返回
    emit initializationCompleted();
    return;
  }

  if (m_initState == Initializing) {
    // 正在初始化中，不重复执行
    return;
  }

  m_initState = Initializing;
  emit initializationStarted();

  // 在后台线程中执行初始化
  m_initFuture = QtConcurrent::run([this]() {
    emit initializationProgress(tr("正在初始化SANE库..."));

    // 初始化SANE
    SANE_Int version_code;
    SANE_Status status = sane_init(&version_code, nullptr);

    if (status != SANE_STATUS_GOOD) {
      qDebug() << "SANE初始化失败: " << sane_strstatus(status);
      m_initState = InitializationFailed;
      emit initializationFailed(tr("SANE初始化失败"));
      return;
    }

    qDebug() << "SANE异步初始化成功，版本: " << SANE_VERSION_MAJOR(version_code)
             << "." << SANE_VERSION_MINOR(version_code) << "."
             << SANE_VERSION_BUILD(version_code);

    m_saneInitialized = true;

    emit initializationProgress(tr("正在查找扫描仪设备..."));

    // 获取设备列表
    const SANE_Device **device_list;
    status = sane_get_devices(&device_list, SANE_TRUE); // 跳过网络设备

    QStringList scanners;
    if (status == SANE_STATUS_GOOD && device_list && device_list[0]) {
      for (int i = 0; device_list[i] != nullptr; ++i) {
        QString name = QString::fromUtf8(device_list[i]->name);
        QString model = QString::fromUtf8(device_list[i]->model);
        QString vendor = QString::fromUtf8(device_list[i]->vendor);
        QString type = QString::fromUtf8(device_list[i]->type);

        // 跳过网络设备
        if (name.contains("net:") || name.contains("http:") ||
            name.contains("https:")) {
          qDebug() << "跳过网络设备: " << name;
          continue;
        }

        // 格式化显示名称
        QString displayName;
        if (!vendor.isEmpty() && !model.isEmpty()) {
          if (model.startsWith(vendor)) {
            displayName = model;
          } else {
            displayName = QString("%1 %2").arg(vendor, model);
          }
        } else {
          displayName = !model.isEmpty() ? model : name;
        }

        // 进一步清理显示名称，去掉括号内容
        QString cleanDisplayName = displayName;
        // 找到第一个左括号的位置，截取之前的部分
        int firstParenIndex = cleanDisplayName.indexOf('(');
        if (firstParenIndex > 0) {
          cleanDisplayName = cleanDisplayName.left(firstParenIndex);
        }
        // 移除多余的空格
        cleanDisplayName = cleanDisplayName.trimmed();

        // 如果清理后名称为空，使用原始显示名称
        if (cleanDisplayName.isEmpty()) {
          cleanDisplayName = displayName;
        }

        // 只显示清理后的名称，不显示技术细节
        scanners.append(cleanDisplayName);
        // 建立显示名称到设备名称的映射
        m_displayNameToDeviceName[cleanDisplayName] = name;
        qDebug() << "找到本地扫描仪(异步初始化):";
        qDebug() << "  显示名称: \"" << cleanDisplayName << "\"";
        qDebug() << "  原始名称: \"" << displayName << "\"";
        qDebug() << "  设备名称: \"" << name << "\"";
        qDebug() << "  类型: \"" << type << "\"";
      }
    }

    // 缓存结果
    m_availableScanners = scanners;
    m_cachedScanners = scanners;

    m_initState = Initialized;
    emit initializationProgress(tr("初始化完成"));
    emit initializationCompleted();
  });
}

bool ScannerController::isInitialized() const { return m_saneInitialized; }

ScannerController::InitializationState
ScannerController::getInitializationState() const {
  return m_initState;
}

void ScannerController::updateScannerParameters() {
  if (!m_saneInitialized || m_currentScanner.isEmpty()) {
    qDebug() << "无法更新扫描参数：SANE未初始化或未选择扫描仪";
    return;
  }

  // 提取设备名称
  QString deviceName;
  int startPos = m_currentScanner.lastIndexOf('(');
  int endPos = m_currentScanner.lastIndexOf(')');
  if (startPos >= 0 && endPos > startPos) {
    deviceName = m_currentScanner.mid(startPos + 1, endPos - startPos - 1);
  } else {
    deviceName = m_currentScanner;
  }

  // 打开设备
  SANE_Handle deviceHandle = nullptr;
  SANE_Status status = sane_open(deviceName.toUtf8().constData(), &deviceHandle);

  if (status != SANE_STATUS_GOOD || !deviceHandle) {
    qDebug() << "无法打开设备更新参数: " << sane_strstatus(status);
    return;
  }

  // 获取选项数量
  SANE_Int num_options;
  status = sane_control_option(deviceHandle, 0, SANE_ACTION_GET_VALUE, &num_options, nullptr);

  if (status == SANE_STATUS_GOOD && num_options > 0) {
    qDebug() << "更新扫描参数: 设备有" << num_options << "个选项";
    
    // 遍历选项更新参数
    for (SANE_Int i = 1; i < num_options; ++i) {
      const SANE_Option_Descriptor *opt = sane_get_option_descriptor(deviceHandle, i);

      if (!opt || !opt->name) {
        continue;
      }

      // 检查选项是否活动和可设置
      bool isActive = !((opt->cap) & (1 << 5));  // SANE_CAP_INACTIVE
      bool isSettable = (opt->cap & (1 << 2));   // SANE_CAP_SOFT_SELECT
      
      if (!isActive || !isSettable) {
        continue;  // 跳过不活动或不可设置的选项
      }

      QString optName;
      try {
        optName = QString::fromUtf8(opt->name);
      } catch (...) {
        continue;
      }

      // 更新分辨率
      if (optName == "resolution" && opt->type == SANE_TYPE_INT) {
        SANE_Int dpi = m_resolution;
        SANE_Status set_status = sane_control_option(deviceHandle, i, SANE_ACTION_SET_VALUE, &dpi, nullptr);
        if (set_status == SANE_STATUS_GOOD) {
          qDebug() << "已设置分辨率:" << dpi;
        }
      }

      // 更新颜色模式
      if ((optName == "mode" || optName == "color-mode") && opt->type == SANE_TYPE_STRING) {
        // 使用QByteArray保持字符串生命周期
        QByteArray modeBytes = m_colorMode.toUtf8();
        char modeBuffer[256];
        strncpy(modeBuffer, modeBytes.constData(), sizeof(modeBuffer) - 1);
        modeBuffer[sizeof(modeBuffer) - 1] = '\0';
        
        SANE_Status set_status = sane_control_option(deviceHandle, i, SANE_ACTION_SET_VALUE, modeBuffer, nullptr);
        if (set_status == SANE_STATUS_GOOD) {
          qDebug() << "已设置颜色模式:" << m_colorMode;
        }
      }

      // 更新纸张来源（只在有值且选项支持时设置）
      if ((optName == "source" || optName == "adf" || optName == "feeder") && 
          opt->type == SANE_TYPE_STRING && 
          !m_paperSource.isEmpty()) {
        // 使用QByteArray保持字符串生命周期
        QByteArray sourceBytes = m_paperSource.toUtf8();
        char sourceBuffer[256];
        strncpy(sourceBuffer, sourceBytes.constData(), sizeof(sourceBuffer) - 1);
        sourceBuffer[sizeof(sourceBuffer) - 1] = '\0';
        
        SANE_Status set_status = sane_control_option(deviceHandle, i, SANE_ACTION_SET_VALUE, sourceBuffer, nullptr);
        if (set_status == SANE_STATUS_GOOD) {
          qDebug() << "已设置纸张来源:" << m_paperSource;
        }
      }
    }
  }

  sane_close(deviceHandle);
  qDebug() << "扫描参数已更新";
}

void ScannerController::ensureInitialized() {
  if (m_initState == NotInitialized) {
    qDebug() << "开始同步初始化SANE...";
    if (initSane()) {
      m_saneInitialized = true;
      m_initState = Initialized;
      qDebug() << "SANE同步初始化成功";
    } else {
      m_initState = InitializationFailed;
      qDebug() << "SANE同步初始化失败";
    }
  }
}