#ifndef SCANNERCONTROLLER_H
#define SCANNERCONTROLLER_H

#include <QObject>
#include <QString>
#include <QStringList>
#include <QImage>
#include <QThread>
#include <QMutex>
#include <QWaitCondition>
#include <QFuture>
#include <QMap>
#include <QtConcurrent/QtConcurrentRun>

// SANE API 头文件
extern "C" {
#include <sane/sane.h>
#include <sane/saneopts.h>
}

class ScanWorker;

class ScannerController : public QObject
{
    Q_OBJECT

public:
    enum InitializationState {
        NotInitialized,
        Initializing,
        Initialized,
        InitializationFailed
    };

    explicit ScannerController(QObject *parent = nullptr);
    ~ScannerController();

    // 扫描仪管理
    QStringList getAvailableScanners();
    bool selectScanner(const QString &name);
    bool isScannerAvailable(const QString &deviceName);
    QString getCurrentScanner() const;

    // 扫描参数设置
    void setResolution(int dpi);
    void setColorMode(const QString &mode);
    void setScanArea(double left, double top, double width, double height);
    void setPageSize(const QString &size);
    void setPaperSource(const QString &source);

    // ADF支持检测
    bool supportsADF() const;
    bool hasDocumentInADF() const;
    QStringList getAvailablePaperSources() const;
    void debugScannerOptions() const;
    
    // 双面扫描支持
    bool supportsDuplex() const;
    QStringList getAvailableDuplexModes() const;
    void setDuplexMode(const QString &mode);

    // 扫描操作
    bool startScan(const QString &scannerName = QString(), 
                  const QString &colorMode = QString(), 
                  int resolution = -1,
                  const QString &paperSource = QString(),
                  const QString &duplexMode = QString());
    bool startPreview(const QString &scannerName = QString());
    void cancelScan();

    // 刷新扫描仪列表
    QStringList refreshScanners(bool includeNetwork = false);
    
    // 更新扫描参数
    void updateScannerParameters();
    
    // 初始化管理
    void ensureInitialized();
    void initializeAsync();
    bool isInitialized() const;
    InitializationState getInitializationState() const;

private:
    QStringList m_cachedScanners;

signals:
    // 扫描进度和结果信号
    void scanProgress(int percentage);
    void scanComplete(const QImage &scannedImage);
    void scanError(const QString &errorMessage);
    
    // 异步初始化信号
    void initializationStarted();
    void initializationProgress(const QString &message);
    void initializationCompleted();
    void initializationFailed(const QString &error);
    void previewComplete(const QImage &previewImage);
    
    // ADF相关信号
    void adfStatusChanged(bool hasDocument);
    void adfEmptyWarning();
    
    // 多页扫描信号
    void multiPageScanStarted();
    void multiPageScanFinished(int totalPages);
    void currentPageScanning(int pageNumber);

    // 内部信号
    void startScanWorker(const QString &deviceName, const QString &colorMode, int resolution, bool isPreview, const QString &paperSource = QString(), const QString &duplexMode = QString(), double width = 215.9, double height = 279.4);
    void cancelScanWorker();

private:
    // 初始化SANE
    bool initSane();
    void closeSane();
    // 检查扫描仪设备是否可用

    // 检测ADF支持
    void detectADFSupport();

    // 扫描仪设备列表
    QStringList m_availableScanners;
    QString m_currentScanner;
    // 显示名称到设备名称的映射
    QMap<QString, QString> m_displayNameToDeviceName;

    // 扫描参数
    int m_resolution;
    QString m_colorMode;
    double m_left, m_top, m_width, m_height;
    QString m_pageSize;
    QString m_paperSource;

    // 扫描工作线程
    QThread m_workerThread;
    ScanWorker *m_scanWorker;

    // SANE状态
    bool m_saneInitialized;
    
    // 初始化状态
    InitializationState m_initState;
    QFuture<void> m_initFuture;
    
    // ADF支持状态
    bool m_supportsADF;
    QStringList m_availablePaperSources;
    
    // 双面扫描支持状态
    bool m_supportsDuplex;
    QStringList m_availableDuplexModes;
    QString m_duplexMode;
    
    // 多页扫描状态
    bool m_isMultiPageScanning;
    int m_currentPageCount;
};

// 扫描工作线程类
class ScanWorker : public QObject
{
    Q_OBJECT
    
public:
    explicit ScanWorker(QObject *parent = nullptr);
    ~ScanWorker();

public slots:
    void doScan(const QString &deviceName, const QString &colorMode, int resolution, bool isPreview, const QString &paperSource = QString(), const QString &duplexMode = QString(), double width = 215.9, double height = 279.4);
    void cancelScan();

signals:
    void scanProgress(int percentage);
    void scanComplete(const QImage &scannedImage);
    void scanError(const QString &errorMessage);
    void multiPageScanStarted();
    void multiPageScanFinished(int totalPages);
    void currentPageScanning(int pageNumber);

private:
    // 打开扫描仪设备
    bool openDevice(const QString &deviceName);
    // 设置扫描参数
    bool setOptions(const QString &colorMode, int resolution, bool isPreview, const QString &paperSource = QString(), const QString &duplexMode = QString(), double width = 215.9, double height = 279.4);
    // 执行扫描
    QImage performScan();
    // 执行多页扫描
    void performMultiPageScan();
    // 读取扫描数据
    QImage readScanData(const SANE_Parameters &params);
    // 关闭设备
    void closeDevice();

    // SANE设备句柄
    SANE_Handle m_deviceHandle;
    // 扫描状态
    bool m_scanning;
    // 取消标志
    bool m_cancelled;
    // 纸张来源和双面扫描模式
    QString m_paperSource;
    QString m_duplexMode;
    // 互斥锁和条件变量，用于线程同步
    QMutex m_mutex;
    QWaitCondition m_condition;
};

#endif // SCANNERCONTROLLER_H